/*
** ComponentContext.h
*/

#ifndef TECGRAF_SCS_COMPONENT_CONTEXT_H
#define TECGRAF_SCS_COMPONENT_CONTEXT_H

#include <scs/private/config.h>
#include <scs/IComponent.h>
#include <scs/private/Receptacle.h>
#include <scs/Facet.h>

#ifndef SCS_ORBIX
#  include <CORBA.h>
#  include <stubs/scs.h>
#else
#  include <cstdio> // Orbix uses EOF without including this header
#  include <stubs/orbix/scsS.hh>
#  include <orbix/corba.hh>
#endif

#include <string>
#include <map>
#include <memory>

namespace scs {
  namespace core {
    class IComponentImpl;
    //! Classe que contém o estado do componente
    /**
     * Essa classe pode ser derivada para que outras facetas possam
     * compartilhar dados constantes do componente. Uma instância
     * dessa classe representa um componente.
     */
    class SCS_DECL ComponentContext {
      private:
#ifdef SCS_THREADING_ENABLED
        MICOMT::Mutex facets_mutex, receptacles_mutex;
#endif

      public:
        //! Identificador para proxima conexao requisitada por IReceptacles
        ConnectionId current_connection_id;
      private:
        std::map<std::string, Receptacle> receptacles;
        std::map<std::string, CORBA::Object_var> facets_by_interface;
        std::map<std::string, Facet> facets_by_name;
        ComponentId component_id;
        PortableServer::POA_var poa;
        PortableServer::ObjectId_var component_object_id;
        PortableServer::ObjectId_var receptacles_object_id;
        PortableServer::ObjectId_var meta_interface_object_id;
        IComponent_var component_reference;
        IReceptacles_var receptacles_reference;
        IMetaInterface_var meta_interface_reference;

        friend class component_context_access;

        void addFacet_internal(const char* name, const char* interface_name
                               , std::auto_ptr<PortableServer::ServantBase>& servant);
        void removeFacet_internal(const char* name);
        void deactivateFacet_internal(const char* name);
      public:
#ifdef SCS_THREADING_ENABLED
        //! Classe para RAII do lock do mutex do estado das facetas no componente
        struct facet_lock
        {
          /**
           * \brief Construtor
           * \param component Referência para o componente que deve ter seu mutex trancado
           * 
           * A instanciação dessa classe tentará fazer o lock do mutex
           * para o estado das facetas desse componente. Enquanto o
           * mesmo não conseguir acesso exclusivo, a execução ficará
           * em suspenso.
           */
          facet_lock(ComponentContext& component)
            : mutex(&component.facets_mutex)
          {
            this->mutex->lock();
            locked = true;
          }

          /**
           * \brief Destrutor
           * 
           * A destruição dessa classe fará o unlock do mutex, se essa
           * instancia de facet_lock tiver o ownership do lock do mutex
           * sobre o estado das facetas do componente.
           */
          ~facet_lock()
          {
            if(locked)
              unlock();
          }

          /**
           * \brief Faz o lock do mutex
           * 
           * Faz o lock do mutex se esse lock não tiver ownership
           * sobre o mutex. Se ja tiver ownership, essa funcao
           * possui comportamento indefinido.
           */
          void lock()
          {
            mutex->lock();
            locked = true;
          }

          /**
           * \brief Faz o unlock do mutex
           * 
           * Faz o unlock do mutex. A instância deve ter ownership
           * sobre o mutex, caso contrario o comportamento e indefinido.
           */
          void unlock()
          {
            mutex->unlock();
            locked = false;
          }
        private:
          bool locked;
          MICOMT::Mutex* mutex;
        };

        //! Classe para RAII do lock do mutex do estado dos receptáculos no componente
        struct receptacles_lock
        {
          /**
           * \brief Construtor
           * \param component Referência para o componente que deve ter seu mutex trancado
           * 
           * A instanciação dessa classe tentará fazer o lock do mutex
           * para o estado dos receptaculos desse componente. Enquanto
           * o mesmo não conseguir acesso exclusivo, a execução ficará
           * em suspenso.
           */
          receptacles_lock(ComponentContext& component)
            : mutex(&component.receptacles_mutex)
          {
            this->mutex->lock();
            locked = true;
          }

          /**
           * \brief Destrutor
           * 
           * A destruição dessa classe fará o unlock do mutex, se essa
           * instancia de receptacles_lock tiver o ownership do lock
           * do mutex sobre o estado das facetas do componente.
           */
          ~receptacles_lock()
          {
            if(locked)
              unlock();
          }

          /**
           * \brief Faz o lock do mutex
           * 
           * Faz o lock do mutex se esse lock não tiver ownership
           * sobre o mutex. Se ja tiver ownership, essa funcao
           * possui comportamento indefinido.
           */
          void lock()
          {
            mutex->lock();
            locked = true;
          }

          /**
           * \brief Faz o unlock do mutex
           * 
           * Faz o unlock do mutex. A instância deve ter ownership
           * sobre o mutex, caso contrario o comportamento e indefinido.
           */
          void unlock()
          {
            mutex->unlock();
            locked = false;
          }
        private:
          bool locked;
          MICOMT::Mutex* mutex;
        };
#endif

        //! Construtor
        /**
         * \param orb Instância do ORB que o componente usara
         * \param poa POA usado para registrar as tres facetas basicas
         * \param id Identificador do componente
         */
        ComponentContext(CORBA::ORB_var orb, PortableServer::POA_var poa, ComponentId const& id);
        ComponentContext(CORBA::ORB_var orb, ComponentId const& id);
        //! Destrutor
        ~ComponentContext();

        //! Retorna o POA passado para a construção do componente
        PortableServer::POA_var getPOA() const { return poa; }
        //! Retorna o identificador passado para a construção do componente
        ComponentId getComponentId();
        //! Retorna um container associativo de facetas ordenado por interface
        std::map<std::string, CORBA::Object_var> const& 
          getMapFacets() const { return facets_by_interface; }
        //! Retorna um container associativo de facetas ordenado por nome
        std::map<std::string, Facet> const& 
          getMapFacetsByName() const { return facets_by_name; }
        //! Retorna um container associativo de receptaculos ordenado por nome
        std::map<std::string, Receptacle> const&
          getMapReceptacles() const { return receptacles; }
        /**
         * Retorna uma referência mutável ao container associativo de
         * receptaculos do componente, ordenado por nome
         */
        std::map<std::string, Receptacle>&
          getMapReceptacles() { return receptacles; }
        //! Retorna uma referência CORBA a faceta IComponent
        scs::core::IComponent_var getIComponent();
        //! Retorna uma referência CORBA a faceta IReceptacles
        scs::core::IReceptacles_var getIReceptacles() { return receptacles_reference; }
        //! Retorna uma referência CORBA a faceta IMetaInterface
        scs::core::IMetaInterface_var getIMetaInterface() { return meta_interface_reference; }

        /**
         * \brief Adiciona uma faceta ao componente
         * \param name Nome da faceta
         * \param interface_name Interface da faceta
         * \param servant Ponteiro para o servante da faceta
         *
         * Ownership é transfeirido incondicionalmente. Se a função
         * falhar o servante é destruido
         */
        void addFacet(const char* name, const char* interface_name, PortableServer::ServantBase* servant);

        /**
         * \brief Adiciona uma faceta ao componente
         * \param name Nome da faceta
         * \param interface_name Interface da faceta
         * \param servant Referência para auto_ptr do servante da faceta
         * 
         * Ownership é transferido condicionalmente. Se a funçao falhar
         * antes de transferir o ownership para o ORB, então servant
         * não é modificado. Se não o ownership de servant é transferido
         * para o ORB e destruido pelo POA.
         */
        void addFacet(const char* name, const char* interface_name
                      , std::auto_ptr<PortableServer::ServantBase>& servant);

        /**
         * \brief Substitui o servant de uma faceta ao componente
         * \param name Nome da faceta
         * \param interface_name Interface da faceta
         * \param servant Ponteiro para o servante da faceta
         *
         * Ownership é transfeirido incondicionalmente. Se a função
         * falhar o servante é destruído
         */
        void updateFacet(const char* name, PortableServer::ServantBase* servant);

        /**
         * \brief Substitui o servant de uma faceta ao componente
         * \param name Nome da faceta
         * \param interface_name Interface da faceta
         * \param servant Referência para auto_ptr do servante da faceta
         * 
         * Ownership é transferido condicionalmente. Se a funçao falhar
         * antes de transferir o ownership para o ORB, então servant
         * não é modificado. Se não o ownership de servant é transferido
         * para o ORB e destruido pelo POA.
         */
        void updateFacet(const char* name, std::auto_ptr<PortableServer::ServantBase>& servant);

        /**
         * \brief Registra um receptaculo ao componente
         * \param name Nome do receptaculo
         * \param interface_name Nome da interface do receptaculo
         * \param multiplexed Se o receptaculo pode ter apenas apenas uma conexao
         *
         * Registra um receptaculo de nome name, e interface
         * interface_name. Esse receptaculo inicialmente não terá
         * nenhuma conexão. Conexões podem ser criadas através
         * da faceta IReceptacles do componente.
         */
        void addReceptacle(const char* name, const char* interface_name, bool multiplexed = false);

        /**
         * \brief Remove faceta do componente
         * \param name Nome da faceta
         *
         * Remove a faceta de nome name. Se nenhuma faceta com esse
         * nome existe no componente, essa função lança SCSException.
         */
        void removeFacet(const char* name);

        /**
         * \brief Desativa a faceta do componente no POA
         * \param name Nome da faceta
         *
         * Desativa a faceta de nome name no POA. Se nenhuma faceta com esse
         * nome existe no componente, essa função lança SCSException.
         */
        void deactivateFacet(const char* name);

        /**
         * \brief Returns a object with description about a facet
         * \param name Facet's name
         *
         * Returns a object with description about a facet, throws otherwise.
         */
        Facet getFacetByName(const char* name);
      };
  }
}

#endif

