/**
* \file logger.h
*/

#ifndef LOGGER_H_
#define LOGGER_H_

#include <fstream>
#include <string>
#include <sstream>
#include <map>
#ifdef LOGGER_MULTITHREAD
#include <pthread.h>
#endif

#if defined(_WIN32) && defined(LOGGER_SHARED)
#  ifdef LOGGER_SOURCE
#define LOGGER_DECL __declspec(dllexport)
#  else
#define LOGGER_DECL __declspec(dllimport)
#  endif
#endif

#ifndef LOGGER_DECL
#define LOGGER_DECL
#endif

namespace logger {

  /**
  * \brief Nvel de mensagem.
  * O nvel ALL e OFF possuem um comportamento distinto  perante os demais. ALL representa  
  * todos  os demais nveis com exceo do OFF. O  nvel   OFF  representa  a  ausncia de 
  * todos os demais nveis com exceo do ALL.
  */
  enum Level {
    OFF,
    ERROR,
    WARNING,
    INFO,
    ALL
  };

    #ifdef LOGGER_MULTITHREAD
    class Mutex_auto {
    public:
      Mutex_auto(pthread_mutex_t *mutex) : _mutex(mutex) {
        pthread_mutex_lock(_mutex);
        _locked = true;
      }
      
      ~Mutex_auto() {
        if (_locked) pthread_mutex_unlock(_mutex);
      }
    
      void lock() {
        pthread_mutex_lock(_mutex);
        _locked = true;
      }

      void unlock() {
        pthread_mutex_unlock(_mutex);
        _locked = false;
      }
    private:
      pthread_mutex_t *_mutex;
      bool _locked;
    };
    #endif

  /**
  * \brief Representa um objeto do tipo Logger  a   ser utilizado como mecanismo de log.
  */ 
  class LOGGER_DECL Logger {
  public:
    Logger();
    ~Logger();
          
    /**
    * Define a sada do logger.
    *
    * O padro  sada padro stdout.
    *
    * @param[in] filename Caminho relativo ou absoluto do arquivo de sada do logger. 
    * O valor 0 (NULL) representa a sada padro stdout.
    */
    void setOutput(const char *filename);

    /**
    * Ativa ou desativa um determinado nvel de mensagem.
    *
    * Se o nvel est desativado, a chamada a este mtodo ativar o mesmo, e vice-versa.
    *
    * @param [in] Nvel de mensagem.
    */
    void setLevel(Level l) { _level = l; }

    /**
    * Informa se um nvel est ativo ou no.
    *
    * @return True se o nvel est ativo, caso contrrio retorna false.
    */
    Level getLevel(Level l) { return _level; }

    /**
    * Registra uma mensagem utilizando um determinado nvel.
    *
    * @param [in] l Nvel de mensagem.
    * @param [in] message Mensagem.
    */
    void log(Level l, std::string message);

    /**
    * Edenta em um nvel as mensagens seguintes.
    *
    * Abre-se um novo escopo para as mensagens seguintes a esta chamada.
    */
    void indent() { ++_numIndent; };

    /**
    * Edenta em um nvel as mensagens seguintes.
    *
    * Registra uma mensagem e abre um novo escopo para as mensagens seguintes a esta chamada.
    *
    * @param [in] l Nvel de mensagem.
    * @param [in] message Mensagem.
    */
    void indent(Level l, std::string message);

    /**
    * Desfaz uma edentao.
    *
    * Fecha o escopo atual de mensagens.
    */
    void dedent() { --_numIndent; }; 

    /**
    * Desfaz uma edentao.
    *
    * Fecha o escopo atual de mensagens e logo aps registra uma mensagem.
    *
    * @param [in] l Nvel de mensagem.
    * @param [in] message Mensagem.
    */
    void dedent(Level l, std::string message);
      
  private:
    #ifdef LOGGER_MULTITHREAD
    pthread_mutex_t _mutex;
    #endif    

    /**
    * Relao entre os nveis como elementos de enumerao com as representaes dos nveis em 
    * strings.
    */
    static const char *_levelStr[];

    Level _level;

    /**
    * Contador de edentao utilizado para se gerar   uma sada hierrquica conforme a pilha de 
    * chamadas.
    */
    short int _numIndent;

    /**
    * Path relativo ou absoluto do arquivo  de  sada  do Logger.
    */
    const char *_filename;
    std::ostream *_output;
  };
}

#endif
