/**
* \mainpage API - Openbus C++
* \file openbus.h
*/

#ifndef TECGRAF_OPENBUS_H_
#define TECGRAF_OPENBUS_H_

#include "logger.h"
#include "util/Helper.h"
#include "openbus_decl.h"

#include "interceptors/ORBInitializerImpl.h"

#ifdef OPENBUS_ORBIX
  #include <omg/orb.hh>
  #include <omg/PortableServer.hh>
  #include <it_ts/thread.h>
  #include <it_ts/timer.h>
  #include <it_ts/mutex.h>
  #include "stubs/orbix/access_control_service.hh"
  #include "stubs/orbix/registry_service.hh"
  #ifdef OPENBUS_SDK_FT_ENABLED
  #include "stubs/orbix/fault_tolerance.hh"
  #endif
#else
  #include <CORBA.h>
  #include "stubs/mico/access_control_service.h"
  #include "stubs/mico/registry_service.h"
  #ifdef OPENBUS_SDK_FT_ENABLED
  #include "stubs/mico/fault_tolerance.h"
  #endif
#endif

#include <stdexcept>
#include <string>
#include <map>
#include <set>
#ifdef OPENBUS_SDK_FT_ENABLED
#include <lua.hpp>
#endif

/**
 * \brief Stubs dos servios bsicos.
 */
namespace tecgraf {
namespace openbus {
namespace core {
namespace v1_05 {
  /**
   * \brief Stub do servio de acesso.
   */
  namespace access_control_service {

  /**
   * \class Credential
   * \brief Credencial de acesso ao barramento.
   */
}}}}}

/**
* \brief openbus
*/
namespace openbus {

typedef std::set<std::string> MethodSet;
typedef std::map<std::string, MethodSet*> MethodsNotInterceptable;

/**
 * \brief Falha no processo de login, ou seja, o par nome de usurio e senha no foi validado.
 */
struct LOGIN_FAILURE : public std::runtime_error {
  LOGIN_FAILURE(const std::string &msg = "") : runtime_error(msg) {}
};

/**
 * \brief Falha no mecanismo de autenticao por certificado digital.
 * Algumas possveis causas:
 *  + No foi possvel obter o desafio.
 *  + Falha na manipulao de uma chave privada ou pblica.
 *  + Falha na manipulao de um certificado.
 */
struct SECURITY_EXCEPTION : public std::runtime_error {
  SECURITY_EXCEPTION(const std::string &msg = "") : runtime_error(msg) {}
};

/**
 * \brief Representa um barramento.
 */
class OPENBUS_SDK_DECL Openbus {
public:
  class LeaseExpiredCallback;

private:
  static char* debugFile;
  logger::Level debugLevel;

/**
 * Mutex. 
 */
#ifdef OPENBUS_ORBIX
  static IT_Mutex mutex;
#else
  static MICOMT::Mutex mutex;
#endif

  /**
   * A instncia nica do barramento.
   */
  static Openbus *bus;

  /**
   * Inicializador do ORB. 
   */
  static interceptors::ORBInitializerImpl *ini;

  /**
   * ORB. 
   */
  static CORBA::ORB_var orb;
      
  /**
   * Determina se o ORB est escutando requisies CORBA.
   */
  static bool orbRunning;

  /**
   * POA.
   */
  static PortableServer::POA_var poa;

  #ifdef OPENBUS_SDK_FT_ENABLED
  /**
   * Mquina virtual de Lua.
   */
  static lua_State *luaState;

  /**
   * Nome do arquivo de configurao das rplicas.
   */
  static char *FTConfigFilename;
  #endif

  /**
   * Parmetro argc da linha de comando. 
   */
  int _argc;

  /**
   * Parmetro argv da linha de comando. 
   */
  char **_argv;
      
  /**
   * Ponteiro para o stub do servio de acesso.
   */
  tecgraf::openbus::core::v1_05::access_control_service::IAccessControlService_var iAccessControlService;

  /**
   * Ponteiro para o stub do servio de registro.
   */
  tecgraf::openbus::core::v1_05::registry_service::IRegistryService *iRegistryService;

  /**
   * Ponteiro para a faceta ILeaseProvider. 
   */
  tecgraf::openbus::core::v1_05::access_control_service::ILeaseProvider_var iLeaseProvider;

  /**
   * Ponteiro para o IComponent do servio de acesso. 
   */
  scs::core::IComponent_var iComponentAccessControlService;

  #ifdef OPENBUS_SDK_FT_ENABLED
  /**
   * Ponteiro para o stub do servio de tolerncia a falhas.
   */
  tecgraf::openbus::fault_tolerance::v1_05::IFaultTolerantService_var iFaultTolerantService;
  #endif

  /**
   * Gerenciador do POA. 
   */
  PortableServer::POAManager_var poa_manager;

  /**
   * Intervalo de tempo que determina quando a credencial expira. 
   */
  tecgraf::openbus::core::v1_05::access_control_service::Lease lease;

  /**
   * Credencial de identificao do usurio frente ao barramento. 
   */
  tecgraf::openbus::core::v1_05::access_control_service::Credential *credential;

#if (!OPENBUS_ORBIX && OPENBUS_SDK_MULTITHREAD)
  static MICOMT::Thread::ThreadKey threadKey;
#endif
    
  /**
   * Poltica de validao de credenciais.
   */
  interceptors::CredentialValidationPolicy credentialValidationPolicy;

  /**
   * Mquina em que est o barramento. 
   */
  std::string hostBus;

  /**
   * A porta da mquina em que se encontra o barramento.
   */
  unsigned short portBus;

  /**
   * Possveis estados para a conexo. 
   */
  enum ConnectionStates {
    CONNECTED,
    DISCONNECTED
  };

  /**
   * Indica o estado da conexo. 
   */
  ConnectionStates connectionState;

  /**
   * Intervalo de tempo em segundos que determina quando que a credencial ser renovada.
   */
  unsigned short timeRenewing;

  /**
   * Especifica se o tempo de renovao da credencial  fixo.
   */
  bool timeRenewingFixe;

  /*
   * Intervalo de validao das credenciais do cache.
   */
  unsigned long credentialsValidationTime;

  /**
   * Mapa de mtodos relacionados a suas respectivas interfaces que devem ser ignorados pelo 
   * interceptador servidor.
   */
  MethodsNotInterceptable methodsNotInterceptable;

  /**
   * Trata os parmetros de linha de comando.
   */
  void commandLineParse();

  /**
   * Inicializa um valor default para a mquina e porta do barramento. 
   */
  void initialize();

  /**
   * Cria implicitamente um ORB e um POA. 
   */
  void createORB();

  /**
   * Registra os interceptadores cliente e servidor. 
   */
  void registerInterceptors();

  /**
   * Cria um estado novo. 
   */
  void newState();
      
  /**
   * Cria o proxy para o servio de acesso.
   */
  void createProxyToIAccessControlService();
  friend class openbus::interceptors::ClientInterceptor;

  /**
   * Cria o objeto registryService.
   */
  void setRegistryService();

  /**
   * Callback registrada para a notificao da expirao do lease.
   */
  static LeaseExpiredCallback * _leaseExpiredCallback;

  /**
   * Thread/Callback responsvel pela renovao da credencial do usurio que est logado neste 
   * barramento.
   */
#ifdef OPENBUS_ORBIX
  IT_Thread renewLeaseIT_Thread;
  class OPENBUS_SDK_DECL RenewLeaseThread : public IT_ThreadBody {
  public:
    void* run();
    bool runningLeaseExpiredCallback;
    RenewLeaseThread();
    void stop();
  private:
    bool sigINT;
    bool sleep(unsigned short time);
  };
  static RenewLeaseThread *renewLeaseThread;
#else
  #ifdef OPENBUS_SDK_MULTITHREAD
  class OPENBUS_SDK_DECL RenewLogin : public MICOMT::Thread {
  public:
    RenewLogin(const tecgraf::openbus::core::v1_05::access_control_service::ILeaseProvider_var, unsigned short);
    ~RenewLogin();
    void _run(void *);
    void stop();
    void pause();
    void run();
  private:
    MICOMT::Mutex _mutex;
    MICOMT::CondVar _condVar;
    tecgraf::openbus::core::v1_05::access_control_service::ILeaseProvider_var _iLeaseProvider;
    tecgraf::openbus::core::v1_05::access_control_service::Lease _validityTime;
    bool _pause;
    bool _stop;
    tecgraf::openbus::core::v1_05::access_control_service::Lease renew();
  };
  static RenewLogin *renewLogin;
  #else
  class OPENBUS_SDK_DECL RenewLeaseCallback : public CORBA::DispatcherCallback {
  public:
  RenewLeaseCallback();
    void callback(CORBA::Dispatcher *dispatcher, Event event);
  };
  RenewLeaseCallback renewLeaseCallback;
  #endif
#endif

    /**
     * [OPENBUS-410] - Mico
     * 
     * O modo reativo no funciona adequadamente na verso *multithread*. A alternativa encontrada 
     * foi utilizar uma thread dedicada ao orb->run() afim de evitar _deadlock_ distribudo.
     */
#ifndef OPENBUS_ORBIX
#ifdef OPENBUS_SDK_MULTITHREAD
    class OPENBUS_SDK_DECL RunThread : public MICOMT::Thread {
    public:
      void _run(void *);
    };
    friend class Openbus::RunThread;

    static RunThread *runThread;
#endif
#endif
    Openbus();
    #ifdef OPENBUS_SDK_FT_ENABLED
    /**
     * Flag que informa se o mecanismo de tolerncia a falhas est ativado.
     */
    bool faultToleranceEnable;
    friend class FaultToleranceManager;
    #endif
  public:

    static logger::Logger *logger;

    ~Openbus();

    /**
     * Fornece a nica instncia do barramento.
     *
     * @return Openbus
     */
    static Openbus * getInstance();

    /**
     * Inicializa uma referncia a um barramento.
     *
     * Um ORB e um POA so criados implicitamente.
     * A fbrica de componentes SCS  criada.
     * Os argumentos Openbus de linha de comando (argc e argv) so tratados.
     * Parmetros de linha de comando: 
     *   "-OpenbusHost": Mquina em que se encontra o barramento.
     *   "-OpenbusPort": Porta do barramento.
     *   "-OpenbusDebug":
     *     ALL - Ativa todos os nveis de verbose.
     *     ERROR - Ativa o nvel ERROR do debug.
     *     INFO - Ativa o nvel INFO do debug.
     *     WARNING - Ativa o nvel WARNING do debug.
     *   "-OpenbusDebugFile": Caminho completo ou relativo do arquivo que armazenar as  mensagens 
     *     de verbose. Se este parmetro no for definido, a sada do verbose ser a sada padro.
     *     OBS.:  Se for  definido, o  verbose  somente  ser armazenado  no  arquivo  em questo, 
     *     ou seja, a sada padro no ser mais utilizada.
     *   "-OpenbusValidationPolicy": Define uma poltica de validao das credenciais. Por padro, 
     *     a poltica ALWAYS  adotada.
     *     NONE: No h validao de credenciais.
     *     ALWAYS: Sempre valida cada credencial no ACS.
     *     CACHED: Primeiro  tenta  validar  a   credencial    consultando    um    cache   local, 
     *       se no conseguir, a validao  transcorre normalmente atravs de  uma  chamada remota 
     *       ao ACS.
     *   "-OpenbusValidationTime": Define o intervalo de tempo(em milisegundos)   de  validao do 
     *     cache de credenciais. O tempo padro  de 30000ms.
     *   "-OpenbusFTConfigFilename": Caminho completo  ou  relativo  do  arquivo  que  descreve as 
     *     rplicas a serem utilizadas pelo mecanismo de tolerncia a falhas.
     *   "-OpenbusTimeRenewing": Tempo em segundos de renovao da credencial.
     *
     * @param[in] argc
     * @param[in] argv
     */
    void init(int argc, char **argv);

    /**
     * Inicializa uma referncia a um barramento.
     *
     * Um ORB e um POA so criados implicitamente.
     * A fbrica de componentes SCS  criada.
     * Os argumentos Openbus de linha de comando (argc e argv) so tratados.
     * Parmetros de linha de comando: 
     *   "-OpenbusHost": Mquina em que se encontra o barramento.
     *   "-OpenbusPort": Porta do barramento.
     *   "-OpenbusDebug":
     *     ALL - Ativa todos os nveis de verbose.
     *     ERROR - Ativa o nvel ERROR do debug.
     *     INFO - Ativa o nvel INFO do debug.
     *     WARNING - Ativa o nvel WARNING do debug.
     *   "-OpenbusDebugFile": Caminho completo ou relativo do arquivo que armazenar as  mensagens 
     *     de verbose. Se este parmetro no for definido, a sada do verbose ser a sada padro.
     *     OBS.:  Se for  definido, o  verbose  somente  ser armazenado  no  arquivo  em questo, 
     *     ou seja, a sada padro no ser mais utilizada.
     *   "-OpenbusValidationPolicy": Define uma poltica de validao das credenciais. Por padro, 
     *     a poltica ALWAYS  adotada.
     *     NONE: No h validao de credenciais.
     *     ALWAYS: Sempre valida cada credencial no ACS.
     *     CACHED: Primeiro  tenta  validar  a   credencial    consultando    um    cache   local, 
     *       se no conseguir, a validao  transcorre normalmente atravs de  uma  chamada remota 
     *       ao ACS.
     *   "-OpenbusValidationTime": Define o intervalo de tempo(em milisegundos)   de  validao do 
     *     cache de credenciais. O tempo padro  de 30000ms.
     *   "-OpenbusFTConfigFilename": Caminho completo  ou  relativo  do  arquivo  que  descreve as 
     *     rplicas a serem utilizadas pelo mecanismo de tolerncia a falhas.
     *   "-OpenbusTimeRenewing": Tempo em segundos de renovao da credencial.
     *   
     *   Ateno:Os parmetros de linha de comando podem sobrescrer os parmetros de funo: host,
     *           port e policy.
     *
     * @param[in] argc
     * @param[in] argv
     * @param[in] host Mquina em que se encontra o barramento.
     * @param[in] port A porta da mquina em que se encontra o barramento.
     * @param[in] policy Poltica de renovao de credenciais. Este parmetro  opcional. O padro
     *    adotar a poltica ALWAYS.
     */
    void init(int argc,
              char **argv,
              char *host,
              unsigned short port,
              interceptors::CredentialValidationPolicy policy = interceptors::ALWAYS);

    /**
     * Informa o estado de conexo com o barramento.
     *
     * @return true caso a conexo esteja ativa, ou false, caso contrrio.
     */
    bool isConnected();

    /** 
     *  Disponibiliza um *termination handler* que desconecta o usurio do barramento e finaliza a 
     *  execuo do Openbus::run().
     *
     *  Essa callback pode ser utlizada em uma implementao de um *termination handler* a ser es-
     *  crito pelo usurio. No caso do Orbix, o mtodo pode ser registrado diretamente na   classe
     *  IT_TerminationHandler(), e.g.:
     *
     *  IT_TerminationHandler termination_handler(openbus::Openbus::terminationHandlerCallback)
     *
     *  O mtodo desconecta o usurio do barramento,   se   este   estiver  conectado,  executa um
     *  Openbus::stop()  seguido  por  um  Openbus::finish(),  e,  por  ltimo  faz   _delete_  da 
     *  instanciao do Openbus.
     *
     *  @param signalType
     */
    static void terminationHandlerCallback(long signalType);

    /**
     *  Retorna o ORB utilizado.
     *  @return ORB
     */
    CORBA::ORB * getORB();

    /**
     *  Retorna o RootPOA.
     *
     *  OBS: A chamada a este mtodo ativa o POAManager.
     *
     *  @return POA
     */
    PortableServer::POA * getRootPOA();

    /**
     * Retorna a credencial interceptada pelo interceptador servidor. 
     * @return Credencial. \see openbusidl::acs::Credential
     */
    tecgraf::openbus::core::v1_05::access_control_service::Credential_var getInterceptedCredential();

    /**
     * Retorna o servio de acesso. 
     * @return Servio de acesso
     */
    tecgraf::openbus::core::v1_05::access_control_service::IAccessControlService * getAccessControlService();

    /**
     * Retorna o servio de registro. 
     * @return Servio de registro
     */
    tecgraf::openbus::core::v1_05::registry_service::IRegistryService * getRegistryService();

    /**
     * Retorna a credencial de identificao do usurio frente ao barramento. 
     * @return credencial
     */
    tecgraf::openbus::core::v1_05::access_control_service::Credential * getCredential();

    /**
     * Retorna a poltica de validao de credenciais.
     * @return Poltica de validao de credenciais.
     */
    interceptors::CredentialValidationPolicy getCredentialValidationPolicy();

    /**
     * Define uma credencial a ser utilizada no lugar da credencial corrente. 
     * til para fornecer uma credencial com o campo delegate preenchido.
     * 
     * !Ateno!
     *   Na verso Orbix este mtodo no considera um comportamento multithread, 
     *   sendo assim a alterao de credencial ser vlida para todas as threads.
     *
     * @param[in] credential Credencial a ser utilizada nas requisies a serem
     *   realizadas.
     */
    void setThreadCredential(tecgraf::openbus::core::v1_05::access_control_service::Credential *credential);

    /**
     * Representa uma callback para a notificao de que um lease expirou.
     */
    class OPENBUS_SDK_DECL LeaseExpiredCallback {
    public:
      virtual ~LeaseExpiredCallback() {};
      virtual void expired() = 0;
    };

    /**
     * Registra uma callback para a notificao de que o lease da credencial de identificao 
     * do usurio, frente ao barramento, expirou.
     *
     * @param[in] A callback a ser registrada.
     * @return True se a callback foi registrada com sucesso, ou false 
     * se a callback j estava registrada.
     */
    void setLeaseExpiredCallback(LeaseExpiredCallback *leaseExpiredCallback);

    /**
     * Remove uma callback previamente registra para a notificao de lease expirado.
     *
     * @param[in] A callback a ser removida.
     * @return True se a callback foi removida com sucesso, ou false 
     * caso contrrio.
     */
    void removeLeaseExpiredCallback();

    /**
     *  Realiza uma tentativa de conexo com o barramento.
     *
     *  @param[in] user Nome do usurio.
     *  @param[in] password Senha do usurio.
     *  @throw LOGIN_FAILURE O par nome de usurio e senha no foram validados.
     *  @throw CORBA::SystemException Alguma falha de comunicao com o barramento ocorreu.
     *  @return  Se a tentativa de conexo for bem sucedida, uma instncia que representa o 
     *    servio  retornada.
     */
    tecgraf::openbus::core::v1_05::registry_service::IRegistryService * connect(const char *user, const char *password);

    /**
     *  Realiza uma tentativa de conexo com o barramento utilizando o mecanismo de 
     *  certificao para o processo de login.
     *
     *  @param[in] entity Nome da entidade a ser autenticada atravs de um certificado digital.
     *  @param[in] privateKeyFilename Nome do arquivo que armazena a chave privada do servio.
     *  @param[in] ACSCertificateFilename Nome do arquivo que armazena o certificado do servio.
     *  @throw LOGIN_FAILURE O par nome de usurio e senha no foram validados.
     *  @throw CORBA::SystemException Alguma falha de comunicao com o barramento ocorreu.
     *  @throw SECURITY_EXCEPTION Falha no mecanismo de autenticao por certificado digital.
     *    Algumas possveis causas:
     *     + No foi possvel obter o desafio.
     *     + Falha na manipulao de uma chave privada ou pblica.
     *     + Falha na manipulao de um certificado.
     *       entidade ou do certificado do ACS.
     *  @return  Se a tentativa de conexo for bem sucedida, uma instncia que representa o 
     *    servio  retornada.
     */
    tecgraf::openbus::core::v1_05::registry_service::IRegistryService * connect(const char *entity, 
                                                                                const char *privateKeyFilename,
                                                                                const char *ACSCertificateFilename);

    /**
     *  Desfaz a conexo atual.
     *
     *  @return Caso a conexo seja desfeita, true  retornado, caso contrrio, o valor de 
     *  retorno  false.
     */
    bool disconnect();

    /**
     * Loop que processa requisies CORBA.
     *
     * !Ateno! Este mtodo na verso Mico multithread faz um wait() na RunThread.
     */
    void run();
      
    /**
     * Pra de processar requisies CORBA. Finaliza a execuo do run.
     * 
     * !Ateno! No faz nada na verso Mico multithread.
     */  
    void stop();

    /**
     * Finaliza a execuo do ORB.
     *
     * O mtodo chama a dupla orb->shutdown(force) e orb->destroy().
     *
     * @param[in] bool force Se a finalizao deve ser forada ou no.
     */
    static void finish(bool force = 0);

    /**
     * Define se os mtodos de uma determinada interface devem ou no ser interceptados pelo
     * interceptador servidor.
     *
     * @param[in] string interfaceRepID RepID da interface.
     * @param[in] string method Nome do mtodo.
     * @param[in] bool isInterceptable Indica se o mtodo deve ser interceptado.
     *
     */
    void setInterceptable(std::string interfaceRepID, std::string method, bool isInterceptable);

    /**
     * Consulta se o mtodo est sendo interceptado.
     *
     * @param[in] string interfaceRepID RepID da interface.
     * @param[in] string method Nome do mtodo.
     *
     * @return true se o mtodo  interceptado ou false caso contrrio.
     */
    bool isInterceptable(std::string interfaceRepID, std::string method);

    #ifdef OPENBUS_SDK_FT_ENABLED
    /**
     * Consulta se o mecanismo de tolerancia a falhas est ativado.
     *
     * @return true se o mecanismo de tolerancia a falhas est ativado ou false 
     * caso contrario.
     */
    bool isFaultToleranceEnable();
    #endif
  };
}

#endif
