/*
** ftc.h
*/

#ifndef  FTC_H_
#define  FTC_H_

#if defined(_WIN32) || defined(WIN32) || defined(_WINNT_VER)
#define TEC_WINDOWS
#include <winsock2.h>
#endif

#include <stdio.h>
#include "ftc_exception.h"
#include "ftc_decl.h"

/** 
 * @brief Classe que representa a API de acesso a um arquivo remoto. 
 */
struct ftc
{
    /** 
     * @brief Construtor 
     * 
     * @param id Array contendo o identificador do arquivo.
     * @param id_size Tamanho do array que contem o identificador do arquivo.
     * @param writable Indica se o arquivo deve ser aberto para leitura.
     * @param host Endereço do servidor.
     * @param port Porta TCP do servidor.
     * @param accessKey Array contendo a chave de acesso ao arquivo.
     * @param key_size Tamanho do array que contem a chave de acesso.
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    ftc(const char* id, const char id_size, bool writable, const char* host
        , unsigned short port, const char* accessKey, const char key_size);

    /** 
     * @brief Destrutor
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    ~ftc();

    /** 
     * @brief Abre o arquivo remoto. 
     * 
     * @param readonly Indica se o arquivo deve ser aberto apenas para leitura.
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    void open( bool readonly ) ;

    /** 
     * @brief Indica se o arquivo está aberto.
     * 
     * @return Verdadeiro se o arquivo estiver aberto
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    bool isOpen() ;

    /** 
     * @brief Fecha o arquivo. 
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    void close() ;

    /** 
     * @brief Posiciona o cursor de leitura no arquivo.
     * 
     * @param position Numero de bytes a partir do inicio do arquivo.
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    void setPosition( unsigned long long position ) ;

    /** 
     * @brief Define o tamanho do arquivo.
     * Pode ser usado para alocar um espaço ou truncar o arquivo. 
     * 
     * @param size Novo tamanho do arquivo em bytes. 
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    void setSize( unsigned long long size );

    /** 
     * @brief Define o tamanho do buffer de leitura utilizado na leitura do
     * arquivo.
     * 
     * @param size Tamanho do buffer de leitura em bytes.
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    void setReadBufferSize( unsigned long long size );

    /** 
     * @brief Retorna a atual posição do cursor de leitura no arquivo.
     * 
     * @return Posição a partir no inicio do arquivo em bytes. 
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    unsigned long long getPosition() ;

    /** 
     * @brief Retorna o tamanho atual do arquivo. 
     * 
     * @return O numero de bytes que o arquivo possui.
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    unsigned long long getSize() const;

    /** 
     * @brief Retorna o tamanho atual do buffer de leitura. 
     * 
     * @return Tamanho do buffer em bytes.
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    unsigned long long getReadBufferSize() ;

    /** 
     * @brief Le uma quantidade de bytes a partir de uma dada posição.
     * 
     * @param data Ponteiro para o espaço onde os bytes lidos serão escritos.
     * @param nbytes Quantidades de bytes a serem lidos. 
     * @param position Posição inicial da leitura.
     * 
     * @return Retorna o numero de bytes lidos.
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    unsigned long long read(char* data, unsigned long long nbytes, unsigned long long position);

    /** 
     * @brief Escreve uma quantidade de bytes no arquivo. 
     * 
     * @param data Ponteiro de onde os dados serão lidos.
     * @param nbytes Quantidade de bytes a serem escritos. 
     * @param position Posição inicial da escrita.
     * 
     * @return Retorna a quantidade de bytes escritos. 
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    unsigned long long write(const char* data, unsigned long long nbytes, unsigned long long position);

    /** 
     * @brief Transfere os dados do arquivo remoto diretamente para um arquivo
     * local. 
     * 
     * @param position Posição do cursor no arquivo remoto.
     * @param nbytes Quantidade de bytes a serem transferidos.
     * @param fd Descritor do arquivo local.
     * @param buffer Deprecado
     * 
     * @return Retorna o número de bytes escritos.
     */
#if !defined(FTC_DOXYGEN)
    TEC_FTC_DECL
#endif
    unsigned long long transferTo(unsigned long long position, unsigned long long nbytes, FILE* fd, char* buffer);

private:
  ftc(ftc const&);
  ftc& operator=(ftc const&);

  const char* hostname;
  unsigned short tcp_port;
  const char* identifier;
  const char identifier_size;
#ifndef TEC_WINDOWS
  int sock;
#else
  SOCKET sock;
#endif
  bool writable;
  bool is_open;
  const char* access_key;
  const char access_key_size;
};


#endif
