// -*- coding: iso-8859-1 -*-
/**
* API - SDK Openbus C++
* \file openbus/OpenBusContext.hpp
*/

#ifndef TECGRAF_SDK_OPENBUS_OPENBUS_CONTEXT_H_
#define TECGRAF_SDK_OPENBUS_OPENBUS_CONTEXT_H_

#include "openbus/decl.hpp"
#include "openbus/Connection.hpp"
#include "openbus/ORBInitializer.hpp"
#include "openbus/interceptors/ClientInterceptor_impl.hpp"
#include "stubs/core.h"
#include "stubs/access_control.h"
#include "stubs/offer_registry.h"

#include <CORBA.h>
#include <boost/function.hpp>
#ifdef OPENBUS_SDK_MULTITHREAD
  #include <boost/thread.hpp>
#endif

#include <string>

namespace openbus 
{
  namespace idl_ac = tecgraf::openbus::core::v2_0::services::access_control;
  namespace idl_cr = tecgraf::openbus::core::v2_0::credential;
  namespace idl_or = tecgraf::openbus::core::v2_0::services::offer_registry;
}

namespace tecgraf 
{ 
namespace openbus 
{ 
namespace core 
{ 
namespace v2_0 
{ 
namespace services 
{ 
namespace access_control 
{

inline bool operator==(const LoginInfo &lhs, const LoginInfo &rhs)
{
  return lhs.id.in() == rhs.id.in() 
    || (lhs.id.in() && rhs.id.in() && !std::strcmp(lhs.id.in(), rhs.id.in()));
}

inline bool operator!=(const LoginInfo &lhs, const LoginInfo &rhs)
{
  return !(lhs == rhs);
}

} 
} 
} 
} 
} 
}

/**
* \brief openbus
*/
namespace openbus 
{
/**
 * \brief Cadeia de chamadas oriundas de um barramento.
 * 
 * Coleo de informaes dos logins que originaram chamadas em cadeia
 * atravs de um barramento. Cadeias de chamadas representam chamadas
 * aninhadas dentro do barramento e so teis para que os sistemas que
 * recebam essas chamadas possam identificar se a chamada foi
 * originada por entidades autorizadas ou no.
 */
struct OPENBUS_SDK_DECL CallerChain 
{
  /**
  * Barramento atravs do qual as chamadas foram originadas.
  */
  const std::string busid() const 
  {
    return _busid;
  }

	/**
   * Login para o qual a chamada estava destinada. S  possvel fazer chamadas
   * dentro dessa cadeia (atravs do mtodo joinChain da interface 
   * OpenBusContext) se o login da conexo corrente for o mesmo do target.
   *
   * No caso de conexes legadas, este campo ser nulo e ser possvel fazer
   * qualquer chamada como parte dessa cadeia. Contudo, todas as chamadas
   * feitas como parte de uma cadeia de uma chamada legada sero feitas
   * utilizando apenas o protocolo do OpenBus 1.5 (apenas com credenciais
   * legadas) e portanto sero recusadas por servios que no aceitem chamadas
   * legadas (OpenBus 1.5).
   */
  const idl_ac::LoginInfo &target() const
  {
    return _target;
  }
  
	/**
	 * Lista de informaes de login de todas as entidades que originaram as
	 * chamadas nessa cadeia. Quando essa lista  vazia isso indica que a
	 * chamada no est inclusa em outra cadeia de chamadas.
	 */
  const idl_ac::LoginInfoSeq &originators() const 
  {
    return _originators;
  }
  
  /**
   * Informao de login da entidade que realizou a ltima chamada da cadeia.
   */
  const idl_ac::LoginInfo &caller() const 
  {
    return _caller;
  }

  /**
   * \brief Construtor default que indica o valor de CallChain "vazio"
   *
   * O valor de um CallerChain default-constructed pode ser usado para
   * verificar a ausencia de CallerChain da seguinte forma:
   * 
   * CallerChain chain = openbusContext.getCallerChain();
   * if(chain != CallerChain())
   *   // Possui CallerChain
   * else
   *   // Nao possui CallerChain
   *
   */
  CallerChain() 
  {
  }
private:
  CallerChain(const std::string busid, const idl_ac::LoginInfo &t,
              const idl_ac::LoginInfoSeq &b, const idl_ac::LoginInfo &c,
              const idl_cr::SignedCallChain &d) 
    : _busid(busid), _target(t), _originators(b), _caller(c), 
      _signedCallChain(d) 
  {
  }
  
  CallerChain(const std::string busid, const idl_ac::LoginInfo &t,
              const idl_ac::LoginInfoSeq &b, const idl_ac::LoginInfo &c) 
    : _busid(busid), _target(t), _originators(b), _caller(c) 
  { 
  }
  
  std::string _busid;
  idl_ac::LoginInfo _target;
  idl_ac::LoginInfoSeq _originators;
  idl_ac::LoginInfo _caller;
  idl_cr::SignedCallChain _signedCallChain;
  const idl_cr::SignedCallChain *signedCallChain() const 
  { 
    return &_signedCallChain; 
  }
  void signedCallChain(idl_cr::SignedCallChain p) 
  { 
    _signedCallChain = p; 
  }
  friend class OpenBusContext;
  friend class openbus::interceptors::ClientInterceptor;
  friend inline bool operator==(CallerChain const &lhs, CallerChain const &rhs) 
  {
    return lhs._busid == rhs._busid && lhs._originators == rhs._originators
      && lhs._caller == rhs._caller;
  }
};

inline bool operator!=(CallerChain const &lhs, CallerChain const &rhs) 
{
  return !(lhs == rhs);
}

class OPENBUS_SDK_DECL OpenBusContext : public CORBA::LocalObject 
{
public:
  typedef boost::function<Connection* (
    OpenBusContext &context, const std::string busId, 
    const std::string loginId, const std::string operation)> 
  CallDispatchCallback;

  void onCallDispatch(CallDispatchCallback c);

  CallDispatchCallback onCallDispatch() const;

  /**
   * \brief Cria uma conexo para um barramento.
   * 
   * Cria uma conexo para um barramento. O barramento  indicado por
   * um nome ou endereo de rede e um nmero de porta, onde os
   * servios ncleo daquele barramento esto executando.
   * 
   * @param[in] host Endereo ou nome de rede onde os servios ncleo do 
   *            barramento estao executando.
   * @param[in] port Porta onde os servios ncleo do barramento esto 
   *            executando.
   * @param[in] props Lista opcional de propriedades que definem algumas
   *        configuraes sobre a forma que as chamadas realizadas ou validadas
   *        com essa conexo so feitas. A seguir so listadas as propriedades
   *        vlidas:
   *        - legacy.disable: desabilita o suporte a chamadas usando protocolo
   *          OpenBus 1.5. Por padro o suporte est habilitado.
   *        - legacy.delegate: indica como  preenchido o campo 'delegate' das
   *          credenciais enviadas em chamadas usando protocolo OpenBus 1.5. H
   *          duas formas possveis (o padro  'caller'):
   *          - caller: o campo 'delegate'  preenchido sempre com a entidade
   *            do campo 'caller' da cadeia de chamadas.
   *          - originator: o campo 'delegate'  preenchido sempre com a
   *            entidade que originou a cadeia de chamadas, que  o primeiro
   *            login do campo 'originators' ou o campo 'caller' quando este
   *             vazio.
   *   
   * @throw InvalidPropertyValue O valor de uma propriedade no  vlido.
   * @throw CORBA::Exception
   *
   * @return Conexo criada.
   */
  std::auto_ptr<Connection> createConnection(
    const std::string host, unsigned short port, 
    const Connection::ConnectionProperties &props = 
    Connection::ConnectionProperties());
   
  /**
   * \brief Define a conexo padro a ser usada nas chamadas.
   * 
   * Define uma conexo a ser utilizada como "Requester" e
   * "Dispatcher" de chamadas sempre que no houver uma conexo
   * "Requester" e "Dispatcher" especfica definida para o caso
   * especfico, como  feito atravs das operaes
   * 'setCurrentConnection' e 'setDispatcher'.
   * 
   * @param[in] conn Conexo a ser definida como conexo padro. O
   * 'ownership' da conexo no  transferida para o OpenBusContext, e
   * a conexo deve ser removida do OpenBusContext antes de destruida
   */
  Connection *setDefaultConnection(Connection *);
   
  /**
   * \brief Devolve a conexo padro.
   * 
   * Veja operao 'setDefaultConnection'.
   * 
   * \return Conexo definida como conexo padro. OpenBusContext no
   * possui ownership dessa conexo e o mesmo no  transferido para o
   * cdigo de usurio na execuo desta funo
   */
  Connection *getDefaultConnection() const;
   
  /**
   * \brief Define a conexo "Requester" do contexto corrente.
   * 
   * Define a conexo "Requester" a ser utilizada em todas as chamadas
   * feitas no contexto atual. Quando 'conn'  'null' o contexto passa
   * a ficar sem nenhuma conexo associada.
   * 
   * @param[in] conn Conexo a ser associada ao contexto corrente. O
   * 'ownership' da conexo no  transferida para o OpenBusContext, e
   * a conexo deve ser removida do OpenBusContext antes de destruida
   */
  Connection *setCurrentConnection(Connection *);
   
  /**
   * \brief Devolve a conexo associada ao contexto corrente.
   * 
   * @throw CORBA::Exception
   *
   * @return Conexo a barramento associada a thread
   * corrente. OpenBusContext no possui ownership dessa conexo e o
   * mesmo no  transferido para o cdigo de usurio na execuo
   * desta funo
   */
  Connection *getCurrentConnection() const;
       
  /**
   * \brief Devolve a cadeia de chamadas  qual a execuo corrente pertence.
   * 
   * Caso a contexto corrente (e.g. definido pelo 'CORBA::PICurrent')
   * seja o contexto de execuo de uma chamada remota oriunda do
   * barramento dessa conexo, essa operao devolve um objeto que
   * representa a cadeia de chamadas do barramento que esta chamada
   * faz parte. Caso contrrio, devolve uma cadeia de chamadas
   * 'vazia', representada por um CallerChain 'default-constructed'.
   *
   * Para verificar se a cadeia retornada  vlida, o seguinte idioma
   *  usado:
   *
   * CallerChain chain = connection.getCallerChain()
   * if(chain != CallerChain())
   *   // chain  vlido
   * else
   *   // chain  invlido
   * 
   * \return Cadeia da chamada em execuo.
   */
  CallerChain getCallerChain();
  
  /**
   * \brief Associa uma cadeia de chamadas ao contexto corrente.
   * 
   * Associa uma cadeia de chamadas ao contexto corrente, de forma que
   * todas as chamadas remotas seguintes neste mesmo contexto sejam
   * feitas como parte dessa cadeia de chamadas.
   * 
   * \param chain Cadeia de chamadas a ser associada ao contexto corrente.
   * @throw CORBA::NO_PERMISSION {minor = NoLoginCode}
   * @throw CORBA::NO_PERMISSION {minor = InvalidChainCode}
   * @throw CORBA::Exception
   */
  void joinChain(const CallerChain &chain);
  
  /**
   * \brief Faz com que nenhuma cadeia de chamadas esteja associada ao
   * contexto corrente.
   * 
   * Remove a associao da cadeia de chamadas ao contexto corrente,
   * fazendo com que todas as chamadas seguintes feitas neste mesmo
   * contexto deixem de fazer parte da cadeia de chamadas associada
   * previamente. Ou seja, todas as chamadas passam a iniciar novas
   * cadeias de chamada.
   */
  void exitChain();

  /**
   * \brief Devolve a cadeia de chamadas associada ao contexto corrente.
   * 
   * Devolve um objeto que representa a cadeia de chamadas associada
   * ao contexto corrente nesta conexo.  A cadeia de chamadas
   * informada foi associada previamente pela operao
   * 'joinChain'. Caso o contexto corrente no tenha nenhuma cadeia
   * associada, essa operao devolve uma cadeia 'vazia'
   * 'default-constructed'
   * 
   * Para verificar se a cadeia retornada  vlida, o seguinte idioma  usado:
   *
   * CallerChain chain = openbusContext.getCallerChain()
   * if(chain != CallerChain())
   *   // chain  vlido
   * else
   *   // chain  invlido
   * 
   * \return Cadeia de chamadas associada ao contexto corrente ou uma
   * cadeia 'vazia'.
   *
   * @throw CORBA::Exception
   */
  CallerChain getJoinedChain();
  
  /** 
   * ORB utilizado pela conexo. 
   */
  CORBA::ORB_ptr orb() const 
  {
    return _orb;
  }
  
  idl_or::OfferRegistry_ptr getOfferRegistry() const;
  idl_ac::LoginRegistry_ptr getLoginRegistry() const;
private:
  /**
   * OpenBusContext deve ser adquirido atraves de:
   *   orb->resolve_initial_references("OpenBusContext")
   */
  OpenBusContext(CORBA::ORB_ptr, IOP::Codec *, 
                 PortableInterceptor::SlotId slotId_joinedCallChain, 
                 PortableInterceptor::SlotId slotId_signedCallChain, 
                 PortableInterceptor::SlotId slotId_legacyCallChain,
                 PortableInterceptor::SlotId slotId_requesterConnection,
                 PortableInterceptor::SlotId slotId_receiveConnection);
  
  OpenBusContext(const OpenBusContext&);
  OpenBusContext &operator=(const OpenBusContext &);

  void orb(CORBA::ORB_ptr o) 
  {
    _orb = o;
  }

  Connection *getDispatchConnection();
  typedef std::map<std::string, Connection *> BusidConnection;
#ifdef OPENBUS_SDK_MULTITHREAD
  mutable boost::mutex _mutex;
#endif
  CORBA::ORB_ptr _orb;
  PortableInterceptor::Current_var _piCurrent;
  IOP::Codec *_codec;
  PortableInterceptor::SlotId _slotId_joinedCallChain; 
  PortableInterceptor::SlotId _slotId_signedCallChain;
  PortableInterceptor::SlotId _slotId_legacyCallChain;
  PortableInterceptor::SlotId _slotId_requesterConnection;
  PortableInterceptor::SlotId _slotId_receiveConnection;
  Connection *_defaultConnection;
  BusidConnection _busidConnection;
  CallDispatchCallback _callDispatchCallback;

  friend CORBA::ORB_ptr openbus::ORBInitializer(int &argc, char **argv);
};
}

#endif

