// -*- coding: iso-8859-1 -*-

/**
* API - SDK Openbus C++
* \file openbus/Connection.hpp
* 
*/

#ifndef TECGRAF_SDK_OPENBUS_CONNECTION_H_
#define TECGRAF_SDK_OPENBUS_CONNECTION_H_

#include "openbus/decl.hpp"
#include "stubs/scs.h"
#include "stubs/access_control.h"
#include "stubs/offer_registry.h"
#include "openbus/interceptors/ClientInterceptor_impl.hpp"
#include "openbus/interceptors/ServerInterceptor_impl.hpp"
#include "openbus/crypto/PrivateKey.hpp"
#include "openbus/crypto/PublicKey.hpp"

#include <CORBA.h>
#include <boost/function.hpp>
#ifdef OPENBUS_SDK_MULTITHREAD
  #include <boost/thread.hpp>
#endif

#include <memory>
#include <stdexcept>
#include <string>
#include <vector>
#include <cstring>

/**
* \brief openbus
*/
namespace openbus 
{
namespace idl_ac = tecgraf::openbus::core::v2_0::services::access_control;
namespace idl_or = tecgraf::openbus::core::v2_0::services::offer_registry;

class OpenBusContext;
#ifndef OPENBUS_SDK_MULTITHREAD
class RenewLogin;
#endif
  
struct OPENBUS_SDK_DECL BusChanged : public std::exception 
{ 
  const char *what() const throw()
  { 
    return "openbus::BusChanged";
  }
};

struct OPENBUS_SDK_DECL AlreadyLoggedIn : public std::exception 
{ 
  const char *what() const throw()
  { 
    return "openbus::AlreadyLoggedIn"; 
  }
};

struct OPENBUS_SDK_DECL InvalidLoginProcess : public std::exception 
{
  const char *what() const throw()
  { 
    return "openbus::InvalidLoginProcess"; 
  }
};

struct OPENBUS_SDK_DECL InvalidPropertyValue : public std::exception 
{
  InvalidPropertyValue(const std::string &p, const std::string &v) throw()
    : property(p), value(v) 
  { 
  }

  ~InvalidPropertyValue() throw()
  { 
  }

  const char *what() const throw() 
  { 
    return "openbus::InvalidPropertyValue"; 
  }

  const std::string property;
  const std::string value;
};

/**
 * \brief Conexo para acesso identificado a um barramento.
 *
 * Uma conexo  usada para realizar acessos identificados a um barramento.
 * Denominamos esses acessos identificados ao barramento de login. Cada login
 * possui um identificador nico e est sempre associado ao nome de uma
 * entidade que  autenticada no momento do estabelecimento do login.
 * H basicamente trs formas de autenticao de entidade disponveis:
 * - Por Senha: veja a operao 'loginByPassword'
 * - Por Certificado de login: veja a operao 'loginByCertificate'
 * - Por Autenticao compartilhada: veja a operao 'loginBySharedAuth'
 *
 * A entidade associada ao login  responsvel por todas as chamadas feitas
 * atravs daquela conexo e essa entidade deve ser levada em considerao
 * pelos servios ofertados no barramento para decidir aceitar ou recusar
 * chamadas.
 *
 *  possvel uma aplicao assumir mltiplas identidades ao acessar um ou mais
 * barramentos criando mltiplas conexes para esses barramentos.
 * 
 *  importante notar que a conexo no  usada diretamente pela aplicao ao
 * realizar ou receber chamadas, pois as chamadas ocorrem usando proxies e
 * servants de um ORB. As conexes que so efetivamente usadas nas chamadas do
 * ORB so definidas atravs do OpenBusContext associado ao ORB.
 */
class OPENBUS_SDK_DECL Connection 
{
public:
  typedef std::vector<std::pair<std::string, std::string> > 
    ConnectionProperties;
  /**
   * \brief Callback de login invlido.
   * 
   * Tipo que representa um objeto funo ('function object') a ser chamado
   * quando uma notificao de login invlido  recebida. Caso alguma exceo
   * ocorra durante a execuo do mtodo e no seja tratada, o erro ser
   * capturado pelo interceptador e registrado no log.
   * 
   * O tipo InvalidLoginCallback_t  um typedef de boost::function. Para
   * documentao dessa biblioteca acesse
   * http://www.boost.org/doc/libs/1_47_0/doc/html/function.html
   *
   * \param conn Conexo que recebeu a notificao de login invlido.
   * \param login Informaes do login que se tornou invlido.
   */
  typedef boost::function<void (Connection &, idl_ac::LoginInfo)> 
    InvalidLoginCallback_t;
  
  /**
  * Efetua login no barramento como uma entidade usando autenticao por senha.
  * 
  * A autenticao por senha  validada usando um dos validadores de senha
  * definidos pelo adminsitrador do barramento.
  *
  * @param[in] entity Identificador da entidade a ser autenticada.
  * @param[in] password Senha de autenticao no barramento da entidade.
  * 
  * @throw AlreadyLoggedIn A conexo j est logada.
  * @throw idl_ac::AccessDenied
  *        Senha fornecida para autenticao da entidade no foi validada pelo 
  *        barramento.
  * @throw idl::ServiceFailure 
  *        Ocorreu uma falha interna nos servios do barramento que impediu a 
  *        autenticao da conexo.
  * @throw CORBA::Exception
  */
  void loginByPassword(const std::string &entity, const std::string &password);
  
 /**
  * \brief Efetua login de uma entidade usando autenticao por certificado.
  * 
  * A autenticao por certificado  validada usando um certificado de login
  * registrado pelo adminsitrador do barramento.
  * 
  * @param[in] entity Identificador da entidade a ser conectada.
  * @param[in] privKey Chave privada da entidade utilizada na autenticao.
  * 
  * @throw AlreadyLoggedIn A conexo j est autenticada.
  * @throw idl_ac::AccessDenied 
  *        A chave privada fornecida no corresponde ao certificado da entidade 
  *        registrado no barramento indicado.
  * @throw idl_ac::MissingCertificate 
  *        No h certificado para essa entidade registrado no barramento
  *        indicado.
  * @throw idl::ServiceFailure 
  *        Ocorreu uma falha interna nos servios do barramento que impediu a 
  *        autenticao da conexo.
  * @throw CORBA::Exception
  */
  void loginByCertificate(const std::string &entity, const PrivateKey &);
  
  /**
  * \brief Inicia o processo de login por autenticao compartilhada.
  * 
  * A autenticao compartilhada permite criar um novo login compartilhando a
  * mesma autenticao do login atual da conexo. Portanto essa operao s pode
  * ser chamada enquanto a conexo estiver autenticada, caso contrrio a exceo
  * de sistema CORBA::NO_PERMISSION{NoLogin}  lanada. As informaes
  * fornecidas por essa operao devem ser passadas para a operao
  * 'loginBySharedAuth' para concluso do processo de login por autenticao
  * compartilhada. Isso deve ser feito dentro do tempo de lease definido pelo
  * administrador do barramento. Caso contrrio essas informaes se tornam
  * invlidas e no podem mais ser utilizadas para criar um login.
  * 
  * @return Um par composto de um objeto que representa o processo de login 
  *         iniciado e de um segredo a ser fornecido na concluso do processo 
  *         de login.
  *
  * @throw idl::ServiceFailure 
  *        Ocorreu uma falha interna nos servios do barramento que impediu 
  *        o estabelecimento da conexo.
  * @throw CORBA::Exception
  */
  std::pair <idl_ac::LoginProcess_ptr, idl::OctetSeq> startSharedAuth();
  
  /**
  * \brief Efetua login de uma entidade usando autenticao compartilhada.
  * 
  * A autenticao compartilhada  feita a partir de informaes obtidas a 
  * atravs da operao 'startSharedAuth' de uma conexo autenticada.
  * 
  * @param[in] loginProcess Objeto que represeta o processo de login iniciado.
  * @param[in] secret Segredo a ser fornecido na concluso do processo de login.
  * 
  * @throw InvalidLoginProcess O LoginProcess informado  invlido, por exemplo 
  *        depois de ser cancelado ou ter expirado.
  * @throw AlreadyLoggedIn A conexo j est autenticada.
  * @throw idl_ac::AccessDenied 
  *        O segredo fornecido no corresponde ao esperado pelo barramento.
  * @throw idl::ServiceFailure 
  *        Ocorreu uma falha interna nos serviços do barramento que impediu o 
  *        estabelecimento da conexo.
  * @throw CORBA::Exception
  */
  void loginBySharedAuth(idl_ac::LoginProcess_ptr loginProcess, 
                         const idl::OctetSeq &secret);
  
 /**
  * \brief Efetua logout da conexo, tornando o login atual invlido.
  * 
  * Aps a chamada a essa operao a conexo fica desautenticada, implicando que
  * qualquer chamada realizada pelo ORB usando essa conexo resultar numa
  * exceo de sistema 'CORBA::NO_PERMISSION{NoLogin}' e chamadas recebidas por
  * esse ORB sero respondidas com a exceo 'CORBA::NO_PERMISSION{UnknownBus}'
  * indicando que no foi possvel validar a chamada pois a conexo est
  * temporariamente desautenticada.
  * 
  * @return Verdadeiro se o processo de logout for concludo com xito e falso
  *         se a conexo j estiver desautenticada (login invlido).
  *
  * @throw CORBA::Exception
  */
  bool logout();
  	
  /**
   * \brief Callback a ser chamada quando o login atual se tornar invlido.
   *
   * Esse atributo  utilizado para definir um objeto funo (function object)
   * que implementa uma interface de callback a ser chamada sempre que a conexo
   * receber uma notificao de que o seu login est invlido. Essas
   * notificaes ocorrem durante chamadas realizadas ou recebidas pelo
   * barramento usando essa conexo. Um login pode se tornar invlido caso o
   * administrador explicitamente o torne invlido ou caso a thread interna de
   * renovao de login no seja capaz de renovar o lease do login a tempo. Caso
   * esse atributo seja um InvalidLoginCallback_t 'default-constructed', nenhum
   * objeto de callback  chamado na ocorrncia desse evento.
   *
   * Durante a execuo dessa callback um novo login pode ser restabelecido.
   * Neste caso, a chamada do barramento que recebeu a notificao de login
   * invlido  refeita usando o novo login, caso contrrio, a chamada original
   * lana a exceo de de sistema 'CORBA::NO_PERMISSION{NoLogin}'.
   * 
   * O tipo InvalidLoginCallback_t  um typedef de boost::function. Para
   * documentao dessa biblioteca acesse
   * http://www.boost.org/doc/libs/1_47_0/doc/html/function.html
   */
  void onInvalidLogin(InvalidLoginCallback_t p);
  
  /**
   * \brief Retorna a callback configurada para ser chamada quando o login atual
   * se torna invlido.
   */
  InvalidLoginCallback_t onInvalidLogin() const;
  
  /**
   * \brief Informaes do login dessa conexo ou 'null' se a conexo no est
   * autenticada, ou seja, no tem um login vlido no barramento.
   */
  const idl_ac::LoginInfo *login() const;
  
  /**
   * Identificador do barramento ao qual essa conexo se refere.
   */
  const std::string busid() const;
  ~Connection();  
private:
  /**
  * Connection deve ser adquirido atraves de: OpenBusContext::createConnection()
  */
  Connection(const std::string host, const unsigned short port, CORBA::ORB_ptr, 
             IOP::Codec *, PortableInterceptor::SlotId slotId_joinedCallChain, 
             PortableInterceptor::SlotId slotId_signedCallChain, 
             PortableInterceptor::SlotId slotId_legacyCallChain, 
             PortableInterceptor::SlotId slotId_receiveConnection, 
             OpenBusContext &, 
             const ConnectionProperties &props);

  Connection(const Connection &);
  Connection &operator=(const Connection &);

#ifdef OPENBUS_SDK_MULTITHREAD  
  static void renewLogin(Connection &conn, idl_ac::AccessControl_ptr acs, 
                         OpenBusContext &ctx, idl_ac::ValidityTime t);
#endif
  void login(idl_ac::LoginInfo &loginInfo, 
             idl_ac::ValidityTime validityTime);

  void checkBusid() const;
  bool _logout(bool local = true);
  CORBA::ORB_ptr orb() const 
  { 
    return _orb; 
  }

  idl_ac::LoginRegistry_var login_registry() const 
  { 
    return _login_registry; 
  }

  idl_ac::AccessControl_var access_control() const 
  { 
    return _access_control; 
  }

  const idl_ac::LoginInfo *_login() const 
  { 
#ifdef OPENBUS_SDK_MULTITHREAD
    boost::lock_guard<boost::mutex> lock(_mutex);;
#endif
    return _loginInfo.get(); 
  }

  idl_or::OfferRegistry_var getOfferRegistry() const
  { 
    return _offer_registry;
  }

  idl_ac::LoginRegistry_var getLoginRegistry() const
  {
    return _login_registry;
  }

  const std::string _host;
  const unsigned short _port;
  CORBA::ORB *_orb;
  IOP::Codec *_codec;
  PortableInterceptor::SlotId _slotId_joinedCallChain; 
  PortableInterceptor::SlotId _slotId_signedCallChain;
  PortableInterceptor::SlotId _slotId_legacyCallChain;
  PortableInterceptor::SlotId _slotId_receiveConnection;
#ifdef OPENBUS_SDK_MULTITHREAD
  boost::thread _renewLogin;
  mutable boost::mutex _mutex;
#else
  std::auto_ptr<RenewLogin> _renewLogin;
#endif
  std::auto_ptr<idl_ac::LoginInfo> _loginInfo;
  InvalidLoginCallback_t _onInvalidLogin;
  
  enum LegacyDelegate 
  {
    CALLER,
    ORIGINATOR
  };
  
  enum State 
  {
    LOGGED,
    UNLOGGED,
    INVALID
  } _state;
  
  /* Variaveis que sao modificadas somente no construtor. */
  OpenBusContext &_openbusContext;
  PrivateKey _key;
  PortableInterceptor::Current_var _piCurrent;
  scs::core::IComponent_var _iComponent;
  idl_ac::AccessControl_var _access_control;
  idl_ac::LoginRegistry_var _login_registry;
  idl_or::OfferRegistry_var _offer_registry;
  std::auto_ptr<LoginCache> _loginCache;
  std::string _busid;
  std::auto_ptr<PublicKey> _buskey;
  LegacyDelegate _legacyDelegate;
  bool _legacyEnabled;
  /**/
    
  friend class openbus::interceptors::ServerInterceptor;
  friend class openbus::interceptors::ClientInterceptor;
  friend class openbus::OpenBusContext;
};
}

#endif
