/**
* \file logger.h
*/

#ifndef LOGGER_LOGGER_H
#define LOGGER_LOGGER_H

#include <string>
#include <memory>
#include <cstdarg>
#include <list>

#ifndef LOGGER_DISABLE_THREADS
#include <boost/thread.hpp>
#endif
#include <boost/ptr_container/ptr_vector.hpp>
#include <boost/noncopyable.hpp>
#include <boost/utility.hpp>

#ifdef _WIN32
#  ifdef LOGGER_SHARED
#    ifdef LOGGER_SOURCE
#      define LOGGER_DECL __declspec(dllexport)
#    else
#      define LOGGER_DECL __declspec(dllimport)
#    endif
#  endif
#endif

#ifndef LOGGER_DECL
#define LOGGER_DECL
#endif

namespace logger {

/**
* \brief Nvel de mensagem.
* O nvel maior inclui todos os de menor prioridade. Assim, ao setar
* um nvel INFO, teremos a impresso de ERROR, WARNING e INFO.
*/
enum level
{
  disabled_level = 0,
  error_level,
  warning_level,
  info_level,
  debug_level,
};

/**
 * \brief Classe que abstrai um output para o Logger
 */
struct output_base
{
  virtual ~output_base() {}
  virtual void print(std::string const&) const = 0;
  virtual output_base* clone() const = 0;
protected:
  output_base() {}
  output_base(output_base const&) {}
  output_base& operator=(output_base const&) { return *this; }
};

class logger;

struct scope_info
{
  bool deleted;
  const char* label;

  scope_info(const char* label)
    : deleted(false), label(label) {}
};

struct scope_token
{
  scope_token prior() const
  {
    return scope_token(boost::prior(scope_iterator));
  }
private:
  scope_token(std::list<scope_info>::iterator scope_iterator)
    : scope_iterator(scope_iterator) {}
  std::list<scope_info>::iterator scope_iterator;
  friend class logger;
};

/**
 * \brief Classe que abstrai um formatter para o logger
 */
struct formatter_base
{
  virtual ~formatter_base() {}
  virtual void format(logger const&, level l, scope_token const& token, std::string&) const = 0;
  virtual formatter_base* clone() const = 0;
protected:
  formatter_base() {}
  formatter_base(formatter_base const&) {}
  formatter_base& operator=(formatter_base const&) { return *this; }
};

/**
* \brief Representa um objeto do tipo Logger  a ser utilizado como mecanismo de log.
*/ 
class logger
{
#ifndef LOGGER_DISABLE_THREADS
  boost::mutex mutex;
#endif

  /**
   * Mapeamento que descreve quais nveis esto ativados.
   */
  level level_;
#ifndef LOGGER_DISABLE_THREADS
  boost::thread_specific_ptr<std::list<scope_info> > scope_stack;
#else
  std::auto_ptr<std::list<scope_info> > scope_stack;
#endif
  boost::ptr_vector<output_base> outputs;
  boost::ptr_vector<formatter_base> formatters;
public:

  LOGGER_DECL logger(level l);
  LOGGER_DECL logger(std::auto_ptr<output_base> output);
  LOGGER_DECL logger();

  /**
   * Ativa ou desativa um determinado nvel de mensagem.
   *
   * Se o nvel est desativado, a chamada a este mtodo ativar o mesmo, e vice-versa.
   *
   * @param [in] Nvel de mensagem.
   */
  void set_level(level l)
  {
    level_ = l;
  }

  /**
   * Informa se um nvel est ativo ou no.
   *
   * @return True se o nvel est ativo, caso contrrio retorna false.
   */
  level get_level()
  {
    return level_;
  }

  void add_formatter(std::auto_ptr<formatter_base> formatter)
  {
    formatters.push_back(formatter);
  }

  void add_output(std::auto_ptr<output_base> output)
  {
    outputs.push_back(output);
  }

  LOGGER_DECL scope_token create_scope(const char* label);
  LOGGER_DECL void remove_scope(scope_token const&);
  LOGGER_DECL std::size_t identation_level(scope_token const&) const;

#ifndef LOGGER_DISABLE_VARARGS
  LOGGER_DECL void vlog(level l, scope_token const&, const char* format, va_list);
  LOGGER_DECL void log(level l, scope_token const&, const char* format, ...)
#if defined(__GNUC__) && __GNUC__ >= 4
    __attribute__ ((format (printf, 4, 5)))
#endif
    ;
#endif
  LOGGER_DECL void log(level l, scope_token const&, std::string string);
};

/**
 * \brief Representa um escopo de logging
 */
struct log_scope : boost::noncopyable
{
  log_scope(logger& l, level lev, const char* label)
    : l(l), lev(lev), label(label), token(l.create_scope(label))
  {
#ifndef LOGGER_DISABLE_VARARGS
    l.log(lev, token.prior(), "%s BEGIN", label);
#else
    std::string label_(label);
    label_ += " BEGIN";
    l.log(lev, token.prior(), label_);
#endif
  }
  ~log_scope()
  {
#ifndef LOGGER_DISABLE_VARARGS
    l.log(lev, token.prior(), "%s END", label);
#else
    std::string label_(label);
    label_ += " END";
    l.log(lev, token.prior(), label_);
#endif
    l.remove_scope(token);
  }
  
  /**
   * Registra uma mensagem utilizando o nvel pr-determinado
   * na construo do log_scope
   *
   * @param [in] message Mensagem.
   */
  void log(const char* string)
  {
    l.log(lev, token, "%s", string);
  }

#ifndef LOGGER_DISABLE_VARARGS
  void vlog(const char* format, ...)
#if defined(__GNUC__) && __GNUC__ >= 4
    __attribute__ ((format (printf, 2, 3)))
#endif
  {
    va_list argptr;
    va_start(argptr, format);
    l.vlog(lev, token, format, argptr);
  }
#endif

  /**
   * Registra uma mensagem utilizando um determinado nvel.
   *
   * @param [in] message Mensagem.
   */
  void level_log(level lev, const char* string)
  {
    l.log(lev, token, "%s", string);
  }

#ifndef LOGGER_DISABLE_VARARGS
  void level_vlog(level lev, const char* format, ...)
#if defined(__GNUC__) && __GNUC__ >= 4
    __attribute__ ((format (printf, 3, 4)))
#endif
  {
    va_list argptr;
    va_start(argptr, format);
    l.vlog(lev, token, format, argptr);
  }
#endif

  logger& l;
  level lev;
  const char* label;
  scope_token token;
};

}

#endif
