// -*- coding: iso-8859-1-unix -*-
#ifndef SERVERCONFIG_V1_04_01_H_
#define SERVERCONFIG_V1_04_01_H_

#if defined(_MSC_VER) && (_MSC_VER >= 1020)
# pragma once
#endif

#include <boost/cstdint.hpp>
#include <string>
#include <ostream>

namespace tecgraf { namespace ftc { namespace v1_04_01 {

  /**
  * @brief Define as configuraes utilizadas no BaseServer.
  *
  * O comportamento de vrias variveis internas do servidor  definido nas configuraes. O valor
  * para cada varivel depende de cada utilizao e pode ser refinado para atender melhor cada caso.
  *
  * @author Tecgraf
  */
  class ServerConfig {
  public:

    /**
    * @brief Nveis de log.
    *
    * Cada nvel de log inclui o anterior na ordem crescente que se segue. Por exemplo, LOG_WARN tem
    * um nvel que ir incluir logs dos nveis inferiores LOG_ERROR e LOG_FATAL tambm no log. Logo,
    * o ltimo nvel, LOG_ALL, ir incluir o log de todos os nveis anteriores.
    */
    enum ServerLogLevel {
      LOG_OFF,
      LOG_FATAL,
      LOG_ERROR,
      LOG_WARN,
      LOG_INFO,
      LOG_DEBUG,
      LOG_TRACE,
      LOG_ALL = LOG_TRACE
    };

    /**
    * @brief Construtor
    */
    ServerConfig(){}

    /**
    * @brief Destrutor
    */
    virtual ~ServerConfig(){}

    /**
    * @brief Retorna o endereo/IP para o servidor.
    * @return O endereo/IP para o servidor.
    */
    virtual const std::string& hostname() const = 0;

    /**
    * @brief Define o endereo/IP para o servidor.
    * @param value O endereo/IP para o servidor.
    */
    virtual void hostname(const std::string& value) = 0;

    /**
    * @brief Retorna a porta que o servidor ir escutar as conexes de novos clientes.
    * @return A porta que o servidor ir escutar as conexes de novos clientes.
    */
    virtual uint16_t port() const = 0;

    /**
    * @brief Define a porta que o servidor ir escutar as conexes de novos clientes.
    *
    *  comum a utilizao da porta 0 para conexo. A porta 0  especial, pois obriga o servidor a utilizar uma
    * porta que esteja disponvel no sistema na hora da conexo.
    *
    * @param value A porta que o servidor ir escutar as conexes de novos clientes. Caso seja 0,
    *        uma porta disponvel ser obtida.
    */
    virtual void port(uint16_t value) = 0;

    /**
    * @brief Retorna o tempo mximo para o consumo de uma requisio (DataChannelInfo).
    *
    * Aps o registro de uma requisio para um canal de dados (DataChannelInfo),
    * o cliente tem X milisegundos para se conectar ao servidor, se aps esse tempo o cliente FTC no se conectar
    * ao servidor e solicitar o canal de dados a requisio  removida.
    *
    * @return Retorna o tempo mximo em milisegundos de timeout para requisio de um canal
    */
    virtual uint32_t channel_request_timeout() const = 0;

    /**
    * @brief Define o timeout para o consumo de uma requisio (DataChannelInfo).
    *
    * Aps o registro de uma requisio para um canal de dados (DataChannelInfo),
    * o cliente tem X milisegundos para se conectar ao servidor, se aps esse tempo o cliente FTC no se conectar
    * ao servidor e solicitar o canal de dados a requisio  removida.
    *
    * @param value Tempo mximo em milisegundos que o servidor espera para
    *              remover uma requisio no consumida
    */
    virtual void channel_request_timeout(const uint32_t value) = 0;

    /**
    * @brief Retorna o tempo mximo em milisegundos que um cliente  desconectado por inatividade.
    *
    * Caso um cliente conectado no efetue nenhuma operao no tempo mximo configurado,  forado a desconexo do mesmo por timeout.
    *
    * @return Tempo mximo em milisegundos que um cliente  desconectado por inatividade.
    */
    virtual uint32_t client_timeout() const = 0;

    /**
    * @brief Define o tempo mximo em milisegundos que um cliente  desconectado por inatividade.
    *
    * Caso um cliente conectado no efetue nenhuma operao no tempo mximo configurado,  forado a desconexo do mesmo por timeout.
    *
    * @param value Tempo mximo em milisegundos que um cliente  desconectado por inatividade.
    */
    virtual void client_timeout(const uint32_t value) = 0;

    /**
    * @brief Retorna o nmero maximo de clientes que o servidor suporta.
    *
    * Caso o nmero de clientes conectados alcance o nmero mximo configurado, novos clientes que tentarem se conectar sero
    * foradamente desconectados.
    *
    * @return O nmero maximo de clientes que o servidor suporta.
    */
    virtual std::size_t max_clients() const = 0;

    /**
    * @brief Define o nmero maximo de clientes que o servidor suporta.
    *
    * Caso o nmero de clientes conectados alcance o nmero mximo configurado, novos clientes que tentarem se conectar sero
    * foradamente desconectados.
    *
    * @param value O nmero maximo de clientes que o servidor suporta.
    */
    virtual void max_clients(std::size_t value) = 0;

    /**
    * @brief Retorna o nmero mximo permitido de possveis registros  requisies de canais de dados (DataChannelInfo).
    *
    * Nmero mximo de registros  requisies de canais de dados que o servidor ir permitir.
    *
    * @return Nmero maximo de requisies.
    */
    virtual std::size_t max_channel_requests() const = 0;

    /**
    * @brief Nmero mximo permitido de possveis registros  requisies de canais de dados (DataChannelInfo).
    *
    * Nmero mximo de registros  requisies de canais de dados que o servidor ir permitir.
    *
    * @param value Nmero maximo de requisioes.
    */
    virtual void max_channel_requests(std::size_t value) = 0;

    /**
    * @brief Retorna o tamanho do espao de memria utilizado na comunicao com cada cliente.
    *
    * Esse espao de memria  criado para cada nova conexo com um cliente. Sua funo principal  a de armazenar a transmisso parcial
    * de um dado cliente/servidor ou vice-versa. O mesmo  utilizado tambm quando o canal no suporta transferTo/transferFrom.
    *
    * @return O tamanho do espao de memria utilizado na comunicao com cada cliente.
    */
    virtual std::size_t client_buffer_size() const = 0;

    /**
    * @brief Define o tamanho do espao de memria utilizado na comunicao com cada cliente.
    *
    * Esse espao de memria  criado para cada nova conexo com um cliente. Sua funo principal  a de armazenar a transmisso parcial
    * de um dado cliente/servidor ou vice-versa. O mesmo  utilizado tambm quando o canal no suporta transferTo/transferFrom.
    *
    * @param value Retorna o tamanho do espao de memria utilizado na comunicao com cada cliente.
    */
    virtual void client_buffer_size(std::size_t value) = 0;

    /**
    * @brief Retorna <c>true</c> se o servidor deve executar em modo de teste.
    *
    * No modo teste, o servidor mantm todos os registros s possveis requisies  canais de dados, mesmo que um cliente feche o canal. Permitindo
    * assim um cliente se reconectar com a mesma chave de acesso.
    *
    * @return <c>true</c> se o servidor deve executar em modo de teste.
    */
    virtual bool is_test_mode() const = 0;

    /**
    * @brief Define se o servidor deve executar em modo de teste.
    *
    * No modo teste, o servidor mantm todos os registros s possveis requisies  canais de dados, mesmo que um cliente feche o canal. Permitindo
    * assim um cliente se reconectar com a mesma chave de acesso.
    *
    * @param value Define se o servidor deve executar em modo de teste.
    */
    virtual void is_test_mode(bool value) = 0;

    /**
    * @brief Retorna o nvel de logging.
    * @return O nivel de logging.
    */
    virtual ServerLogLevel log_level() const = 0;

    /**
    * @brief Define o nvel de logging.
    * @param value O nvel de logging.
    */
    virtual void log_level(ServerLogLevel value) = 0;

    /**
    * @brief Retorna o nome do arquivo de sada que ser gravado o log.
    * @return O nome do arquivo de sada que ser gravado o log.
    */
    virtual const std::string& log_filename() const = 0;

    /**
    * @brief Define o nome do arquivo de sada que ser gravado o log.
    * @param value O nome do arquivo de sada que ser gravado o log.
    */
    virtual void log_filename(const std::string& value) = 0;

    /**
    * @brief Retorna se faz o log da data e hora.
    * @return Retorna se faz o log da data e hora.
    */
    virtual bool log_date_time() const = 0;

    /**
    * @brief Define se faz o log da data e hora.
    * @param value Informa se faz o log da data e hora.
    */
    virtual void log_date_time(bool value) = 0;

    /**
    * @brief Retorna o tamanho do pool de threads utilizado no servidor.
    *
    * O servidor implementa internamente um pool de threads para atender as mensagens cliente/servidor. Os threads so criados na
    * instanciao do servidor. Por isso, deve ser levado em conta o custo para a mquina para a criao do nmero total configurado
    * de threads.
    *
    * Deve ser levado em conta o caso em que uma requisio seja bloqueada num thread por uma implementao de um canal de
    * dados (IDataChannel). Se isso for verdade, e todos os threads forem bloqueados, o servidor ir parar de responder. Assim, nesse
    * caso onde a implementao do canal de dados tem operaes  bloqueantes, uma boa prtica seria limitar o nmero de clientes ao
    * nmero de threads menos 1. Dessa maneira sempre vai existir um thread no mnimo para recusar novas conexes.
    *
    * @return O tamanho do pool de threads utilizado no servidor.
    */
    virtual std::size_t thread_pool_size() const = 0;

    /**
    * @brief Define o tamanho do pool de threads utilizado no servidor.
    *
    * O servidor implementa internamente um pool de threads para atender as mensagens cliente/servidor. Os threads so criados na
    * instanciao do servidor. Por isso, deve ser levado em conta o custo para a mquina para a criao do nmero total configurado
    * de threads.
    *
    * Deve ser levado em conta o caso em que uma requisio seja bloqueada num thread por uma implementao de um canal de
    * dados (IDataChannel). Se isso for verdade, e todos os threads forem bloqueados, o servidor ir parar de responder. Assim, nesse
    * caso onde a implementao do canal de dados tem operaes  bloqueantes, uma boa prtica seria limitar o nmero de clientes ao
    * nmero de threads menos 1. Dessa maneira sempre vai existir um thread no mnimo para recusar novas conexes.
    *
    * @param value O tamanho do pool de threads utilizado no servidor.
    */
    virtual void thread_pool_size(const std::size_t value) = 0;

    /**
    * @brief Imprime as configurao para o stream de sada.
    * @param out Stream de sada.
    */
    virtual void print(std::ostream& out) const;

    /**
    * @brief Operador global para suportar stream
    * @param out Stream de sada.
    * @param config Objeto que implementa o ServerConfig
    */
    friend std::ostream& operator<<(std::ostream& out, const ServerConfig& config)
    {
      config.print(out);
      return out;
    }
  };

}}}

#endif

