// -*- coding: iso-8859-1-unix -*-
#ifndef REMOTEDATACHANNEL_V1_04_01_H_
#define REMOTEDATACHANNEL_V1_04_01_H_

#if defined(_MSC_VER) && (_MSC_VER >= 1020)
# pragma once
#endif

#include "ftc/FTCException.h"
#include "ftc/IDataChannel.h"
#include "ftc/Buffer.h"
#include "ftc/ftc_decl.h"
#include "ftc/AccessKey.h"

#include <string>
#include <vector>
#include <cstdio>

#if defined(FTC_DOXYGEN)
#define FTC_DECL
#endif

namespace tecgraf { namespace ftc { namespace v1_04_01
{
  /**
  * @brief Classe que representa a API de acesso a um canal de dados remoto.
  *
  * Esta classe representa o cliente FTC. Tendo o endereo/porta e chave de acesso previamente conhecido,
  *  possvel se conectar com o servidor FTC abrindo o canal de dados e, assim ento, executar operaes
  * sobre o canal de dados.
  *
  * A obteno das informaes para conexo com o servidor FTC no faz parte do escopo do FTC. Logo, deve
  * existir uma maneira para que essa informao seja transmitida do servidor FTC para o cliente FTC.
  *
  */
  struct RemoteDataChannel : public IDataChannel
  {
  public:
    /**
    * @brief Construtor
    *
    * Obs: A conexo com o servidor s  estabelecida no mtodo open.
    *
    * @param writable Indica se o canal de dados deve ser aberto para leitura.
    * @param hostname Endereo do servidor (hostname ou endereco IP vlido).
    * @param port Porta TCP do servidor (1-65535).
    * @param access_key Chave de acesso ao canal de dados (Tamanho maximo: AccessKey::MAX_SIZE).
    *
    * @throw std::invalid_argument Caso algum argumento esteja invlido
    */
    FTC_DECL
      RemoteDataChannel(const bool writable, const std::string& hostname, uint16_t port, const std::vector<uint8_t>& access_key);

    /**
    * @brief Destrutor
    */
    FTC_DECL
      virtual ~RemoteDataChannel();

    /**
    * @brief Abre o canal de dados remoto.
    *
    * Estabelece uma conexo TCP com o servidor FTC e requisita a abertura do canal de dados.
    *
    * @throw FailureException Caso j esteja aberto, falha na conexo/comunicao ponto a ponto, falha na autenticao, falha na negociao ou falha ao abrir canal de dados
    * @throw InvalidKeyException Caso a chave passada no seja reconhecida
    * @throw InvalidProtocolVersionException Verso no reconhecida ou no suportada pelo servidor
    * @throw MaxClientsReachedException O servidor se encontra com o nmero mximo de conexes suportadas preenchidas
    * @throw FileNotFoundException Arquivo solicidado no encontrado/sem view
    * @throw NoPermissionException Sem permisso para abrir o canal de dados remoto
    * @throw IllegalStateException Caso sejam lidos dados no esperados
    */
    FTC_DECL
      virtual void open();

    /**
    * @brief Fecha o canal de dados remoto.
    *
    * Informa ao servidor FTC que o arquivo deve ser fechado e em seguida fecha a conexo TCP.
    *
    * @throw FailureException Caso ocorra uma falha na comunicao ponto a ponto.
    * @throw IllegalStateException Caso o canal no esteja aberto.
    */
    FTC_DECL
      virtual void close();

    /** @copydoc IDataChannel::supportedOperations() */
    FTC_DECL
      virtual uint16_t supportedOperations() const;

    /** @copydoc IDataChannel::isOpen() */
    FTC_DECL
      virtual bool isOpen() const;

    /** @copydoc IDataChannel::remaining() */
    FTC_DECL
    virtual int64_t remaining() const;

    /** @copydoc IDataChannel::skip() */
    FTC_DECL
    virtual uint64_t skip(const uint64_t count);

    /**
    * @copydoc IDataChannel::position()
    *
    * @throw UnsupportedOperationException Operao no  suportada pelo canal remoto.
    * @throw FailureException Caso ocorra uma falha na comunicao ponto a ponto.
    * @throw IllegalStateException Caso o canal no esteja aberto.
    */
    FTC_DECL
      virtual uint64_t position() const;

    /**
    * @copydoc IDataChannel::position(const uint64_t)
    *
    * @throw UnsupportedOperationException Operao no  suportada pelo canal remoto.
    * @throw FailureException Caso ocorra uma falha na comunicao ponto a ponto.
    * @throw IllegalStateException Caso o canal no esteja aberto.
    */
    FTC_DECL
      virtual void position(const uint64_t position);

    /**
    * @copydoc IDataChannel::size() const
    *
    * @throw UnsupportedOperationException Operao no  suportada pelo canal remoto.
    * @throw FailureException Caso ocorra uma falha na comunicao ponto a ponto.
    * @throw IllegalStateException Caso o canal no esteja aberto.
    */
    FTC_DECL
      virtual uint64_t size() const;

    /**
    * @copydoc IDataChannel::size(const uint64_t)
    *
    * @throw UnsupportedOperationException Operao no  suportada pelo canal remoto.
    * @throw NoPermissionException Arquivo remoto foi aberto somente para leitura.
    * @throw ReadOnlyException Canal remoto foi aberto somente para leitura.
    * @throw FailureException Caso ocorra uma falha na comunicao ponto a ponto.
    * @throw IllegalStateException Caso o canal no esteja aberto.
    */
    FTC_DECL
      virtual void size(const uint64_t size);

    /** @copydoc IDataChannel::read(Buffer&) */
    FTC_DECL
      virtual int64_t read(Buffer& dest);

    /** @copydoc IDataChannel::read(Buffer&, const uint64_t) */
    FTC_DECL
      virtual int64_t read(Buffer& dest, const uint64_t remotePosition);

    /**
    * @brief L uma quantidade de bytes do canal remoto.
    *
    * @param dest Ponteiro para o espao onde os bytes lidos sero escritos.
    * @param nbytes Quantidades de bytes a serem lidos.
    *
    * @return Retorna o numero de bytes lidos.
    */
    FTC_DECL
      virtual int64_t read(char* dest, const std::size_t nbytes);

    /**
    * @brief L uma quantidade de bytes a partir de uma dada posio no canal remoto.
    *
    * @param dest Ponteiro para o espao onde os bytes lidos sero escritos.
    * @param nbytes Quantidades de bytes a serem lidos.
    * @param remotePosition A posio deste canal remoto a partir da qual a leitura vai ser iniciada.
    *
    * @return Retorna o numero de bytes lidos.
    */
    FTC_DECL
      virtual int64_t read(char* dest, const std::size_t nbytes, const uint64_t remotePosition);

    /**
    * @brief L uma quantidade de bytes a partir de uma dada posio no canal remoto.
    *
    * @param dest Ponteiro para o espao onde os bytes lidos sero escritos.
    * @param offset Posio em dest que comeara a ser transferido os dados do canal remoto.
    * @param nbytes Quantidades de bytes a serem lidos.
    * @param remotePosition A posio deste canal remoto a partir da qual a leitura vai ser iniciada.
    *
    * @return Retorna o numero de bytes lidos.
    */
    FTC_DECL
      virtual int64_t read(char* dest, const std::size_t offset, const std::size_t nbytes, const uint64_t remotePosition);

    /** @copydoc IDataChannel::write(Buffer &) */
    FTC_DECL
      virtual int64_t write(Buffer& source);

    /** @copydoc IDataChannel::write(Buffer&, uint64_t) */
    FTC_DECL
      virtual int64_t write(Buffer& source, const uint64_t remotePosition);

    /**
    * @brief Escreve uma quantidade de bytes do canal remoto.
    *
    * @param source Ponteiro de onde os dados sero lidos.
    * @param nbytes Quantidade de bytes para serem escritos.
    *
    * @return Retorna a quantidade de bytes escritos.
    */
    FTC_DECL
      virtual int64_t write(const char* source, const uint64_t nbytes);

    /**
    * @brief Escreve uma quantidade de bytes a partir de uma posio no canal remoto.
    *
    * @param source Ponteiro de onde os dados sero lidos.
    * @param nbytes Quantidade de bytes para serem escritos.
    * @param remotePosition A posio deste canal remoto a partir da qual a escrita vai ser iniciada.
    *
    * @return Retorna a quantidade de bytes escritos.
    */
    FTC_DECL
      virtual int64_t write(const char* source, const uint64_t nbytes, const uint64_t remotePosition);

    /**
    * @brief Escreve uma quantidade de bytes a partir de uma posio no canal remoto.
    *
    * @param source Ponteiro de onde os dados sero lidos.
    * @param offset Posio inicial em source ser transferido para o canal remoto.
    * @param nbytes Quantidade de bytes para serem escritos.
    * @param remotePosition A posio deste canal remoto a partir da qual a escrita vai ser iniciada.
    *
    * @return Retorna a quantidade de bytes escritos.
    */
    FTC_DECL
      virtual int64_t write(const char* source, const std::size_t offset, const uint64_t nbytes, const uint64_t remotePosition);

    /** @copydoc IDataChannel::transferTo(const uint64_t, const uint64_t, IDataChannel&) */
    FTC_DECL
      virtual int64_t transferTo(const uint64_t remotePosition, const uint64_t nbytes, IDataChannel& dest);

    /** @copydoc IDataChannel::transferFrom(IDataChannel&, const uint64_t, const uint64_t) */
    FTC_DECL
      virtual int64_t transferFrom(IDataChannel& source, const uint64_t remotePosition, const uint64_t nbytes);

    /**
    * @brief Transfere os dados do canal de dados remoto diretamente para um arquivo
    * local.
    *
    * @param remotePosition A posio inicial a partir da qual este canal ser lido.
    * @param nbytes Quantidade de bytes a serem transferidos.
    * @param fd Descritor do arquivo local.
    *
    * @return Retorna o nmero de bytes escritos.
    */
    FTC_DECL
      virtual int64_t transferTo(const uint64_t remotePosition, const uint64_t nbytes, FILE& fd);

    /**
    * @brief Notifica o servidor sobre o interesse de manter a conexo ativa.
    *
    * Envia uma pequena mensagem para o servidor com o intuito de manter a conexo ativa.
    */
    FTC_DECL
      void keepAlive();

    /**
    * @brief Tamanho do buffer em bytes utilizado nas operaes.
    *
    * @return O tamanho do buffer em bytes.
    */
    FTC_DECL
      std::size_t bufferSize() const;

    /**
    * @brief Define o tamanho em bytes do buffer utilizado nas operaes
    *
    * @param value O tamanho do buffer em bytes.
    */
    FTC_DECL
      void bufferSize(std::size_t value);


  private:
    /** Construtor */
    RemoteDataChannel(RemoteDataChannel const&);
    /** Operador de atribuicao */
    RemoteDataChannel & operator=(RemoteDataChannel const&);

    /** Metodo auxiliar para realizar o handshake inicial */
    void protocolVersionHandshake();
    /** Metodo auxiliar para realizar a autenticacao */
    void authenticate();
    /** Metodo auxiliar para realizar a conexao com o servidor */
    void doOpen();

    /** Metodo auxiliar para verificar se o canal esta aberto. */
    void checkIsOpen() const;

    /** Implementacao interna */
    struct ftc_socket_t;
    /** Referencia para implementacao do ftc_socket */
    ftc_socket_t* ftc_socket;

    const bool writable; /**< Indica permisso de escrita */
    const std::string hostname; /**< Endereo do servidor de arquivos */
    const uint16_t port; /**< Porta TCP do servidor */
    const std::vector<uint8_t> access_key; /**< Chave de acesso */
    uint16_t supported_operations; /**< Operacoes suportadas por esse canal */
    std::size_t buffer_size; /**< Tamanho do buffer */
  };

}}}

#endif

