// -*- coding: iso-8859-1-unix -*-
#ifndef IDATACHANNEL_V1_04_01_H_
#define IDATACHANNEL_V1_04_01_H_

#if defined(_MSC_VER) && (_MSC_VER >= 1020)
# pragma once
#endif

#include <boost/cstdint.hpp>

namespace tecgraf { namespace ftc { namespace v1_04_01 {

  namespace Operations {
    /**
     * @brief Enumeracao de operacoes suportadas pelo DataChannel.
     */
    enum Operation
    {
      no_operations = 0x00
      , get_size = 0x01
      , set_size = 0x02
      , get_position = 0x04
      , set_position = 0x08
      , read = 0x10
      , write = 0x20
      , transfer_from = 0x40
      , transfer_to = 0x80
    };
  }

  class Buffer;

  /**
   * @brief Classe que representa um canal de acesso a dados.
   *
   * Dessa classe podem ser implementados especializaes de um canal de dados para um servidor FTC trabalhar
   * com arquivos, streaming, sockets etc.
   *
   */
  class IDataChannel
  {
  public:
    /**
    * @brief Destrutor
    */
    virtual ~IDataChannel() {}

    /**
    * @brief Retorna os mtodos/operaes suportados pela implementao do canal de
    * dados. Todas as implementaes devem definir as operaes suportadas.
    *
    * Implementao obrigatoria.
    *
    * \see Operations::Operation
    *
    * @return As operaes suportadas (Flags de Operations::Operation)
    */
    virtual uint16_t supportedOperations() const = 0;

    /**
    * @brief Informa se o canal est aberto ou no
    *
    * Implementao obrigatoria.
    *
    * @return <c>true</c> se, e somente se, o canal est aberto.
    */
    virtual bool isOpen() const = 0 ;

    /**
    * @brief Retorna a quantidade de bytes no canal a partir da posio atual e o limite.
    *
    * O valor -1 significa end-of-stream.
    *
    * O valor 0 significa que atualmente no h dados, mas pode vir a ter.
    * Utilizado em live-streams
    *
    * Implementao obrigatoria.
    * @return A quantidade de bytes existentes no canal.
    */
    virtual int64_t remaining() const = 0;

    /**
    * @brief Metodo que pula/ignora uma quantidade de bytes a partir da posicao atual.
    *
    * Implementao obrigatoria.
    *
    * @param count quantidade de bytes a serem pulados.
    * @return Quantidade de bytes pulados
    */
    virtual uint64_t skip(const uint64_t count) = 0;

    /**
    * @brief Obtm a quantidade de bytes no canal a partir da posio inicial.
    *
    * Implementao opcional. No caso da operao no ser suportada deve ser
    * lanada UnsupportedOperationException.
    *
    * @return O tamanho em bytes.
    * @exception UnsupportedOperationException Caso a operao no seja suportada nesse canal.
    */
    virtual uint64_t size() const = 0;

    /**
    * @brief Define a quantidade de bytes para o canal.
    *
    * Implementao opcional. No caso da operao no ser suportada deve ser
    * lanada UnsupportedOperationException.
    *
    * @param size O novo tamanho
    * @exception UnsupportedOperationException Caso a operao no seja suportada nesse canal.
    */
    virtual void size(const uint64_t size) = 0;

    /**
    * @brief Obtm a posio atual do canal.
    *
    * Implementao opcional. No caso da operao no ser suportada deve ser
    * lanada UnsupportedOperationException.
    *
    * @return A posio atual do arquivo.
    * @exception UnsupportedOperationException Caso a operao no seja suportada nesse canal.
    */
    virtual uint64_t position() const = 0;

    /**
    * @brief Define a nova posio do arquivo.
    *
    * Implementao opcional. No caso da operao no ser suportada deve ser
    * lanada UnsupportedOperationException.
    *
    * @param position A nova posio
    * @exception UnsupportedOperationException Caso a operao no seja suportada nesse canal.
    */
    virtual void position(const uint64_t position) = 0;

    /**
    * @brief L uma sequncia de bytes do arquivo.
    *
    * Implementao opcional. No caso da operao no ser suportada deve ser
    * lanada UnsupportedOperationException.
    *
    * @param dest O buffer de destino dos bytes a serem lidos do canal
    * @return A quantidade de bytes lidos, ou -1, caso chegue ao final do arquivo.
    * @exception UnsupportedOperationException Caso a operao no seja suportada nesse canal.
    */
    virtual int64_t read(Buffer& dest) = 0;

    /**
    * @brief L uma sequncia de bytes do arquivo.
    *
    * Implementao opcional. No caso da operao no ser suportada deve ser
    * lanada UnsupportedOperationException.
    *
    * @param dest O buffer de destino dos bytes a serem lidos do canal
    * @param remotePosition A posio deste canal partir da qual a leitura vai ser iniciada.
    * @return A quantidade de bytes lidos.
    * @exception UnsupportedOperationException Caso a operao no seja suportada nesse canal.
    */
    virtual int64_t read(Buffer& dest, const uint64_t remotePosition) = 0;

    /**
    * @brief Escreve uma sequncia de bytes no arquivo.
    *
    * Implementao opcional. No caso da operao no ser suportada deve ser
    * lanada UnsupportedOperationException.
    *
    * @param source O buffer de origem dos bytes a serem escritos no canal
    * @return A quantidade de bytes escritos.
    * @exception UnsupportedOperationException Caso a operao no seja suportada nesse canal.
    */
    virtual int64_t write(Buffer& source) = 0;

    /**
    * @brief Escreve uma sequncia de bytes no arquivo.
    *
    * Implementao opcional. No caso da operao no ser suportada deve ser
    * lanada UnsupportedOperationException.
    *
    * @param source O buffer de origem dos bytes a serem escritos no canal
    * @param remotePosition A posio deste canal a partir da qual a escrita vai ser iniciada.
    * @return A quantidade de bytes escritos.
    * @exception UnsupportedOperationException Caso a operao no seja suportada nesse canal.
    */
    virtual int64_t write(Buffer& source, const uint64_t remotePosition) = 0;

    /**
    * @brief Transfere os dados do arquivo para um fluxo de sada.
    *
    * Implementao opcional. No caso da operao no ser suportada deve ser
    * lanada UnsupportedOperationException.
    *
    * @param remotePosition A posio inicial a partir da qual este canal ser lido.
    * @param nbytes A quantidade de bytes para serem transferidos.
    * @param dest O fluxo de sada que receber os dados do canal.
    * @return A quantidade de bytes efetivamente transferidos.
    * @exception UnsupportedOperationException Caso a operao no seja suportada nesse canal.
    */
    virtual int64_t transferTo(const uint64_t remotePosition, const uint64_t nbytes, IDataChannel& dest) = 0;

    /**
    * @brief Transfere os dados de um fluxo de entrada para o arquivo.
    *
    * Implementao opcional. No caso da operao no ser suportada deve ser
    * lanada UnsupportedOperationException.
    *
    * @param source O fluxo de entrada.
    * @param remotePosition A posio inicial a partir da qual este canal ser escrito.
    * @param nbytes A quantidade de bytes para serem transferidos
    * @return A quantidade de bytes efetivamente transferidos
    * @exception UnsupportedOperationException Caso a operao no seja suportada nesse canal.
    */
    virtual int64_t transferFrom(IDataChannel& source, const uint64_t remotePosition, const uint64_t nbytes) = 0;
  };

}}}

#endif

