// -*- coding: iso-8859-1-unix -*-
#ifndef FTC_EXCEPTION_V1_04_01_H_
#define FTC_EXCEPTION_V1_04_01_H_

#if defined(_MSC_VER) && (_MSC_VER >= 1020)
# pragma once
#endif

#include "ftc/ErrorCodes.h"

#include <string>
#include <stdexcept>

namespace tecgraf { namespace ftc { namespace v1_04_01
{
  /**
  * @brief Define a excecao base da biblioteca FTC.
  */
  class FTCException : public std::exception
  {
  public:
    /**
    * @brief Construtor
    *
    * @param code Cdigo de erro da exceo em ErrorCodes::ErrorCode
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    FTCException( ErrorCodes::ErrorCode code, const std::string& msg, const std::string& location)
      : code(code), msg(msg), m_location(location)
    {
    }

    /**
    * @brief Destrutor
    */
    virtual ~FTCException() throw() {}

    /**
    * @brief Retorna a mensagem de erro associada a exceo.
    *
    * @return Mensagem de erro.
    */
    char const* what() const throw()
    {
      return msg.c_str();
    }

    /**
    * @brief Retorna a informao adicional da localizao do erro associado a exceo.
    *
    * @return Informao adicional da localizao de erro.
    */
    const char* location() const
    {
      return m_location.c_str();
    }

    /**
    * @brief Retorna o cdigo de erro da exceo
    *
    * @return ErrorCodes::ErrorCode
    */
    ErrorCodes::ErrorCode error_code()
    {
      return code;
    }

  protected:
    /**
    * @brief Enumeracao contendo o codigo de erro
    */
    ErrorCodes::ErrorCode code;
    /**
    * @brief String contendo mensagem de erro
    */
    std::string msg;
    /**
    * @brief String contendo localizacao do erro
    */
    std::string m_location;
  };

  /**
  * @brief Essa exceo  lanada quando o metodo close  chamado sem
  * que o arquivo remoto esteja aberto
  */
  class FileNotOpenException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit FileNotOpenException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::file_not_open, msg, location){}
  };

  /**
  * @brief Essa exceo  lanada quando o servidor atingiu o nmero mximo
  * de clientes.
  */
  class MaxClientsReachedException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit MaxClientsReachedException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::max_clients_reached, msg, location){}
  };

  /**
  * @brief Essa exceo  lanada quando o arquivo est reservado para
  * acesso exclusivo.
  */
  class FileLockedException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit FileLockedException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::file_locked, msg, location){}
  };

  /**
  * @brief Essa exceo  lanada quando uma operao falha por causa
  * de falta de permisses para realizar-la.
  */
  class NoPermissionException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit NoPermissionException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::no_permission, msg, location){}
  };

  /**
  * @brief Essa exceo  lanada quando o arquivo no  encontrado no
  * servidor.
  */
  class FileNotFoundException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit FileNotFoundException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::read_only, msg, location) {}
  };

  /**
  * @brief Essa exceo  lanada quando a chave de acesso utilizada na
  * conexo  invlida.
  */
  class InvalidKeyException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit InvalidKeyException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::invalid_key, msg, location){}
  };

  /**
  * @brief Essa exceo  lanada quando uma falha desconhecida aconteceu.
  */
  class FailureException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit FailureException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::failure, msg, location){}
  };

  /**
  * @brief Essa exceo  lanada quando uma tentativa de escrita  realizada
  * em um arquivo aberto somente para leitura.
  */
  class ReadOnlyException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit ReadOnlyException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::read_only,msg,location){}
  };

  /**
  * @brief Essa exceo  lanada quando o servidor no suporta a verso
  * do protocolo implementada por este cliente.
  */
  class InvalidProtocolVersionException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit InvalidProtocolVersionException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::invalid_version, msg, location){}
  };

  /**
  * @brief Essa exceo  lanada quando um comando de leitura que
  * requisitou mas bytes do que existiam no arquivo atinge o final
  * do arquivo.
  */
  class EndOfFileException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit EndOfFileException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::end_of_file,msg, location){}
  };

  /**
  * @brief Essa exceo  lanada quando uma operacao nao suportada
  * tenta ser executada
  */
  class UnsupportedOperationException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit UnsupportedOperationException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::unsupported_operation, msg, location){}
  };

  /**
  * @brief Representa uma exceo onde o objeto est num estado inconsistente
  */
  class IllegalStateException : public FTCException
  {
  public:
    /**
    * @brief Construtor
    *
    * @param msg Mensagem de erro
    * @param location Informao adicional de onde ocorreu o erro
    */
    explicit IllegalStateException( const std::string& msg, const std::string& location )
      : FTCException(ErrorCodes::invalid_protocol_message, msg, location) {}
  };

}}}

#endif

