// -*- coding: iso-8859-1-unix -*-
#ifndef FILECHANNELINFO_V1_04_01_H_
#define FILECHANNELINFO_V1_04_01_H_

#if defined(_MSC_VER) && (_MSC_VER >= 1020)
# pragma once
#endif

#include <iostream>
#include <boost/chrono.hpp>
#include <boost/cstdint.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/scoped_ptr.hpp>

namespace tecgraf { namespace ftc { namespace v1_04_01 {

  class AccessKey;

  /**
  * @brief Representa informaes sobre o acesso a um canal de dados. \see DataChannelInfo_ptr
  *
  * O servidor FTC retorna as informaes de um registro de uma requisio para um canal de dados por essa classe. As informaes
  * contidas nessa classe devem de alguma maneira serem enviadas para o cliente FTC (Isto est fora do escopo do FTC).
  *
  */
  class DataChannelInfo {
  public:

    /**
    * @brief Cria informaes sobre o acesso a um canal de dados.
    *
    * O servidor FTC utiliza esse construtor no momento que  registrado uma requisio para um canal de dados.
    *
    * @param access_key A chave de acesso.
    * @param requester Dado para utilizao livre do usurio da API.
    * @param hostname O endereo da mquina do servidor.
    * @param port A porta da mquina do servidor.
    * @param data_id O identificador do canal de dados.
    * @param use_transfer_to Informa a possibilidade de utilizao do mtodo TransferTo caso o canal suporte.
    */
    DataChannelInfo(const AccessKey * const access_key, void * requester, const std::string& data_id, const std::string& hostname, uint16_t port, bool use_transfer_to);

    /**
    * @brief Destrutor
    */
    ~DataChannelInfo();

    /**
    * @brief Obtm o endereo da mquina do servidor.
    *
    * @return O endereo da mquina do servidor.
    */
    const std::string& hostname() const;

    /**
    * @brief Obtm a porta da mquina do servidor.
    *
    * @return A porta da mquina do servidor.
    */
    uint16_t port() const;

    /**
    * @brief Obtm a chave de acesso.
    *
    * @return A chave de acesso.
    */
    const AccessKey* access_key() const;

    /**
    * @brief Obtm o identificador do canal de dados.
    *
    * @return O identificador do canal de dados.
    */
    const std::string& data_id() const;

    /**
    * @brief Obtm o dado de utilizao livre para o usurio da API.
    *
    * @return O dado de utilizao livre para o usurio da API.
    */
    void * requester() const;

    /**
    * @brief Informa a possibilidade de utilizao do mtodo TransferTo se o canal suportar.
    *
    * @return Se libera a utilizao do mtodo TransferTo caso o canal suporte.
    */
    bool use_transfer_to() const;

    /**
    * @brief Informa se essa requisio j expirou passado o timeout.
    *
    * @param timeout_ms Timeout em milisegundos.
    *
    * @return <c>true</c> se essa requisio j expirou.
    */
    bool check_timeout(const uint64_t& timeout_ms) const;
  private:
    std::string m_hostname;
    uint16_t m_port;
    std::string m_data_id;
    void * m_requester;
    boost::scoped_ptr<const AccessKey> m_access_key;
    boost::chrono::system_clock::time_point m_creation_time;
    bool m_use_transfer_to;
  };

  /**
  * @brief Smart pointer que mantm um contador de referncia para destruir automaticamente um ponteiro para DataChannelInfo.
  */
  typedef boost::shared_ptr<DataChannelInfo> DataChannelInfo_ptr;

}}}

#endif

