// -*- coding: iso-8859-1-unix -*-
#ifndef  BUFFER_V1_04_01_H_
#define  BUFFER_V1_04_01_H_

#if defined(_MSC_VER) && (_MSC_VER >= 1020)
# pragma once
#endif

#include "ftc/ftc_decl.h"

#include <boost/cstdint.hpp>

/**
* @file Buffer.h
*
* @brief <b>Buffer API</b>
*
*/
namespace tecgraf { namespace ftc { namespace v1_04_01
{
  /**
  * @brief Implementao da classe Buffer do Java com mais a gravao padro em BIG ENDIAN dos tipos short, int e long
  * do protocolo FTC.
  *
  * @author Tecgraf/Puc-Rio
  */
  class FTC_DECL Buffer {
  public:
    /**
    * @brief The new buffer's position will be zero, its limit will be its capacity, and its mark will be undefined. It will
    * have a backing array, and its array offset will be zero.
    * @param capacity The new buffer's capacity, in bytes
    */
    Buffer( size_t capacity );

    /**
    * @brief Wraps a byte array into a buffer.
    *
    * The new buffer will be backed by the the given byte array; that is, modifications to the buffer will cause the array
    * to be modified and vice versa. The new buffer's capacity and limit will be array.length, its position will be zero,
    * and its mark will be undefined. Its backing array will be the given array, and its array offset will be zero.
    *
    * @param buffer The array that will back this buffer
    * @param size The size of array
    */
    Buffer(char * buffer, size_t size );

    /**
    * @brief Destrutor
    */
    ~Buffer();

    /**
    * @brief Clears this buffer. The position is set to zero, the limit is set to the capacity, and the mark is discarded.
    *
    * Invoke this method before using a sequence of channel-read or put operations to fill this buffer. For example:
    *
    * @code
    *     buf.clear();     // Prepare buffer for reading
    *     in.read(buf);    // Read data
    * @endcode
    *
    * This method does not actually erase the data in the buffer, but it is named as if it did because it will most
    * often be used in situations in which that might as well be the case.
    */
    void clear();

    /**
    * @brief Flips this buffer. The limit is set to the current position and then the position is set to zero. If the mark is
    * defined then it is discarded.
    *
    * After a sequence of channel-read or put operations, invoke this method to prepare for a sequence of channel-write
    * or relative get operations. For example:
    *
    * @code
    *     buf.write(magic);  // Prepend header
    *     in.read(buf);      // Read data into rest of buffer
    *     buf.flip();        // Flip buffer
    *     out.write(buf);    // Write header + data to out
    * @endcode
    *
    * This method is often used in conjunction with the compact method when transferring data from one place to another.
    */
    void flip();

    /**
    * @brief Gets this buffer's limit.
    *
    * @return Return this buffer's limit.
    */
    size_t limit() const;


    /**
    * @brief Sets this buffer's limit.
    *
    * If the position is larger than the new limit then it is set to the new limit. If
    * the mark is defined and larger than the new limit then it is discarded. The new limit value must be non-negative
    * and no larger than this buffer's capacity
    *
    * @exception std::range_error If the preconditions on new limit do not hold
    */
    void limit( size_t limit );

    /**
    * @brief Sets this buffer's mark at its position.
    */
    void mark();

    /**
    * @brief Buffer's capacity.
    *
    * @return Returns this buffer's capacity.
    */
    size_t size() const;

    /**
    * @brief Gets this buffer's position.
    *
    * @return Returns this buffer's position.
    */
    size_t position() const;

    /**
    * @brief Sets this buffer's position.
    *
    * If the mark is defined and larger than the new position then it is discarded.
    * The new position value must be non-negative and no larger than the current limit.
    */
    void position( size_t position );

    /**
    * @brief Returns the number of elements between the current position and the limit (remaining in this buffer).
    *
    * @return
    */
    size_t remaining() const;

    /**
    * @brief Tells whether there are any elements between the current position and the limit.
    * @return Returns true if, and only if, there is at least one element remaining in this buffer
    */
    bool hasRemaining() const;

    /**
    * @brief Resets this buffer's position to the previously-marked position.
    *
    * Invoking this method neither changes nor discards the mark's value.
    *
    * @exception std:logic_error If the mark has not been set
    */
    void reset();

    /**
    * @brief  Rewinds this buffer. The position is set to zero and the mark is discarded.
    * Invoke this method before a sequence of channel-write or get operations, assuming
    * that the limit has already been set appropriately. For example:
    *
    * @code
    *     out.put(buf);    // Write remaining data
    *     buf.rewind();    // Rewind buffer
    *     buf.get(array);  // Copy data into array
    * @endcode
    */
    void rewind();

    /**
    * @brief Writes eight bytes containing the given long value, in the current byte order, into this buffer at
    * the current position, and then increments the position by eight.
    *
    * @param value The long value to be written
    * @exception std::overflow_error If there are fewer than eight bytes remaining in this buffer
    */
    void putLongInt( uint64_t value );

    /**
    * @brief Reads the next eight bytes at this buffer's current position, composing them into a long value according
    * to the current byte order, and then increments the position by eight.
    *
    * @return The long value at the buffer's current position
    * @exception std::underflow_error If there are fewer than eight bytes remaining in this buffer
    */
    uint64_t getLongInt();

    /**
    * @brief Writes four bytes containing the given int value, in the current byte order, into this buffer at the current position, and then increments the position by four.
    * @param value The int value to be written
    * @exception std::overflow_error If there are fewer than four bytes remaining in this buffer
    */
    void putInteger( uint32_t value );

    /**
    * @brief Reads the next four bytes at this buffer's current position, composing them into an int value
    * according to the current byte order, and then increments the position by four.
    *
    * @return The int value at the buffer's current position
    * @exception std::underflow_error If there are fewer than four bytes remaining in this buffer
    */
    uint32_t getInteger();

    /**
    * @brief Writes two bytes containing the given short value, in the current byte order, into this buffer at the current
    * position, and then increments the position by two.
    *
    * @param value The short value to be written
    * @exception std::overflow_error If there are fewer than two bytes remaining in this buffer
    */
    void putShortInt( uint16_t value );

    /**
    * @brief Reads the next two bytes at this buffer's current position, composing them into a short value according to the
    * current byte order, and then increments the position by two.
    *
    * @return The short value at the buffer's current position
    * @exception std::underflow_error If there are fewer than two bytes remaining in this buffer
    */
    uint16_t getShortInt();

    /**
    * @brief Writes the given byte into this buffer at the current position, and then increments the position.
    *
    * @param value The byte to be written
    * @exception std::overflow_error If this buffer's current position is not smaller than its limit
    */
    void putByte( uint8_t value );

    /**
    * @brief Relative method. Reads the byte at this buffer's current position, and then increments the position.
    *
    * @return The byte at the buffer's current position
    * @exception std::underflow_error If the buffer's current position is not smaller than its limit
    */
    uint8_t getByte();

    /**
    * @brief Absolute get method. Reads the byte at the given index.
    * @param index The index from which the byte will be read
    * @return The byte at the given index
    * @exception std::out_of_range If index is negative or not smaller than the buffer's limit
    */
    uint8_t getByte(size_t index) const;

    /**
    * @brief Relative bulk put method  (optional operation).
    * This method transfers the entire content of the given source byte array into this buffer. An invocation of this
    * method of the form dst.put(a) behaves in exactly the same way as the invocation:
    * @code
    *     dst.putBytes(a, 0, nbytes)
    * @endcode
    * @param source The array from which bytes are to be read
    * @param nbytes The size of array
    * @exception std::overflow_error If there is insufficient space in this buffer
    */
    void putBytes( void* source, size_t nbytes );

    /**
    * @brief Relative bulk get method.
    *
    * This method transfers bytes from this buffer into the given destination array. An invocation of this method of the
    * form src.get(a) behaves in exactly the same way as the invocation:
    * @code
    *     src.getBytes(a, 0, a.length)
    * @endcode
    * @param dest The array into which bytes are to be written
    * @param nbytes The size of array
    * @exception std::underflow_error If there are fewer than length bytes remaining in this buffer
    */
    void getBytes( void* dest, size_t nbytes );


    /**
    * @brief  A typesafe enumeration for byte orders.
    */
    enum ByteOrder
    {
      /**
      * @brief Constant denoting big-endian byte order. In this order, the bytes of a multibyte value are ordered from most significant to least significant.
      */
      BIG_ENDIAN_ORDER,
      /**
      * @brief Constant denoting little-endian byte order. In this order, the bytes of a multibyte value are ordered from least significant to most significant.
      */
      LITTLE_ENDIAN_ORDER
    };

    /**
    * @brief Gets this buffer's byte order.
    *
    * @return Returns this buffer's byte order.
    */
    ByteOrder order() const;

    /**
    * @brief Sets this buffer's byte order.
    *
    * @param order The new byte order.
    */
    void order(ByteOrder order);

    /**
    * @brief Returns the byte array that backs this buffer (optional operation).
    * Modifications to this buffer's content will cause the returned array's content to be modified, and vice versa.
    *
    * @return The array that backs this buffer
    */
    char * buffer();

  private:
    const bool m_wrapped;
    char * const m_buffer;
    const size_t m_size;
    size_t m_limit;
    size_t m_mark;
    size_t m_position;
    static const ByteOrder m_native_order;
    ByteOrder m_order;
  };

}}}

#endif

