// -*- coding: iso-8859-1-unix -*-
#ifndef BASESERVER_V1_04_01_H_
#define BASESERVER_V1_04_01_H_

#if defined(_MSC_VER) && (_MSC_VER >= 1020)
# pragma once
#endif

#include <boost/asio.hpp>
#include <boost/chrono.hpp>
#include <boost/cstdint.hpp>
#include <boost/detail/atomic_count.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/asio/system_timer.hpp>
#include <boost/thread.hpp>
#include <boost/noncopyable.hpp>

#include <string>
#include <stdexcept>
#include <set>

namespace tecgraf { namespace ftc { namespace v1_04_01
{

  class AccessKey;
  class BaseSession;
  class ServerConfig;
  class Connection;
  class ServerExceptionHandler;
  typedef boost::shared_ptr<Connection> Connection_ptr;

  /**
  * @brief Representa a classe base para um servidor TCP.
  *
  * @author Tecgraf/PUC-Rio
  */
  class BaseServer : private boost::noncopyable
  {
  public:
    /**
    * @brief Construtor
    * @param config Configuraes utilizadas pelo servidor
    */
    BaseServer(ServerConfig& config);

    /**
    * @brief Destrutor
    */
    virtual ~BaseServer();

    /**
    * @brief Retorna o objeto com as configuraes utilizadas pelo servidor
    */
    const ServerConfig& config() const;

    /**
    * @brief Inicia o servidor.
    *
    * Este mtodo  bloqueante.
    *
    * Para capturar a porta atribuda na inicializao  necessrio perguntar para o mtodo config().port().
    *
    */
    virtual void dispatch();

    /**
    * @brief Aguarda a inicializao do servidor at que o mesmo comece a aceitar conexes.
    *
    * Este mtodo  til, por exemplo, se o servidor  inicializado num thread  parte, e por vezes  necessrio
    * saber quando o mesmo est pronto para receber conexes.
    *
    */
    void wait_ready();

    /**
    * @brief Aguarda a inicializao do servidor at que o mesmo comece a aceitar conexes passando um tempo mximo para esperar.
    *
    * Este mtodo  til, por exemplo, se o servidor  inicializado num thread  parte, e por vezes  necessrio
    * saber quando o mesmo est pronto para receber conexes.
    *
    * @param timeout_ms Tempo mximo em milisegundos para aguardar a inicializao do servidor at que o mesmo comece a aceitar conexes.
    */
    void timed_wait_ready(uint32_t timeout_ms);

    /**
    * @brief Sinaliza para o servidor interromper.
    *
    * Este mtodo ir sinalizar o servidor para ser interrompido.
    *
    */
    virtual void stop();

    /**
    * @brief Informa se o servidor est parado.
    *
    * @return <c>true</c> se o servidor est parado.
    */
    bool stopped() const;

    /**
    * @brief Cadastra um manipulador de excees
    * @param exception_handler O novo manipulador de excees.
    */
    void exception_handler(ServerExceptionHandler* exception_handler);

    /**
    * @brief Retorna o manipulador de excees
    * @return O manipulador de excees cadastrado
    */
    ServerExceptionHandler* exception_handler() const;

    /**
    * @brief Cria uma sesso de uma conexo com um cliente.
    * @param conn Objeto com a referncia para a conexo.
    */
    virtual BaseSession * create_session(Connection& conn) = 0;

    /**
    * @brief Notificao recebida quando uma conexo  terminada
    * @param conn A conexo.
    */
    void on_connection_stop(Connection_ptr conn);

    /// @cond INTERNAL_IMPLEMENTATION
    /**
    * @brief Chamado quando uma exceo  lanada no servidor.
    * @param msg Mensagem da exceo quando no especfica - <c>catch(...)</c>.
    */
    void exception_raised(const std::string& msg);

    /**
    * @brief Chamado quando uma exceo  lanada no servidor.
    * @param msg Mensagem da exceo quando no especfica - <c>catch(...)</c>.
    * @param data_id Referncia ao identificador do canal de dados
    */
    void exception_raised(const std::string& msg, const std::string& data_id);

    /**
    * @brief Chamado quando uma exceo  lanada no servidor.
    * @param e Objeto da exceo.
    */
    void exception_raised(const std::exception& e);

    /**
    * @brief Chamado quando uma exceo  lanada no servidor.
    * @param e Objeto da exceo.
    * @param data_id Referncia ao identificador do canal de dados
    */
    void exception_raised(const std::exception& e, const std::string& data_id);
    /// @endcond
  protected:
    /**
    * @brief Inicia assincronamente o timer de timeout
    */
    void start_timeout_timer();
    /**
    * @brief Verifica se ocorreu timeout
    */
    virtual void check_timeout(const uint64_t& timeout_ms);
  private:
    void connect();

    void start_accept();
    void handle_accept(const boost::system::error_code& e, Connection_ptr conn);

    void handle_timeout(const boost::system::error_code& ec);

    void handle_stop();

    void run();

    ServerConfig& m_config;

    boost::asio::io_service io_service;

    /// Verifica o timeout para requisio do channel
    boost::asio::system_timer timeout_timer;

    /// Acceptor used to listen for incoming connections.
    boost::asio::ip::tcp::acceptor acceptor;

    /// O manipulador de excees lanadas pelo servidor
    ServerExceptionHandler* m_exception_handler;

    /// Guarda o nmero de conexes ativas
    boost::shared_ptr<boost::detail::atomic_count> num_connections;

    /// mutex para accesso a lista de conexes
    boost::mutex connections_mutex;

    /// Guarda as conexes existentes para fazer um _graceful_ _shutdown_
    std::set<Connection_ptr> connections;

    /// mutex para fazer o wait_ready
    boost::mutex acceptor_mutex;

    /// varivel para notificar quem est aguardando wait_ready
    boost::condition_variable acceptor_notifier;
  };

}}}

#endif

