// -*- coding: iso-8859-1-unix -*-
#ifndef ACCESSKEY_V1_04_01_H_
#define ACCESSKEY_V1_04_01_H_

#if defined(_MSC_VER) && (_MSC_VER >= 1020)
# pragma once
#endif

#include <boost/cstdint.hpp>
#include <vector>

namespace tecgraf { namespace ftc { namespace v1_04_01 {

  /**
  * @author Tecgraf/PUC-Rio
  *
  * @brief Representa uma chave de acesso a um canal de dados.
  *
  * A classe AccessKey contm a chave de acesso que identificar unicamente um canal de dados oferecido pelo servidor FTC. Um
  * cliente FTC utiliza essa chave de acesso para abrir o canal de dados e assim ento executar as operaes
  * no canal.
  *
  * A chave de acesso pode ser gerada automaticamente. Neste caso, a mesma  preenchida com bytes aleatrios utilizando uma
  * implementao de UUID - Universally Unique Identifiers). Para isso, crie uma instncia de AccessKey sem passar nenhum
  * parmetro. Do contrrio,  possvel informar uma chave de acesso prpria passando um array de bytes como parmetro.
  *
  * A chave de acesso pode ter um tamanho de no mximo MAX_SIZE bytes. J a chave de acesso gerada
  * automaticamente tem um tamanho fixo de DEFAULT_SIZE bytes.
  *
  */
  class AccessKey {
  public:
    /// Tamanho padro de uma chave de acesso.
    static const uint8_t DEFAULT_SIZE = 16U;
    /// Tamanho mximo que uma chave de acesso pode ter
    static const uint8_t MAX_SIZE = 255U;

    /**
    * @brief Construtor com gerao automtica da chave de acesso
    *
    * Cria uma chave de acesso com um identificador gerado automaticamente com tamanho fixo de DEFAULT_SIZE bytes.
    *
    */
    AccessKey();

    /**
    * @brief Construtor de cpia
    *
    * @param key Chave de Acesso
    */
    AccessKey(AccessKey const& key);

    /**
    * @brief Construtor com chave de acesso prpria
    *
    * Cria uma chave de acesso passando um identificador prprio.  possvel manter uma referncia para o array de bytes
    * ou fazer uma cpia em memria do mesmo.
    *
    * @param new_key Chave de acesso como buffer.
    * @param size Tamanho da chave de acesso.
    * @param copy Indica se uma cpia do buffer dever ser realizada.
    */
    AccessKey(unsigned char * new_key, uint8_t size, bool copy);

    /**
    * @brief Construtor com chave de acesso prpria
    *
    * Cria uma chave de acesso passando um identificador prprio.
    *
    * @param new_key Chave de acesso como array de bytes.
    */
    AccessKey(const std::vector<uint8_t>& new_key);

    /**
    * @brief Destrutor
    */
    ~AccessKey();

    /**
    * @brief Retorna o buffer da chave de acesso como um array de bytes.
    *
    * @return Retorna o buffer da chave de acesso como um array de bytes.
    */
    const unsigned char * key() const;

    /**
    * @brief Retorna o tamanho da chave de acesso em bytes
    *
    * @return Retorna o tamanho da chave de acesso em bytes
    */
    uint8_t size() const;

    AccessKey & operator=(AccessKey const &);
    bool operator==(AccessKey const &) const;
    bool operator<=(AccessKey const &) const;
    bool operator>=(AccessKey const &) const;
    bool operator<(AccessKey const &) const;
    bool operator>(AccessKey const &) const;
  private:
    unsigned char *m_key;
    uint8_t m_size;
    bool deleteKey;
  };

  /**
  * @brief Classe para fazer a comparao entre ponteiros de AccessKey's em FTCServer.
  */
  class key_less_comp
  {
  public:
    bool operator() (const AccessKey* lhs, const AccessKey* rhs) const 
    {
      return ((*lhs) < (*rhs));
    }
  };

}}}

#endif

