#include "pch.h"

#include <ftc/DefaultDataProvider.h>
#include <ftc/FTCServerConfig.h>
#include <ftc/FTCServer.h>
#include <ftc/AccessKey.h>

#include <string>
#include <csignal>

#include <boost/asio.hpp>
#include <boost/cstdint.hpp>
#include <boost/program_options.hpp>
#include <boost/filesystem.hpp>

#include <boost/version.hpp>
#if (BOOST_VERSION == 105100)
namespace boost { namespace algorithm { namespace detail {
inline unsigned hex_char_to_int ( char c );
}}}
#endif
#include <boost/algorithm/hex.hpp>

#if defined(_MSC_VER)
#pragma comment(lib, "ftc.lib")
#endif

#include "Config.h"

void add_file_to_server(FTCServer &server, const std::string &file);

//[def_add_files_to_server
void add_files_to_server(FTCServer &server, const std::vector<std::string> &files)
{
  for (std::vector<std::string>::const_iterator it = files.begin(); 
       it != files.end(); it++)
  {
    add_file_to_server(server, *it);
  }
}

void add_file_to_server(FTCServer &server, const std::string &file)
{
  boost::filesystem::path path(
    boost::filesystem::system_complete(boost::filesystem::path(file)));
  
  if (boost::filesystem::is_directory(path))
  {
    boost::filesystem::directory_iterator end;
    for(boost::filesystem::directory_iterator dir(path); dir != end; dir++)
    {            
      add_file_to_server(server, dir->path().string());
    }
    return;
  }

  std::cout << "Gerando identificador unico para o arquivo: " << path.string() 
       << std::endl;
  DataChannelInfo_ptr info = 
    server.register_data_channel_info(0, path.string()); /*< 
      Cria uma descrição do canal através do caminho do arquivo.>*/
  std::cout << "Identificador unico para arq. gerado (hex): " 
       << boost::algorithm::hex(
         std::string(info->access_key()->key(), 
                     info->access_key()->key() + info->access_key()->size())) 
       << std::endl; /*< Disponibiliza a chave de acesso. >*/                  
}
//]

//[stop_handler
void stop_handler(
  const boost::system::error_code &error, 
  int signal_number, 
  FTCServer* server)
{
  if (error)
    return;

  std::cout << "Finalizando o servidor..." << std::endl;
  server->stop();
}//]

int main(int argc, char *argv[])
{
  //[Config
  Config config;
  int error = config.parse(argc, argv, std::cerr);
  if (error)
    return error;
  //]

  //[DefaultDataProvider
  DefaultDataProvider provider;
  //]

  //[FTCServer
  FTCServer server(provider, config);
  //]
  
  // registra para capturar o Ctrl+C ou outro sinal de termino
  //[int_signal
  boost::asio::io_service io_service;
  boost::asio::signal_set stop_signals(io_service, SIGINT, SIGTERM);
#if defined(SIGQUIT)
  stop_signals.add(SIGQUIT);
#endif // defined(SIGQUIT)
  stop_signals.async_wait(boost::bind(stop_handler, _1, _2, &server));
  //]

  //[dispatch
  boost::shared_ptr<boost::thread> thread(
    new boost::thread(boost::bind(&FTCServer::dispatch, &server)));
  //]
  //[wait_ready
  server.wait_ready(); // aguarda o servidor comecar a ouvir o hostname:porta
  //]

  //[exec_add_files_to_server
  add_files_to_server(server, config.files());
  //]
  std::cout << "Servidor rodando em: " << server.config().hostname() 
       << ":" << server.config().port() << std::endl;
  std::cout << "Digite Ctrl + C para terminar" << std::endl;  
  
  // aguarda o Ctrl+C
  //[int_signal_run
  io_service.run();
  //]

  // aguarda o thread do servidor finalizar
  //[thread_join
  if (thread->joinable())
    thread->join();
  //]

  std::cout << "Servidor finalizado." << std::endl;
  return 0;
}
