#include "pch.h"
#include "Config.h"

#include <boost/cstdint.hpp>
#include <boost/program_options.hpp>
#include <boost/algorithm/string.hpp>

Config::Config()
{
}

std::istream& operator>>(std::istream& in, ServerConfig::ServerLogLevel& log_level)
{
  std::string token;
  in >> token;
  if (boost::iequals(token, "OFF"))        log_level = ServerConfig::LOG_OFF;
  else if (boost::iequals(token, "FATAL")) log_level = ServerConfig::LOG_FATAL;
  else if (boost::iequals(token, "ERROR")) log_level = ServerConfig::LOG_ERROR;
  else if (boost::iequals(token, "WARN"))  log_level = ServerConfig::LOG_WARN;
  else if (boost::iequals(token, "INFO"))  log_level = ServerConfig::LOG_INFO;
  else if (boost::iequals(token, "DEBUG")) log_level = ServerConfig::LOG_DEBUG;
  else if (boost::iequals(token, "TRACE")) log_level = ServerConfig::LOG_TRACE;
  else log_level = ServerConfig::LOG_ALL;
  return in;
}

int Config::parse(int argc, char *argv[], std::ostream& out)
{
  const std::size_t max_chunk_size = 1024 * 1024 * 10; // 500MiB

  boost::program_options::options_description desc("Usage: " + std::string(argv[0]));
  desc.add_options()    
    ("help"                   , "Exibe esta mensagem")
    ("hostname,h"             , boost::program_options::value<std::string>(&m_hostname)->default_value("localhost"), "Endereco/IP do servidor")
    ("port,p"                 , boost::program_options::value<uint16_t>(&m_port)->default_value(25555), "Nmero da porta do servidor")
    ("buffer_size,b"          , boost::program_options::value<size_t>(&m_client_buffer_size)->default_value(1024 * 1024), "Tamanho do buffer em bytes)")
    ("chunk_size,c"           , boost::program_options::value<size_t>(&m_chunk_size)->default_value(1024 * 1024), "Tamanho do chunk em bytes")
    ("request_timeout,r"      , boost::program_options::value<uint32_t>(&m_channel_request_timeout)->default_value(10 * 60 * 1000), "Timeout para pedido de canal em milisegundos")
    ("client_idle_timeout,i"  , boost::program_options::value<uint32_t>(&m_client_timeout)->default_value(10 * 60 * 1000), "Timeout para conexao com cliente ocioso em milisegundos")
    ("log_level,l"            , boost::program_options::value<ServerLogLevel>(&m_log_level)->default_value(ServerConfig::LOG_ALL, "ALL"), "Nivel de geracao de log (OFF, FATAL, ERROR, WARN, INFO, DEBUG, TRACE, ALL)")
    ("log_filename,f"         , boost::program_options::value<std::string>(&m_log_filename)->default_value("ftcserver.log"), "Nome do arquivo de log")
    ("log_date_time,d"        , boost::program_options::value<bool>(&m_log_date_time)->default_value(true), "Inclui no log a data e hora")
    ("max_channel_requests,m" , boost::program_options::value<size_t>(&m_max_channel_requests)->default_value(2048), "No. maximo de pedido de canais")
    ("max_clients,s"          , boost::program_options::value<size_t>(&m_max_clients)->default_value(150), "No. maximo de clientes simultaneos")
    ("thread_pool_size"       , boost::program_options::value<size_t>(&m_thread_pool_size)->default_value(10), "No. de threads para o pool de threads")
    ("test_mode,t"            , boost::program_options::value<bool>(&m_test_mode)->default_value(true), "Modo de teste (quando em teste, mantem o canal quando consumido por um cliente)")
    ("files,a"                , boost::program_options::value<std::vector<std::string> >(&m_files), "Arquivos para serem servidos pelo servidor")
  ;

  boost::program_options::positional_options_description positional;
  positional.add("files", -1);

  boost::program_options::variables_map vm;
  boost::program_options::store(boost::program_options::command_line_parser(argc, argv).options(desc).positional(positional).run(), vm, false);
  boost::program_options::notify(vm);

  if (vm.count("help"))
  {
    out << desc << std::endl;
    return 1;
  }

  if (m_hostname.empty())
  {
    std::cout << "Endereco deve ser fornecido para o servidor" << std::endl;
    return 2;
  }

  if (m_files.empty())
  {
    std::cout << "Pelo menos um arquivo deve ser fornecido." << std::endl;
    return 3;
  }

  if (m_chunk_size > max_chunk_size) 
    m_chunk_size = max_chunk_size;

  return 0;
}

const std::vector<std::string>& Config::files() const
{
  return m_files;
}

const std::string& Config::hostname() const
{
  return m_hostname;
}

void Config::hostname(std::string const& value)
{
  m_hostname = value;
}

uint16_t Config::port() const
{
  return m_port;
}

void Config::port(uint16_t value)
{
  m_port = value;
}

uint32_t Config::channel_request_timeout() const
{
  return m_channel_request_timeout;
}

void Config::channel_request_timeout(uint32_t value)
{
  m_channel_request_timeout = value;
}

uint32_t Config::client_timeout() const
{
  return m_client_timeout;
}

void Config::client_timeout(uint32_t value)
{
  m_client_timeout = value;
}

std::size_t Config::max_clients() const
{
  return m_max_clients;
}

void Config::max_clients(std::size_t value)
{
  m_max_clients = value;
}

std::size_t Config::max_channel_requests() const
{
  return m_max_channel_requests;
}

void Config::max_channel_requests(std::size_t value)
{
  m_max_channel_requests = value;
}

std::size_t Config::client_buffer_size() const
{
  return m_client_buffer_size;
}

void Config::client_buffer_size(std::size_t value)
{
  m_client_buffer_size = value;
}

std::size_t Config::chunk_size() const
{
  return m_chunk_size;
}

void Config::chunk_size(std::size_t value)
{
  m_chunk_size = value;
}

bool Config::is_test_mode() const
{
  return m_test_mode;
}

void Config::is_test_mode(bool value)
{
  m_test_mode = value;
}

void Config::log_level(Config::ServerLogLevel value)
{
  m_log_level = value;
}

Config::ServerLogLevel Config::log_level() const
{
  return m_log_level;
}

void Config::log_filename(const std::string& value)
{
  m_log_filename = value;
}

const std::string& Config::log_filename() const
{
  return m_log_filename;
}

void Config::log_date_time(bool value)
{
  m_log_date_time = value;
}

bool Config::log_date_time() const
{
  return m_log_date_time;
}

std::size_t Config::thread_pool_size() const
{
  return m_thread_pool_size;
}

void Config::thread_pool_size(const std::size_t value)
{
  m_thread_pool_size = value;
}
