/*
** OpenBus Demo - Orbix 6.3
** server.cpp
*/

#include <fstream>
#include <iostream>

#include <CORBA.h>
#include <csignal>
#define ORB_SIGTYPE int

#include <openbus/OpenBusContext.hpp>
#include <openbus/ORBInitializer.hpp>
#include <scs/ComponentContext.h>
#include <UnstructuredDataViewImpl.h>
#include <HierarchicalNodeDataViewImpl.h>

#include "DataServerContext.h"
#include "DataServiceImpl.h"
#include "ProjectNavigationImpl.h"
#include "HierarchicalManagementImpl.h"
#include "HierarchicalTransferImpl.h"
#include "FileDataViewImpl.h"
#include "ILogFileViewImpl.h"
#include "ProjectDataViewImpl.h"

const char * root_path = "/tmp/data_service";

namespace offer_registry = tecgraf::openbus::core::v2_0::services::offer_registry;
namespace services = tecgraf::openbus::core::v2_0::services;
namespace access_control = tecgraf::openbus::core::v2_0::services::access_control;

namespace file_system = tecgraf::openbus::data_service::core::v1_02::demo::file_system;
namespace api = tecgraf::openbus::data_service::core::v1_02;
namespace hierarchical_api = tecgraf::openbus::data_service::hierarchical::v1_02;

std::auto_ptr <openbus::Connection> conn;
file_system::DataServerContext * componentContext;
offer_registry::ServiceOffer_var registered_offer;

static void registerTypes(CORBA::ORB_var orb){

  // CORBA::ValueFactoryBase_var vf1 = new file_system::FileDataViewFactory();
  // orb->register_value_factory( file_system::_tc_FileDataView->id(), vf1.in());

  CORBA::ValueFactoryBase_var vf2 = new api::UnstructuredDataViewFactory();
  orb->register_value_factory( api::_tc_UnstructuredDataView->id(), vf2.in());
  
  CORBA::ValueFactoryBase_var vf3 = new hierarchical_api::HierarchicalNodeDataViewFactory();
  orb->register_value_factory( hierarchical_api::_tc_HierarchicalNodeDataView->id(), vf3.in());

  CORBA::ValueFactoryBase_var vf4 = new file_system::LogFileValueTypeDataViewFactory;
  orb->register_value_factory( file_system::_tc_LogFileValueTypeDataView->id(), vf4.in());

  namespace project = tecgraf::openbus::data_service::project::v1_02;

  CORBA::ValueFactoryBase_var vf5 = new project::ProjectDataViewFactory;
  orb->register_value_factory( project::_tc_ProjectDataView->id(), vf5.in());
}

#ifdef OPENBUS_SDK_MULTITHREAD
void run_orb(CORBA::ORB_var orb)
{
  orb->run();
}
#endif

struct onReloginCallback
{
  typedef void result_type;
  result_type operator()(openbus::Connection& c, access_control::LoginInfo info) const
  {
    do
    {
      try
      {
        c.loginByPassword("file_system_cpp", "file_system_cpp");
        break;
      }
      catch(tecgraf::openbus::core::v2_0::services::access_control::AccessDenied const& e)
      {
        std::cout << "Falha ao tentar realizar o login por senha no barramento: "
          "a entidade j est com o login realizado. Esta falha ser ignorada." << std::endl;
      }
      catch (services::ServiceFailure e)
      {
        std::cout << "Falha no servio remoto. Causa: " << std::endl;
      }
      catch (CORBA::TRANSIENT const&)
      {
        std::cout << "Erro de comunicacao. Verifique se o sistema se encontra "
          "ainda disponivel ou se sua conexao com o mesmo foi interrompida" << std::endl;
      }
      catch (CORBA::COMM_FAILURE const&)
      {
        std::cout << "Erro de comunicacao. Verifique se o sistema se encontra "
          "ainda disponivel ou se sua conexao com o mesmo foi interrompida" << std::endl;
      }
      catch (CORBA::OBJECT_NOT_EXIST const&)
      {
        std::cout << "Objeto remoto nao existe mais. Verifique se o sistema se encontra disponivel" << std::endl;
      }
      unsigned int t = 30u;
      do { t = sleep(t); } while(t);
    }
    while(true);
  }
};

static void myTerminationHandler(ORB_SIGTYPE signal) {
  try
  {
      std::cout << "Removendo o registro do componente..." << std::endl;
      
      registered_offer->remove();

      conn->logout();
      
      delete componentContext;
      std::cout << "Encerrando o processo servidor..." << std::endl;
  }
  catch (services::UnauthorizedOperation e)
  {
    std::cout << "Falha remover registro do servico de ofertas. Causa: " << std::endl;
  }
  catch (services::ServiceFailure e)
  {
    std::cout << "Falha no servio remoto. Causa: " << std::endl;
  }
  catch (CORBA::TRANSIENT const&)
  {
    std::cout << "Erro de comunicacao. Verifique se o sistema se encontra "
      "ainda disponivel ou se sua conexao com o mesmo foi interrompida" << std::endl;
  }
  catch (CORBA::COMM_FAILURE const&)
  {
    std::cout << "Erro de comunicacao. Verifique se o sistema se encontra "
      "ainda disponivel ou se sua conexao com o mesmo foi interrompida" << std::endl;
  }
  catch (CORBA::OBJECT_NOT_EXIST const&)
  {
    std::cout << "Objeto remoto nao existe mais. Verifique se o sistema se encontra disponivel" << std::endl;
  }

} 

int main(int argc, char** argv)
{
  signal(SIGINT,myTerminationHandler);

  // Inicializando CORBA e ativando o RootPOA
  CORBA::ORB_var orb = openbus::ORBInitializer(argc, argv);
  CORBA::Object_var o = orb->resolve_initial_references("RootPOA");
  PortableServer::POA_var poa = PortableServer::POA::_narrow(o);
  assert(!CORBA::is_nil(poa));
  PortableServer::POAManager_var poa_manager = poa->the_POAManager();
  poa_manager->activate();

  // Registrando as fabricas de valuetypes
  try {
    registerTypes(orb);
  } catch (CORBA::Exception & e) {
      std::cout << "** Nao foi possivel registrar as fabricas de valuetypes. **" << std::endl ;
    exit(-1);
  }

#ifdef OPENBUS_SDK_MULTITHREAD
  boost::thread orb_thread(boost::bind(&run_orb, orb));
#endif

  // Construindo e logando conexao
  openbus::OpenBusContext* manager = dynamic_cast<openbus::OpenBusContext*>
    (orb->resolve_initial_references("OpenBusContext"));
  assert(manager != 0);
  do
  {
    try
    {
      conn = manager->createConnection("localhost", 2089);
      conn->onInvalidLogin( ::onReloginCallback());
      conn->loginByPassword("file_system_cpp", "file_system_cpp");
      manager->setDefaultConnection(conn.get());
      break;
    }
    catch(tecgraf::openbus::core::v2_0::services::access_control::AccessDenied const& e)
    {
      std::cout << "Falha ao tentar realizar o login por senha no barramento: "
        "a entidade j est com o login realizado. Esta falha ser ignorada." << std::endl;
    }
    catch (services::ServiceFailure e)
    {
      std::cout << "Falha no servio remoto. Causa: " << std::endl;
    }
    catch (CORBA::TRANSIENT const&)
    {
      std::cout << "Erro de comunicacao. Verifique se o sistema se encontra "
        "ainda disponivel ou se sua conexao com o mesmo foi interrompida" << std::endl;
    }
    catch (CORBA::COMM_FAILURE const&)
    {
      std::cout << "Erro de comunicacao. Verifique se o sistema se encontra "
        "ainda disponivel ou se sua conexao com o mesmo foi interrompida" << std::endl;
    }
    catch (CORBA::OBJECT_NOT_EXIST const&)
    {
      std::cout << "Objeto remoto nao existe mais. Verifique se o sistema se encontra disponivel" << std::endl;
    }
    unsigned int t = 30u;
    do { t = sleep(t); } while(t);
  }
  while(true);

  scs::core::ComponentId componentId;
  componentId.name = "file_system";
  componentId.major_version = '1';
  componentId.minor_version = '2';
  componentId.patch_version = '0';
  componentId.platform_spec = "";
  
  componentContext 
      =  new file_system::DataServerContext(manager->orb(), componentId, root_path, "EntityName", "filesystem");

/* Descrio dos componentes */
  componentContext->addFacet("data_service",api::_tc_IDataService->id()
                             , new file_system::DataServiceImpl(componentContext));
  componentContext->addFacet("navigation", hierarchical_api::_tc_IHierarchicalNavigationDataService->id()
                             , new file_system::ProjectNavigationImpl(componentContext));
  componentContext->addFacet("management", hierarchical_api::_tc_IHierarchicalManagementDataService->id()
                             , new file_system::HierarchicalManagementImpl(componentContext));
  componentContext->addFacet("transfer", hierarchical_api::_tc_IHierarchicalTransferDataService->id()
                             , new file_system::HierarchicalTransferImpl(componentContext));

  offer_registry::ServicePropertySeq properties;
  properties.length(1);
  properties[0].name = "offer.domain";
  properties[0].value = "Demos";

  try
  {
    do
    {
      try
      {
        registered_offer = manager->getOfferRegistry()->registerService(componentContext->getIComponent(), properties);
        break;
      }
      catch(tecgraf::openbus::core::v2_0::services::access_control::AccessDenied const& e)
      {
        std::cout << "Falha ao tentar realizar o login por senha no barramento: "
          "a entidade j est com o login realizado. Esta falha ser ignorada." << std::endl;
      }
      catch (services::ServiceFailure const& e)
      {
        std::cout << "Falha no servico remoto. Causa: " << std::endl;
      }
      catch (offer_registry::UnauthorizedFacets const& e)
      {
        std::cout << "Faceta nao autorizada no barramento: " << std::endl;
      }
      catch (CORBA::TRANSIENT const&)
      {
        std::cout << "Erro de comunicacao. Verifique se o sistema se encontra "
          "ainda disponivel ou se sua conexao com o mesmo foi interrompida" << std::endl;
      }
      catch (CORBA::COMM_FAILURE const&)
      {
        std::cout << "Erro de comunicacao. Verifique se o sistema se encontra "
          "ainda disponivel ou se sua conexao com o mesmo foi interrompida" << std::endl;
      }
      catch (CORBA::OBJECT_NOT_EXIST const&)
      {
        std::cout << "Objeto remoto nao existe mais. Verifique se o sistema se encontra disponivel" << std::endl;
      }
      unsigned int t = 30u;
      do { t = sleep(t); } while(t);
    }
    while(true);
  
  std::cout << "\n\nDataService registrado." << std::endl;
  std::cout << "Aguardando requisies..." << std::endl;
  
#ifdef OPENBUS_SDK_MULTITHREAD
    orb_thread.join();
#else
    orb->run();
#endif
  }
  catch(offer_registry::InvalidService const&)
  {
    std::cout << "Barramento diz que o servico ofertado nao e um componente valido" << std::endl;
  }
  catch(offer_registry::InvalidProperties const&)
  {
    std::cout << "Barramento diz que o servico ofertado possui ofertas" << std::endl;
  }
}
