
#include <iostream>
#include <ftc/RemoteDataChannel.h>
#include <openssl/md5.h>

#include <stubs/hierarchical_data_service.h>
#include <stubs/project.h>
#include "stubs/file_system.h"

#include <DataService.h>

#include "DataServiceTester.h"
#include "FileDataViewImpl.h"

namespace file_system = tecgraf::openbus::data_service::core::v1_02::demo::file_system;
namespace api = tecgraf::openbus::data_service::core::v1_02;
namespace hierarchical_api = tecgraf::openbus::data_service::hierarchical::v1_02;
namespace project_api = tecgraf::openbus::data_service::project::v1_02;
namespace ftc = tecgraf::ftc::v1_04_01;

/** 
 * @brief Construtor 
 * 
 * @param dataservice
 */
DataServiceTester::DataServiceTester( scs::core::IComponent * component )
{
    _icomponent = component;
}

/** 
 * @brief 
 */
DataServiceTester::~DataServiceTester()
{
}

/** 
 * @brief Executa os testes
 */
bool DataServiceTester::run()
{
    if(_icomponent == NULL)
        return false;

    CORBA::Object* obj = _icomponent->getFacetByName("data_service");
    _dataService = api::IDataService::_narrow(obj);

    obj = _icomponent->getFacetByName("management");
    _management = hierarchical_api::IHierarchicalManagementDataService::_narrow(obj);

    obj = _icomponent->getFacetByName("navigation");
    assert(!CORBA::is_nil(obj));
    _navigation = project_api::IProjectNavigationDataService::_narrow(obj);
    assert(!CORBA::is_nil(_navigation));

    obj = _icomponent->getFacetByName("transfer");
    _transfer   = hierarchical_api::IHierarchicalTransferDataService::_narrow(obj);

    // Metodos de teste
    printTree();
    if(!createTestData())
        return false;
    printTree();
    if(!readDataDescriptions())
        return false;
    if(!testProject())
        return false;
    if(!testILogView())
        return false;
    if(!testDeactivateLogInterface())
        return false;
    if(!destroyTestData())
        return false;
    printTree();
    return true;
}

struct is_project_item_view
{
  bool operator()(tecgraf::openbus::data_service::core::v1_02::DataDescription* description) const
  {
    namespace project = tecgraf::openbus::data_service::project::v1_02;
    return !std::strcmp(description->fDefaultView.fInterfaceName.in()
                        , project::_tc_ProjectItemDataView->id());
  }
};

template <typename SeqVar, typename V>
struct corba_sequence_iterator : boost::iterator_facade
  <corba_sequence_iterator<SeqVar, V>, std::forward_iterator_tag, V>
{
  typedef corba_sequence_iterator<SeqVar, V> self_type;
  corba_sequence_iterator() : index(0u) {}
  corba_sequence_iterator(SeqVar seq_var, std::size_t index = 0)
    : seq(seq_var), index(index) {}

  V& dereference() const { return seq[index]; }
  void increment() { ++index; }
  bool equal(self_type const& other) const { return index == other.index; }

  SeqVar seq;
  std::size_t index;
};

bool DataServiceTester::testProject()
{
  std::cout << "[testProject]" << std::endl;
  bool retValue;
  
  try
  {
    namespace project = tecgraf::openbus::data_service::project::v1_02;
    project::ProjectDataViewSeq_var projects = _navigation->getProject("owner");

    assert(projects->length() == 2);

    namespace core = tecgraf::openbus::data_service::core::v1_02;
    core::DataDescriptionSeq_var description1 = _navigation->getChildren
      (projects[0]->fKey());

    

    retValue = true;
  }
  catch(CORBA::SystemException const& e)
  {
    std::cout << "Corba Error: " << e << std::endl;
    retValue = false;
  }
  catch (api::InvalidDataKey const& e) {
    std::cout << "Invalid Data Key: " << e << std::endl;
    retValue = false;
  }
  catch (api::ServiceFailure const& e) {
    std::cout << "Service Failure: " << e << std::endl;
    retValue = false;
  }
  catch (api::UnsupportedView const& e) {
    std::cout << "UnsupportedView: " << e << std::endl;
    retValue = false;
  }
  return retValue;
}

/** 
 * @brief Imprime a arvore no console 
 */
void DataServiceTester::printTree()
{
    std::cout << "[printTree]" << std::endl;
    try
    {
        // Acessando a raiz da arvore de diretorios
        api::DataDescriptionSeq_var roots = _navigation->getRoots();
        
        // Imprimindo a arvore recursivamente
        std::ostringstream treeText; 
        api::DataDescription * rootDesc;
        for(unsigned int i = 0; i < roots->length() ; i++)
        {
            rootDesc = &(roots[i]);
            printTreeAux(rootDesc,treeText,0);
        }
        std::cout << treeText.str() << std::endl;
    }
    catch(CORBA::SystemException & e )
    {
        std::cout << "Corba Error: " << e << std::endl;
    }
    catch(api::ServiceFailure & e )
    {
        std::cout << "Service Failure: " << e << std::endl;
    }
}

/** 
 * @brief Metodo de impressão auxiliar 
 */
void DataServiceTester::printTreeAux(api::DataDescription * file 
                                     , std::ostringstream & treeText
                                     , unsigned char indent)
{
    // Adiciona o nome a texto de saida
    std::string indentStr( indent, ' ' );
    treeText << indentStr << file->fName << std::endl; 

    api::DefaultView* defView = &(file->fDefaultView);
    if(defView == NULL)
    {
        std::cout << "DataDescription->fDefaultView == NULL " << std::endl;
        return;
    }
    if(defView->fValue == NULL)
    {
        std::cout << "DataDescription->fDefaultView->fValue == NULL " << std::endl;
        return;
    }
    hierarchical_api::HierarchicalNodeDataView * hNode = 
        hierarchical_api::HierarchicalNodeDataView::_downcast(defView->fValue->_to_value());

    // Se for um diretorio devemos percorrer seus filhos
    if(hNode->fIsContainer())
    {
       api::DataDescriptionSeq_var children = _navigation->getChildren(file->fKey);
       for(unsigned int i = 0 ; i < children->length(); i++ )
       {
           api::DataDescription * data = &(children[i]);
           // Chamada recursiva
           printTreeAux(data,treeText,indent+4);
       }
    }
}

/** 
 * @brief Metodo auxiliar que imprime os dados de um FileDataView
 * 
 * @param file
 */
void DataServiceTester::printFileData(api::DataDescription * file)
{
    if (file == NULL) {
      std::cout << " <-- DataDescription está nulo -->" << std::endl;
      return;
    }                   

    api::DefaultView * defView = &(file->fDefaultView);
    if(defView == NULL)
    {
        std::cout << "DataDescription->fDefaultView == NULL " << std::endl;
        return;
    }
    if(defView->fValue == NULL)
    {
        std::cout << "DataDescription->fDefaultView->fValue == NULL " << std::endl;
        return;
    }
    hierarchical_api::HierarchicalNodeDataView* hNode = 
        hierarchical_api::HierarchicalNodeDataView::_downcast(defView->fValue->_to_value());

    std::cout << "Elemento "   << file->fName << std::endl ;
    //std::cout << "  Key = "    << file->fKey << std::endl ;
    std::cout << "  Folder = " << hNode->fIsContainer() << std::endl  ;
    //std::cout << "  Size = "   << file->fSize() << " bytes" << std::endl ;
    std::cout << "  Views {"   << std::endl ;
    for(unsigned int i = 0 ; i < file->fOthersViews.length(); i++ )
    {
      std::cout << "         " << file->fOthersViews[i] << std::endl;
    }
    std::cout << "  }" << std::endl ;

}

/** 
 * @brief 
 * 
 * @param parentKey
 * @param name
 * @param container
 * 
 * @return 
 */
api::DataKey* DataServiceTester::createData(api::DataKey & parentKey, const char* name, bool container)
{
    std::cout << "CreateData: " << name << std::endl;
    api::StringSeq_var views = new api::StringSeq(1);
    views->length(1);
    views[0] = api::_tc_UnstructuredDataView->id() ;
    const api::MetadataSeq metadata(0);
    const api::DataKey dummyKey;
    
    hierarchical_api::HierarchicalNodeDataView* 
      hNode = new hierarchical_api::HierarchicalNodeDataViewImpl(dummyKey,parentKey,container);

    struct api::DefaultView defaultView;
    defaultView.fInterfaceName = hierarchical_api::_tc_HierarchicalNodeDataView->id();
    CORBA::AbstractBase* ab = hNode;
    defaultView.fValue = api::DataView::_narrow(ab);
    
    api::DataDescription_var desc = new api::DataDescription ;
    desc->fKey = dummyKey;
    desc->fName = name;
    desc->fOthersViews = views;
    desc->fMetadata = metadata;
    desc->fDefaultView = defaultView; 
    
    // Create newData
    return _management->createData( desc, parentKey );
}

/** 
 * @brief 
 */
bool DataServiceTester::createTestData() {
    std::cout << "[createTestData BEGIN]" << std::endl;
    
    bool retValue = false;

    try{
       api::DataDescriptionSeq_var roots = _navigation->getRoots();

       api::DataDescription *  rootDesc = &(roots[0]);

       // Testando criacao
       api::DataKey_var newDatakey = createData( rootDesc->fKey, "test.txt", false);
       
       // Imprimindo o FileDataView do arquivo criado
       api::DataDescription_var newDataDesc = _dataService->getDataDescription(newDatakey) ;
       printFileData(newDataDesc);
       
       // Testando remoção do arquivo criado
       _management->deleteData(newDatakey);
       
       
       newDatakey = createData( rootDesc->fKey , "Project1"    , true  );
       newDatakey = createData( newDatakey      , "arquivo.txt" , false );
       newDatakey = createData( rootDesc->fKey , "Project2"    , true  );
       newDatakey = createData( newDatakey      , "arquivo.log" , false );
       
       api::DataDescriptionSeq_var children = _navigation->getChildren(rootDesc->fKey);
       if (children->length() >= 2)
         retValue = true;
       
    }
    catch(CORBA::SystemException & e )
    {
        std::cout << "Corba Error: " << e << std::endl;
        retValue = false;
    }
    catch(api::ServiceFailure & e )
    {
        std::cout << "Service Failure: " << e << std::endl;
        retValue = false;
    }
   
    std::cout << "[createTestData END]" << std::endl;
    return retValue;
}

/** 
 * @brief 
 */
bool DataServiceTester::readDataDescriptions()
{
    std::cout << "[readDataDescriptions]" << std::endl;
    bool retValue = true;
    try {
        api::DataDescriptionSeq_var rootDescList = _navigation->getRoots();
        if (rootDescList->length() < 1)
          return false;
       
        api::DataDescription*  rootDesc = &(rootDescList[0]);
       
        api::DataDescriptionSeq_var children = _navigation->getChildren(rootDesc->fKey);
        if (children->length() < 2)
          return false;
        api::DataDescription * projectDesc  = &(children[0]);
        api::DataDescription * projectDesc2 = &(children[1]);
                                                                        
        api::DataDescriptionSeq_var projChildren = _navigation->getChildren(projectDesc->fKey);       
        if (projChildren->length() < 1)
          return false;
        api::DataDescription * fileDesc = &(projChildren[0]);

        api::DataDescriptionSeq_var projChildren2 = _navigation->getChildren(projectDesc2->fKey);       
        if (projChildren2->length() < 1)
          return false;
        api::DataDescription * fileDesc2 = &(projChildren2[0]);

        printFileData(rootDesc);
        printFileData(projectDesc);
        printFileData(fileDesc);
        printFileData(projectDesc2);
        printFileData(fileDesc2);
    }
    catch(CORBA::SystemException const& e )
    {
        std::cout << "Corba Error: " << e << std::endl;
        retValue = false;
    }
    catch(api::InvalidDataKey const& e)
    {
        std::cout << "Invalid Key: " << e << std::endl;
        retValue = false;
    }
    catch(api::ServiceFailure const& e)
    {
        std::cout << "Service Failure: " << e << std::endl;
        retValue = false;
    }

    return retValue;
}

/** 
 * @brief 
 */
bool DataServiceTester::testILogView()
{
    std::cout << "[testILogView]" << std::endl;
    bool retValue = true;
    try
    {
        api::DataDescriptionSeq_var rootDescList = _navigation->getRoots();
        if (rootDescList->length() < 1)
          return false;
        api::DataDescription* rootDesc = &(rootDescList[0]);
   
        api::DataDescriptionSeq_var children = _navigation->getChildren(rootDesc->fKey);
        if (children->length() < 2)
          return false;
        api::DataDescription* projectDesc = &(children[0]);
   
        api::DataDescriptionSeq_var children2 = _navigation->getChildren(projectDesc->fKey);
        if (children2->length() < 1)
          return false;
        api::DataDescription* logFileDesc = &(children2[0]);
        
        api::DataView_ptr dataView =
          _dataService->getDataView( logFileDesc->fKey, file_system::_tc_LogFileValueTypeDataView->id() );
        assert(!CORBA::is_nil(dataView));
        file_system::LogFileValueTypeDataView_var
          logFileView = file_system::LogFileValueTypeDataView::_downcast(dataView->_to_value());

        assert(!CORBA::is_nil(logFileView));
        assert(!CORBA::is_nil(logFileView->logFile()));
        std::string line = logFileView->logFile()->getLastLine();
        logFileView->logFile()->deactivate();
       
        printFileData(projectDesc);
        printFileData(logFileDesc);

        std::cout << logFileDesc->fName << " " << line << std::endl;

        {
          api::DataView* dataView =
            _dataService->getDataView( logFileDesc->fKey, api::_tc_UnstructuredDataView->id() );
          api::UnstructuredDataView*
            unstructured = api::UnstructuredDataView::_downcast(dataView->_to_value());

          std::cout << "UnstructuredDataView::Host " << unstructured->fHost() << std::endl;
          std::cout << "UnstructuredDataView::Port " << unstructured->fPort() << std::endl;
          std::cout << "UnstructuredDataView::Writable " << unstructured->fWritable() << std::endl;

          std::vector<uint8_t> keystr(MD5_DIGEST_LENGTH);
          api::OctetSeq accessKey = unstructured->fAccessKey ();
          assert(MD5_DIGEST_LENGTH == accessKey.length ());
          for(int i = 0; i != MD5_DIGEST_LENGTH; ++i)
            keystr[i] = accessKey[i];

          ftc::RemoteDataChannel filechannel(false, unstructured->fHost (), unstructured->fPort (), keystr);

          filechannel.open();
          
          std::cout << "File channel opened to unstructuredDataView" << std::endl;
        }
    }
    catch(CORBA::SystemException const& e)
    {
        std::cout << "Corba Error: " << e << std::endl;
        retValue = false;
    }
    catch (api::InvalidDataKey const& e) {
        std::cout << "Invalid Data Key: " << e << std::endl;
        retValue = false;
    }
    catch (api::ServiceFailure const& e) {
        std::cout << "Service Failure: " << e << std::endl;
        retValue = false;
    }
    catch (api::UnsupportedView const& e) {
        std::cout << "UnsupportedView: " << e << std::endl;
        retValue = false;
    }
    return retValue;
}

/** 
 * @brief 
 */
bool DataServiceTester::testDeactivateLogInterface()
{
    std::cout << "[testDeactivateLogInterface]" << std::endl;
    bool retValue = false;
    try {
        api::DataDescriptionSeq_var rootDescList = _navigation->getRoots();
        if (rootDescList->length() < 1)
          return false;
        
        api::DataDescription * rootDesc = &(rootDescList[0]);
        
        api::DataDescriptionSeq_var children = _navigation->getChildren(rootDesc->fKey);
        if (children->length() < 2)
          return false;
        api::DataDescription * projectDesc = &(children[0]);
        
        api::DataDescriptionSeq_var children2 = _navigation->getChildren(projectDesc->fKey);
        if (children2->length() < 1)
          return false;
        api::DataDescription * logFileDesc = &(children2[0]);
        api::DataView * dataView =
          _dataService->getDataView(logFileDesc->fKey, file_system::_tc_LogFileValueTypeDataView->id());
        file_system::LogFileValueTypeDataView_var 
          logFileView = file_system::LogFileValueTypeDataView::_downcast(dataView->_to_value());

        std::cout << logFileView->logFile()->getLastLine() << std::endl;
        logFileView->logFile()->deactivate();

        retValue = true;
        // try 
        // {
        //     while(true)
        //         logFileView->getLastLine();
        // }
        // catch (CORBA::OBJECT_NOT_EXIST & e) 
        // {
        //     std::cout << e << std::endl;
        //     retValue = true;
        // }
    }
    catch(CORBA::SystemException const& e)
    {
        std::cout << "Corba Error: " << e << std::endl;
    }
    catch (api::InvalidDataKey const& e) {
        std::cout << "Invalid Data Key: " << e << std::endl;
    }
    catch (api::ServiceFailure const& e) {
        std::cout << "Service Failure: " << e << std::endl;
    }
    catch (api::UnsupportedView const& e) {
        std::cout << "UnsupportedView: " << e << std::endl;
    }

    return retValue;
}


/** 
 * @brief Remove todo o conteudo do diretorio de teste
 */
bool DataServiceTester::destroyTestData()
{
    std::cout << "[destroyTestData]" << std::endl;
    bool retValue = true;
    api::DataDescriptionSeq_var rootDescList = _navigation->getRoots();
    api::DataDescription * rootDesc = &(rootDescList[0]);
    api::DataDescriptionSeq_var children = _navigation->getChildren(rootDesc->fKey);
    for(unsigned int i = 0 ; i < children->length(); i++ )
    {
        try 
        {
            // Delete newData
            _management->deleteData(children[i].fKey);
        }
        catch(CORBA::SystemException const& e)
        {
            std::cout << "Corba Error: " << e << std::endl;
            retValue = false;
        }
        catch (api::InvalidDataKey const& e) {
            std::cout << "Invalid Data Key: " << e << std::endl;
            retValue = false;
        }
        catch (api::ServiceFailure const& e) {
            std::cout << "Service Failure: " << e << std::endl;
            retValue = false;
        }
    }
    return retValue;
}



