#include "pch.h"
#include "Config.h"

#include <boost/cstdint.hpp>
#include <boost/program_options.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/version.hpp>

#if (BOOST_VERSION == 105100)
namespace boost { namespace algorithm { namespace detail {
inline unsigned hex_char_to_int ( char c );
}}}
#endif
#include <boost/algorithm/hex.hpp>

Config::Config()
{
}

std::istream& operator>>(std::istream& in, Config::Operation& op)
{
  std::string token;
  in >> token;
  if (boost::iequals(token, "READ"))       op = Config::OP_READ;
  else if (boost::iequals(token, "WRITE")) op = Config::OP_WRITE;
  else throw boost::program_options::validation_error(boost::program_options::validation_error::invalid_option_value, "operation", token);
  return in;
}

int Config::parse(int argc, char *argv[], std::ostream& out)
{
  std::string key;
  boost::program_options::options_description desc("Usage: " + std::string(argv[0]));
  desc.add_options()    
    ("help"            , "Exibe esta mensagem")
    ("hostname,h"      , boost::program_options::value<std::string>(&m_hostname)->default_value("localhost"), "Endereco/IP do servidor")
    ("port,p"          , boost::program_options::value<uint16_t>(&m_port)->default_value(25555), "Nmero da porta do servidor")
    ("buffer_size,b"   , boost::program_options::value<size_t>(&m_buffer_size)->default_value(1024 * 1024), "Tamanho do buffer em bytes)")
    ("operation,o"     , boost::program_options::value<Config::Operation>(&m_operation), "Operacao de READ ou WRITE")
    ("filename,f"      , boost::program_options::value<std::string>(&m_filename), "Nome do arquivo local")
    ("key,k"           , boost::program_options::value<std::string>(&key), "Chave do arquivo remoto em hexadecimal")
    ("size,s"          , boost::program_options::value<int64_t>(&m_size)->default_value(-1), "Quantidade em bytes para transferir (-1 usa o tamanho do arquivo de origem)")
    ("close,c"         , boost::program_options::value<bool>(&m_close)->default_value(true), "Fecha a conexao ao final de uma transferncia")
  ;

  boost::program_options::variables_map vm;
  try
  {
    boost::program_options::store(boost::program_options::command_line_parser(argc, argv).options(desc)/*.positional(positional)*/.run(), vm, false);
  }
  catch (boost::program_options::validation_error e)
  {
    if (e.get_option_name() == "--operation")
    {
      out << "Operacao informada e invalida: " << e.what() << std::endl;
      return 5;
    }
    out << "Erro na validacao dos argumentos passados: " << e.what() << std::endl;
    return 9;
  }
  boost::program_options::notify(vm);

  if (vm.count("help"))
  {
    out << desc << std::endl;
    return 1;
  }

  if (m_hostname.empty())
  {
    out << "Endereco deve ser fornecido para o servidor" << std::endl;
    return 2;
  }

  if (m_port == 0)
  {
    out << "A porta para conexao nao pode ser 0" << std::endl;
    return 4;
  }

  if (m_buffer_size < 1)
  {
    out << "O tamanho do buffer nao pode ser menor que 1" << std::endl;
    return 5;
  }

  if (m_filename.empty())
  {
    out << "O arquivo local deve ser informado" << std::endl;
    return 5; 
  }

  if (key.length() == 0 || key.length() % 2 != 0)
  {
    out << "A chave passada tem tamanho invalido: " << key << std::endl;
    return 6;
  }

  m_key.resize(key.length() / 2);
  try
  {
    boost::algorithm::unhex(key.begin(), key.end(), m_key.begin());
  } 
  catch(...)
  {
    m_key.resize(0);
    out << "A chave passada e invalida: " << key << std::endl;
    return 7;
  }

  return 0;
}

const std::string& Config::hostname() const
{
  return m_hostname;
}

void Config::hostname(std::string const& value)
{
  m_hostname = value;
}

uint16_t Config::port() const
{
  return m_port;
}

void Config::port(uint16_t value)
{
  m_port = value;
}

size_t Config::buffer_size() const
{
  return m_buffer_size;
}

void Config::buffer_size(size_t value)
{
  m_buffer_size = value;
}

Config::Operation Config::operation() const
{
  return m_operation;
}

void Config::operation(Config::Operation value)
{
  m_operation = value;
}

const std::string& Config::filename() const
{
  return m_filename;
}

void Config::filename(const std::string& value)
{
  m_filename = value;
}

const std::vector<uint8_t>& Config::key() const
{
  return m_key;
}

void Config::key(const std::vector<uint8_t>& value)
{
  m_key = value;
}

int64_t Config::size() const
{
  return m_size;
}

void Config::size(int64_t value)
{
  m_size = value;
}

bool Config::close() const
{
  return m_close;
}

void Config::close(bool value)
{
  m_close = value;
}
