// -*- coding: utf-8-unix -*-

#include "pch.h"

#include <ftc/RemoteDataChannel.h>
#include <ftc/FileChannel.h>

#include <fstream>

#include <boost/cstdint.hpp>
#include <boost/program_options.hpp>
#include <boost/filesystem.hpp>
#include <boost/scoped_ptr.hpp>

#include <boost/version.hpp>
#if (BOOST_VERSION == 105100)
namespace boost { namespace algorithm { namespace detail {
inline unsigned hex_char_to_int ( char c );
}}}
#endif
#include <boost/algorithm/hex.hpp>

#if defined(_MSC_VER)
#pragma comment(lib, "ftc.lib")
#endif

#include "Config.h"


//[do_read
int do_read(Config &config, RemoteDataChannel &channel)
{
  boost::scoped_ptr<FileChannel> file;
  try
  {
    file.reset(new FileChannel(config.filename(), false)); /*<
       Representação do arquivo local de saída. O cliente vai escrever neste 
       arquivo os dados lidos do arquivo remoto que é fornecido pelo servidor 
       FTC.>*/
  }
  catch(const std::exception &e)
  {
    std::cerr << "Nao foi possivel abrir o arquivo local para escrita: " 
              << config.filename() << " (" << e.what() << ")" << std::endl;
    return 200;
  }
  catch(...)
  {
    std::cerr << "Excecao desconhecida foi lancada ao tentar abrir o arquivo "
              << "local para escrita: " << config.filename() << std::endl;
    return 201;
  }
  int64_t size = config.size(); 
  if (size == -1)
  {
    try
    {
      size = channel.size(); /*<Obtêm o tamanho do arquivo remoto.>*/
    }
    catch(const UnsupportedOperationException &e)
    {
      std::cerr << "O canal remoto nao suporta a aquisicao do tamanho: " 
                << e.what() << " [" << e.location() << "]" << std::endl;
      return 201;
    }
    catch(const FailureException &e)
    {
      std::cerr << "Falha na comunicacao ponto a ponto: " 
                << e.what() << " [" << e.location() << "]" << std::endl;
      return 202;
    }
    catch(const IllegalStateException &e)
    {
      std::cerr << "O canal nao esta aberto: " 
                << e.what() << " [" << e.location() << "]" << std::endl;
      return 203;
    }
    catch(const std::exception &e)
    {
      std::cerr << "Excecao foi lancada ao tentar adquirir o "
                << "tamanho do canal remoto: " << e.what() << std::endl;
      return 204;
    }
    catch(...)
    {
      std::cerr << "Excecao desconhecida foi lancada ao tentar "
                << "adquirir o tamanho do canal remoto" << std::endl;
      return 205;
    }
  }

  std::cout << "Transferindo " << size << " bytes do canal remoto" << std::endl;  

  uint64_t bytes_transfered = 0;
  try
  {
    bytes_transfered = channel.transferTo(0, size, *file); /*<
      Transferência dos dados do arquivo remoto para o arquivo local de saída 
      através do canal de fluxo de dados (`RemoteDataChannel`). >*/
  }
  catch(const UnsupportedOperationException &e)
  {
    std::cerr << "O canal remoto nao suporta leitura: " 
              << e.what() << " [" << e.location() << "]" << std::endl;
    return 206;
  }
  catch(const FailureException &e)
  {
    std::cerr << "Falha na comunicacao ponto a ponto: " 
              << e.what() << " [" << e.location() << "]" << std::endl;
    return 207;
  }
  catch(const std::exception &e)
  {
    std::cerr << "Excecao foi lancada ao ler dados do canal remoto: " 
              << e.what() << std::endl;
    return 208;
  }
  catch(...)
  {
    std::cerr << "Excecao desconhecida foi lancada ao ler "
              << "dados do canal remoto" << std::endl;
    return 209;
  }  
  
  std::cout << "Transferidos " << bytes_transfered << " bytes" << std::endl;  
  return 0;
}//]

//[do_write
int do_write(Config &config, RemoteDataChannel &channel)
{
  boost::scoped_ptr<FileChannel> file;
  try
  {
    file.reset(new FileChannel(config.filename(), false)); /*<
       Representação do arquivo local de entrada. O cliente vai transferir o conteúdo 
       deste arquivo para o arquivo disponibilizado pelo servidor.>*/
  }
  catch(const std::exception &e)
  {
    std::cerr << "Nao foi possivel abrir o arquivo local para escrita: " 
              << config.filename() << " (" << e.what() << ")" << std::endl;
    return 300;
  }
  catch(...)
  {
    std::cerr << "Excecao desconhecida foi lancada ao tentar abrir "
              << "o arquivo local para escrita: " << config.filename() << std::endl;
    return 301;
  }

  int64_t size = config.size();  /*<Obtêm o tamanho do arquivo local.>*/
  if (size == -1)
  {
    boost::system::error_code ec;
    size = boost::filesystem::file_size(
      boost::filesystem::path(config.filename()), ec);
    if(ec)
    {
      std::cerr << "Nao foi possivel adquirir o tamanho do arquivo local: " 
                << config.filename() << std::endl;
      return 302;
    }
  }

  std::cout << "Transferindo " << size 
            << " bytes para o canal remoto" << std::endl;  

  uint64_t bytes_transfered = 0;
  try
  {
    bytes_transfered = channel.transferFrom(*file, 0, size); /*<
      Transferência dos dados do arquivo local para o arquivo remoto 
      através do canal de fluxo de dados (`RemoteDataChannel`). >*/
  }
  catch(const UnsupportedOperationException &e)
  {
    std::cerr << "O canal remoto nao suporta escrita: " 
          << e.what() << " [" << e.location() << "]" << std::endl;
    return 303;
  }
  catch(const FailureException &e)
  {
    std::cerr << "Falha na comunicacao ponto a ponto: " 
          << e.what() << " [" << e.location() << "]" << std::endl;
    return 304;
  }
  catch(const std::exception &e)
  {
    std::cerr << "Excecao foi lancada ao escrever dados no canal remoto: " 
          << e.what() << std::endl;
    return 305;
  }
  catch(...)
  {
    std::cerr << "Excecao desconhecida foi lancada ao escrever "
          << "dados no canal remoto" << std::endl;
    return 306;
  }  

  std::cout << "Transferidos " << bytes_transfered << " bytes" << std::endl;  
  return 0;
}//]

int main(int argc, char *argv[])
{
  //[remote_data_channel
  Config config;  
  int error = config.parse(argc, argv, std::cerr);
  if (error)
  {
    return error;
  }  
  RemoteDataChannel channel(config.operation() == Config::OP_WRITE, 
                            config.hostname(), config.port(), config.key());//]

  //[channel_open
  try
  {
    channel.open();
  }//]
  catch(const FailureException &e)
  {
    std::cerr << "Ja estava aberto, falha na conexao/comunicacao ponto a ponto,"
         << "falha na autenticacao, falha na negociacao ou "
         << "falha ao abrir arquivo: " << e.what()
         << " [" << e.location() << "]" << std::endl;
    return 100;
  }
  catch(const InvalidKeyException &e)
  {
    std::cerr << "A chave passada nao foi reconhecida pelo "
              << "servidor como valida: " 
              << e.what() << " [" << e.location() << "]" << std::endl;
    return 101;
  }
  catch(const InvalidProtocolVersionException &e)
  {
    std::cerr << "Versao nao reconhecida ou nao suportada pelo servidor: " 
         << e.what() << " [" << e.location() << "]" << std::endl;
    return 102;
  }
  catch(const MaxClientsReachedException &e)
  {
    std::cerr << "Número maximo de conexoes suportadas pelo "
              << "servidor foi alcancado: "
              << e.what() << " [" << e.location() << "]" << std::endl;
    return 103;
  }
  catch(const FileNotFoundException &e)
  {
    std::cerr << "Arquivo solicitado nao foi encontrado pelo servidor: " 
              << e.what() << " [" << e.location() << "]" << std::endl;
    return 104;
  }
  catch(const NoPermissionException &e)
  {
    std::cerr << "Sem permissao para abrir o arquivo remoto: " 
              << e.what() << " [" << e.location() << "]" << std::endl;
    return 105;
  }
  catch(const IllegalStateException &e)
  {
    std::cerr << "Sem permissao para abrir o arquivo remoto: " 
              << e.what() << " [" << e.location() << "]" << std::endl;
    return 106;
  }
  catch(const std::exception &e)
  {
    std::cerr << "Excecao foi lancada ao tentar abrir o canal remoto: " 
              << e.what() << std::endl;
    return 107;
  }
  catch(...)
  {
    std::cerr << "Excecao desconhecida foi lancada ao tentar "
              << "abrir o canal remoto" << std::endl;
    return 108;
  }

  std::cout << "Rodando em: " << config.hostname() 
            << ":" << config.port() << std::endl;
  std::cout << "Operacao: " 
            << (config.operation() == Config::OP_READ ? "READ" : "WRITE") 
            << std::endl;  

  error = 999;
  switch (config.operation())
  {
  case Config::OP_READ: 
    //[exec_do_read]
    error = do_read(config, channel);//]
    break;
  case Config::OP_WRITE: 
    //[exec_do_write]
    error = do_write(config, channel);//]
    break;
  }

  //[channel_close
  if (config.close())
  try
  {
    std::cout << "Fechando conexao com o servidor..." << std::endl;
    channel.close();
  }//]
  catch(const FailureException &e)
  {
    std::cerr << "Falha na comunicacao ponto a ponto: " 
         << e.what() << " [" << e.location() << "]" << std::endl;
    return 400;
  }
  catch(const IllegalStateException &e)
  {
    std::cerr << "Canal nao esta aberto: " 
         << e.what() << " [" << e.location() << "]" << std::endl;
    return 401;
  }
  catch(const std::exception &e)
  {
    std::cerr << "Excecao foi lancada ao tentar fechar o canal remoto: " 
         << e.what() << std::endl;
    return 402;
  }
  catch(...)
  {
    std::cerr << "Excecao desconhecida foi lancada ao tentar "
              << "fechar o canal remoto" << std::endl;
    return 403;
  }
  return error;
}
