/** \file
 * \brief Attributes Table.
 *
 * See Copyright Notice in im_lib.h
 * $Id: Exp $
 */

#ifndef __IM_ATTRIB_H_
#define __IM_ATTRIB_H_

struct imAttribTablePrivate;

/** Definition of a callback for each attribute. */
typedef int (*imAttribTableCallback)(void* user_data, int index, const char* name, int data_type, int count, const void* data);

#include "im_attrib_flat.h"

/** \brief Attributes Table.
 *
 * \par
 * All the attributes have a name, a type, a count and the data.\n
 * Names are usually strings with less that 30 chars.
 * \par
 * Attributes are stored in a hash table for fast access. \n
 * We use the hash function described in "The Pratice of Programming" of Kernighan & Pike.
 * \ingroup util */
class imAttribTable
{
  imAttribTablePrivate* ptable; 
public:

  /** Creates an empty table.
   * If size is zero the default size of 101 is used. Size must be a prime number. 
   * Other common values are 67, 599 and 1499.*/ 
  imAttribTable(int hash_size)
    { ptable = imAttribTableCreate(hash_size); }

  /** Destroys the table and all the attributes. */
  ~imAttribTable()
    { imAttribTableDestroy(ptable); ptable = 0; }

  /** Returns the number of elements in the table. */
  int Count() const
    { return imAttribTableCount(ptable); }

  /** Removes all the attributes in the table */
  void RemoveAll()
    { imAttribTableRemoveAll(ptable); }

  /** Copies the contents of the given table into this table. */
  void CopyFrom(const imAttribTable& table)
    { imAttribTableCopyFrom(ptable, table.ptable); }

  /** Inserts an attribute into the table. 
   * Data is duplicated. */
  void Set(const char* name, int data_type, int count, const void* data)
    { imAttribTableSet(ptable, name, data_type, count, data); }

  /** Removes an attribute from the table given its name. */
  void UnSet(const char *name)
    { imAttribTableUnSet(ptable, name); }

  /** Finds an attribute in the table.
   * Returns the attribute if found, NULL otherwise. */
  const void* Get(const char *name, int *data_type = 0, int *count = 0) const
    { return imAttribTableGet(ptable, name, data_type, count); }

  /** For each attribute calls the user callback. If the callback returns 0 the function returns. */
  void ForEach(void* user_data, imAttribTableCallback attrib_func) const
    { imAttribTableForEach(ptable, user_data, attrib_func); }
};

#endif
