%% CrossMember class
%
% This class defines a member of a continuous beam for the Cross Process.
% Euler-Bernoulli flexural behavior is assumed (Navier beam theory).
%
% See definition of <crosssolver.html *CrossSolver*> class.
%
%% Author
% Luiz Fernando Martha
%
%% History
% @version 1.01
%
% Initial version 1.00: August 2025
%%%
% Initially prepared for the course CIV 2801 - Fundamentos de Computao
% Grfica, 2025, second term, Department of Civil Engineering, PUC-Rio.
%
% Version 1.01: October 2025
%%%
% Created constant properties tol_discrim and tol_rootpos;
% Created properties displv, bendmom, spanmom_pos and spanmom.
% Created private methods fixedEndUniformLoadDispl and displShapeFcnVector.
% Created public methods internalDispl, bendingMoment and momentRoots.
%

%% Class definition
classdef CrossMember < handle
    %%
    % <https://www.mathworks.com/help/matlab/ref/handle-class.html
    % See documentation on *handle* super-class>.
    
    %% Public attributes
    properties (SetAccess = public, GetAccess = public)
        EI = 0;            % flexural stiffness
        len = 0;           % member length
        q = 0;             % vertical distributed load (top-down positive)
        ml = 0;            % left end moment
        mr = 0;            % right end moment
        k = 0;             % rotational stiffness coefficient
        displv = [];       % vector of member transversal displacements
        bendmom = [];      % vector of member bending moments
        spanmom_pos = 0;   % position of maximum span moment
        spanmom = 0;       % maximum span moment
    end
    
    %% Class (constant) properties
    properties (Constant)
        tol_discrim = 1e-7; % tolerance for discriminant of quadratic
                            % equation
        tol_rootpos = 1e-2; % tolerance for root position proximity to
                            % member extremities
    end
    
    %% Constructor method
    methods
        %------------------------------------------------------------------
        function memb = CrossMember(EI,len,q)
            if (nargin > 0)
                memb.EI = EI;
                memb.len = len;
                memb.q = q;
                memb.ml = 0;
                memb.mr = 0;
                memb.k = 0;
                memb.displv = [];
                memb.bendmom = [];
                memb.spanmom_pos = 0;
                memb.spanmom = 0;
            end
        end
    end
    
    %% Private methods
    methods (Access = private)
        %------------------------------------------------------------------
        % Computes element transversal fixed-end displacement at given
        % cross-section positions (given in a array) for an applied
        % uniformely distributed load.
        % Output:
        %  dvI: array of fixed-end transversal displacements
        % Input arguments:
        %  contin_ini: rotation continuity at initial node
        %              (0 -> hinge, 1 -> continuous)
        %  contin_end: rotation continuity at end node
        %              (0 -> hinge, 1 -> continuous)
        %  x: array of element cross-section positions in local x-axis
        function dvI = fixedEndUniformLoadDispl(memb,contin_ini,contin_end,x)
            
            % Basic element length properties
            L  = memb.len;
            L2 = L  * L;
            L3 = L2 * L;
 
            % Calculate transversal displacements
%%%%%%% COMPLETE HERE - CROSSMEMBER: 01 %%%%%%%
        end
        
        %------------------------------------------------------------------
        % Evaluates member flexural displacement shape functions in local
        % at given cross-section positions (given in a array).
        % Flexural shape functions give the transversal displacement of an
        % element subjected to unit nodal rotations.
        % Output:
        %  Nv: a (2 x np) matrix with flexural displacement shape functions:
        %      ini -> initial node
        %      end -> end node
        %  Nv = [ N_ini;
        %         N_end ]
        % Input arguments:
        %  contin_ini: rotation continuity at initial node
        %              (0 -> hinge, 1 -> continuous)
        %  contin_end: rotation continuity at end node
        %              (0 -> hinge, 1 -> continuous)
        %  x: array of element cross-section positions in local x-axis
        function Nv = displShapeFcnVector(memb,contin_ini,contin_end,x)
            
            % Get number of points
            np = size(x,2);
            
            % Basic element length properties
            L  = memb.len;
            L2 = L  * L;
                        
            % Compute shape function values based on end rotation
            % continuity conditions
%%%%%%% COMPLETE HERE - CROSSMEMBER: 02 %%%%%%%
            
            Nv = [ Nv2;
                   Nv4 ];
        end
    end
    
    %% Public methods
    methods
        %------------------------------------------------------------------
        % Computes member transversal internal displacements vector at
        % given cross-section positions (given in a array) from nodal
        % rotations. Store them in corresponding member property (displv).
        % Output:
        %  maxv: maximum absolute transversal displacement value.
        % Input arguments:
        %  contin_ini: rotation continuity at initial node
        %              (0 -> hinge, 1 -> continuous)
        %  contin_end: rotation continuity at end node
        %              (0 -> hinge, 1 -> continuous)
        %  rot_ini: rotation of initial node
        %  rot_end: rotation of end node 
        %  x: array of element cross-section positions in local x-axis
        function maxv = internalDispl(memb,contin_ini,contin_end,rot_ini,rot_end,x)

            % Get internal displacements from local fixed-end applyed load
            dvI = memb.fixedEndUniformLoadDispl(contin_ini,contin_end,x);
            
            % Evaluate element shape function matrix at given cross-section
            % positions
            Nv = memb.displShapeFcnVector(contin_ini,contin_end,x);
            
            % Assemble a vector of member end node rotations
            membrot = [rot_ini rot_end];
            
            % Compute internal displacements from global analysis (from
            % nodal rotations)
            dvII = membrot * Nv;

            % Assemble total element transversal displacements vector
            memb.displv = dvI + dvII;
            
            % Compute maximum absolute displacement value
            maxv = max([abs(min(memb.displv)) abs(max(memb.displv))]);
        end
        
        %------------------------------------------------------------------
        % Computes member bending moment vector at given cross-section
        % positions (given in a array) from nodal rotations. Store them
        % in corresponding member property (bendmom).
        % Computes maximum span moment and its section position along
        % member and store them in the corresponding member properties
        % (spanmom and spanmom_pos). Then maximum span moment position
        % corresponds to the cross-section in which the internal shear
        % force is null.
        % Input arguments:
        %  mom_ini: bending moment of initial node
        %  mom_end: bending moment of end node 
        %  x: array of element cross-section positions in local x-axis
        % Output:
        %  maxm: maximum absolute bending moment value
        function maxm = bendingMoment(memb,mom_ini,mom_end,x)
%%%%%%% COMPLETE HERE - CROSSMEMBER: 03 %%%%%%%
            
            % Compute maximum bending moment value
            maxm = max([abs(min(memb.bendmom)) abs(max(memb.bendmom))]);
        end
        
        %------------------------------------------------------------------
        % Computes roots of member quadratic bending moment diagram.
        % Returns number of roots and root positions.
        % If a root is outside member span or close to a member
        % extrimity, it is not considered.
        % Input arguments:
        %  mom_ini: bending moment of initial node
        %  mom_end: bending moment of end node 
        % Output:
        %  nroots: number of roots of quadratic bending moment diagram
        %  xroots: array of roots positions in local x-axis
        function [nroots,xroots] = momentRoots(memb,mom_ini,mom_end)
            nroots = 0;
            xroots = zeros(2);
%%%%%%% COMPLETE HERE - CROSSMEMBER: 04 %%%%%%%
        end
        
        %------------------------------------------------------------------
        % Cleans data structure of a CrossMember object.
        function memb = clean(memb)
            memb.EI = 0;
            memb.len = 0;
            memb.q = 0;
            memb.ml = 0;
            memb.mr = 0;
            memb.k = 0;
            memb.displv = [];
            memb.bendmom = [];
            memb.spanmom_pos = 0;
            memb.spanmom = 0;
        end
    end
end