%% CanvasCross class
%
% This is a sub-class of the <canvas_2d.html *Canvas_2D*> class, to deal
% especifically with axes that present only 2D behavior, on the Cross
% process app.
%
%% Author
%%%
% * Luiz Fernando Martha (lfm@tecgraf.puc-rio.br)
%
%% History
% @version 1.00
%
% Initial version (1.00): September 2025
%%%
% Initially prepared for the course CIV 2801 - Fundamentos de Computao
% Grfica, 2025, second term, Department of Civil Engineering, PUC-Rio.
%

%% Class definition
classdef CanvasCross < Canvas_2D
    %% Public properties - mouse events
    properties (Access = public)
        control = [];               % handle to Cross control object
        firstPosition = [];         % x and y coordinates of the first pointer position.
        firstPositionGiven = false; % flag indicating that first point was selected.
    end

    %% Constructor method
    methods
        function this = CanvasCross(ax,fig,fcn,control)
            if (nargin == 0)
                ax  = [];
                fig = [];
                fcn = [];
            elseif (nargin <= 2)
                fcn = [];
            end
            this = this@Canvas_2D(ax,fig,fcn);
            this.control = control;
        end
    end
    
    %% Public methods
    % Implementation of the abstract methods declared in super-class Canvas.
    methods    
        %------------------------------------------------------------------
        % Mouse button down callback - related to axes object
        % Input:
        % * this  = handle to this canvas object
        % * obj   = handle to graphical object that was clicked on
        % * event = struct with event data
        function ax_onButtonDown(~,~,~)
        end
        
        %------------------------------------------------------------------
        % Mouse button down callback - generic callback, called from an
        % Emouse object.
        % Input:
        % * this  = handle to this canvas object
        % * pt    = current cursor coordinates
        % * whichMouseButton = 'left', 'right', 'center', 'double click'
        function onButtonDown(this,pt,whichMouseButton)
            switch whichMouseButton
                case 'left'
                    % Store first position coordinates
                    this.firstPosition = pt;
                    
                    % Set boolean flag
                    this.firstPositionGiven = true;
                    
                    % Call Cross control button down method
                    this.control.onButtonDown(this,pt);
            end
        end
        
        %------------------------------------------------------------------
        % Mouse button up callback - generic callback, called from an
        % Emouse object.
        % Input:
        % * this  = handle to this canvas object
        % * pt    = current cursor coordinates
        % * whichMouseButton = 'left', 'right', 'center', 'double click'
        function onButtonUp(this,pt,~)
            % Call Cross control button up method
            this.control.onButtonUp(this,pt);

            % Set boolean flag
            this.firstPositionGiven = false;
        end
        
        %------------------------------------------------------------------
        % Mouse move callback - generic callback, called from an
        % Emouse object.
        % Input:
        % * this  = handle to this canvas object
        % * pt    = current cursor coordinates
        function onMouseMove(this,pt)
            % Check if mouse is collecting second point
            if this.firstPositionGiven
                % Check if second point is different from first
                if all(this.firstPosition(1:2,1) ~= pt(1:2,1))                    
                    % Call Cross control mouse move method
                    this.control.onMouseMove(this,pt);
                end
            else
                % If mouse is not collecting second point, change
                % mouse pointer depending on current action.
                this.control.setMousePointer(this,this.fig,pt);
            end
        end
        
        %------------------------------------------------------------------
        % Mouse scroll callback - generic callback, called from an
        % Emouse object.
        % Input:
        % * this  = handle to this canvas object
        % * pt    = current cursor coordinates
        function onMouseScroll(~,~)
        end
    end
end