%% CrossControl class
%
% This class implements methods to control the analysis of continuous
% beams by the Cross process.
%
%% Author
% Luiz Fernando Martha
%
%% History
% @version 1.00
%
% Initial version (1.00): September 2025
%%%
% Initially prepared for version Cross01 application of course CIV 2801 -
% Fundamentos de Computao Grfica, 2025, second term, Department of
% Civil Engineering, PUC-Rio.
%

%% Class definition
classdef CrossControl < handle
    %% Public properties
    properties (Access = public)
        solver = []; % handle to Cross solver object
    end
    
    %% Private properties
    properties (Access = private)
        app = []; % handle to GUI app object
    end
    
    %% Constructor method
    methods
        %------------------------------------------------------------------
        function control = CrossControl(app)
            if (nargin > 0)
                control.app = app;
            end
        end
    end
    
    %% Private methods
    methods (Access = private)
        %------------------------------------------------------------------
        % Reset Cross solver
        function resetSolver(control,nmemb,supinit,supend,decplc,EI,len,q)            
            % Clean current Cross model if it exists
            if ~isempty(control.solver)
                control.solver.clean();
            end
            
            % Create new Cross solver model and initialize it
            if (nargin == 1)
                control.solver = CrossSolver();
            else
                control.solver = CrossSolver(nmemb,supinit,supend,decplc,EI,len,q);
            end
            control.solver.initStiffness();
            control.solver.initNodes();
            control.solver.initMoments();

            % Update init and end support icons
            control.app.updateSupInit(control.solver.supinit);
            control.app.updateSupEnd(control.solver.supend);

            % Print model data and initial bending moments
            control.solver.printModelInfo(1);
            control.solver.printResults(1);
        end
    end
    
    %% Public methods
    methods
        %------------------------------------------------------------------
        % Start up Cross program. Create a Cross solver object with a
        % default continous beam model. Initialize Cross parameters and
        % beam spans with bending fixed-end moments. Print model
        % and initial results.
        function startup(control)
            % Create default Cross solver object
            control.resetSolver();
        end
        
        %------------------------------------------------------------------
        % Open file with continuous beam model for Cross process analysis.
        function openFile(control,filename)
            % Open file with given file name for reading
            fid = fopen(filename,'rt');

            % Read moment precision (number of decimal places)
            decplc = fscanf(fid,'%f',1);
            if decplc < 0 || decplc > 2
                decplc = 1;
            end

            % Read support boundary conditions
            bdrycnd = fscanf(fid,'%f',2);
            supinit = bdrycnd(1);
            supend = bdrycnd(2);
            
            % Read number of members and dimension arrays
            nmemb = fscanf(fid,'%f',1);
            EI = ones(1,nmemb) * 10000;
            len = zeros(1,nmemb);
            q = zeros(1,nmemb);
            
            % Read member lengths and uniform load
            for i = 1:nmemb
                membprop = fscanf(fid,'%f',2);
                len(i) = membprop(1);
                q(i) = membprop(2);
            end
            
            % Create Cross solver object with read data
            control.resetSolver(nmemb,supinit,supend,decplc,EI,len,q);
           
            % Close file
            fclose(fid);
        end
                
        %------------------------------------------------------------------
        % Save current continuous beam model for Cross process analysis in
        % a file
        function saveFile(control,filename)
            % Open file with given file name for writing
            fid = fopen(filename,'wt+');

            % Write moment precision (number of decimal places)
            fprintf(fid, '%d\n', control.solver.decplc);

            % Write support boundary conditions
            fprintf(fid, '%d  %d\n', control.solver.supinit, ...
                                     control.solver.supend);
            
            % Write number of members
            fprintf(fid, '%d\n', control.solver.nmemb);

            % Write member lengths and uniform load
            for i = 1:control.solver.nmemb
                fprintf(fid, '%d  %d\n', control.solver.membs(i).len, ...
                                         control.solver.membs(i).q);
            end

            % Close file
            fclose(fid);
        end
        
        %------------------------------------------------------------------
        % Restart continous beam Cross analysis.
        % Initialize Cross parameters and beam spans bending with
        % fixed-end moments. Print model and initial bending moment
        % results.
        function restart(control)
            control.solver.printModelInfo(1);
%%%%%%% COMPLETE HERE - CROSS_CONTROL: 01 %%%%%%%
            control.solver.resetNodeRotations();
            control.solver.printResults(1);
        end
        
        %------------------------------------------------------------------
        % Solves one step of Cross Process for continuous beams: solve the
        % node with maximum absolute value of unbalanced moment.
        % Print bending moment resulta after step.
        function autoStep(control)
%%%%%%% COMPLETE HERE - CROSS_CONTROL: 02 %%%%%%%
            control.solver.stepVector(...
                control.solver.numSteps).printStep(1,control.solver.decplc);
            control.solver.printResults(1);            
        end
        
        %------------------------------------------------------------------
        % Processes direct solver of Cross Process for continuous beams.
        % Print final bending moment results.
        function goThru(control)
%%%%%%% COMPLETE HERE - CROSS_CONTROL: 03 %%%%%%%
            control.solver.printResults(1);
        end
        
        %------------------------------------------------------------------
        % Set up number of decimal places for iterative Cross Process
        % of continuous beams.
        % Reprint current bending moment results.
        function setPrecision(control,decplc)
%%%%%%% COMPLETE HERE - CROSS_CONTROL: 04 %%%%%%%
            control.solver.printResults(1);
        end
        
        %------------------------------------------------------------------
        % Toggle initial beam support condition and restart continous
        % beam Cross analysis.
        function toggleSupInit(control)
%%%%%%% COMPLETE HERE - CROSS_CONTROL: 05 %%%%%%%
        end
        
        %------------------------------------------------------------------
        % Toggle end beam support condition and restart continous
        % beam Cross analysis.
        function toggleSupEnd(control)
%%%%%%% COMPLETE HERE - CROSS_CONTROL: 06 %%%%%%%
        end
    end
end