%% Read file Function (Version 2.0)
%
% This is an auxiliary file of the <main.html LESM> (Linear Elements
% Structure Model) program that contains functions to read a neutral-format
% file with the _.lsm_ extension 
% This neutral-format file contains all information about a linear elements
% structural model.
%
%% Authors
% Luiz Fernando Martha, Rafael Lopez Rangel and Pedro Cortez Lopes
%
%% Main function
% Output:
%  vs:      flag for version compatibility
%  print:   object of the Print class
%  draw:    object of the Draw class
% Input arguments:
%  fid: integer identifier of the input file
%  drv: handle to an object of the Drv class
function [f,print,draw,nclc,nfixed,nspring] = readFile_v2(fid,drv,flag,GUI_Mode)
    % Initialize nclc, nfixed and nspring
    nclc = 1;
    nfixed = 0;
    nspring = 0;

    f = flag;
    while f
        % Get file line
        tline = fgetl(fid);
        % Get rid of blank spaces
        string = deblank(tline);
        
        % Look for for tag strings
        switch string
            case '%HEADER.ANALYSIS'
                [print,draw] = readHeaderAnalysis(fid,drv,GUI_Mode);
            case '%NODE.COORD'
                readNodeCoord(fid,drv);
            case '%NODE.SUPPORT'
                nfixed = readNodeSupport(fid,drv);
            case '%NODE.SUPPORT.SPRING'
                nspring = readSpringStiff(fid,drv);
            case '%MATERIAL.ISOTROPIC'
                readMaterialIsotropic(fid,drv);
            case '%SECTION.PROPERTY'
                readSectionProperty(fid,drv);
            case '%BEAM.END.LIBERATION'
                rotlib = readBeamEndLiberation(fid);
            case '%ELEMENT.BEAM'
                readElementBeam(fid,drv,rotlib);
            case '%LOAD'
                readLoadCase(fid,drv);
            case '%LOAD.COMBINATION'
                readLoadComb(fid,drv);
            case '%LOAD.CASE'
                nclc = readCurrentLoadCase(fid);
            case '%LOAD.CASE.NODAL.CASES'
                readNodalLoadCase(fid,drv);
            case '%LOAD.CASE.BEAM.CASES'
                readElemLoadCase(fid,drv);
            case '%END'
                setCurrentLoadCase(drv,nclc);
                f = 0;    
        end
    end
end

%% Auxiliary functions
%--------------------------------------------------------------------------
function [print,draw] = readHeaderAnalysis(fid,drv,GUI_Mode)
    % Get file line
    tline = fgetl(fid);
    % Get rid of blank spaces
    string = deblank(tline);
    
    % Get target analysis type
    switch string
        case '''TRUSS2D'''
            drv.anm = Anm_Truss2D();
            print = Print_Truss2D(drv);
            if GUI_Mode == true
                draw = Draw_Truss2D(drv);
            else
                draw = [];
            end
        case '''FRAME2D'''
            drv.anm = Anm_Frame2D();
            print = Print_Frame2D(drv);
            if GUI_Mode == true
                draw = Draw_Frame2D(drv);
            else
                draw = [];
            end
        case '''GRILLAGE'''
            drv.anm = Anm_Grillage();
            print = Print_Grillage(drv);
            if GUI_Mode == true
                draw = Draw_Grillage(drv);
            else
                draw = [];
            end
        case '''TRUSS3D'''
            drv.anm = Anm_Truss3D();
            print = Print_Truss3D(drv);
            if GUI_Mode == true
                draw = Draw_Truss3D(drv);
            else
                draw = [];
            end
        case '''FRAME3D'''
            drv.anm = Anm_Frame3D();
            print = Print_Frame3D(drv);
            if GUI_Mode == true
                draw = Draw_Frame3D(drv);
            else
                draw = [];
            end
    end
end

%--------------------------------------------------------------------------
function readNodeCoord(fid,drv)
    % Read and store total number of nodes and equations
    nnp = fscanf(fid,'%f',1);
    drv.nnp = nnp;
    drv.neq = nnp * drv.anm.ndof;
    
    % Initialize vector of objects of the Node class
    nodes(1,nnp) = Node();
    drv.nodes = nodes;
    
    for n = 1:nnp
        b = fscanf(fid,'%f',4);
        % b(1) = node id
        % b(2) = X coordinate
        % b(3) = Y coordinate
        % b(4) = Z coordinate
        drv.nodes(n).id = b(1);
        drv.nodes(n).coord = [b(2) b(3) b(4)];
    end
end

%--------------------------------------------------------------------------
function nfixed = readNodeSupport(fid,drv)
    % Read total number of nodes with essential boundary conditions (supports)
    nnebc = fscanf(fid,'%f',1);
    
    % Initialize number of fixed dof's
    nfixed = 0;
    
    for n = 1:nnebc
        b = fscanf(fid,'%f',[1 7]);
        % b(1) = node id
        % b(2) = Dx
        % b(3) = Dy
        % b(4) = Dz
        % b(5) = Rx
        % b(6) = Ry
        % b(7) = Rz
        drv.nodes(b(1)).ebc = [b(2) b(3) b(4) b(5) b(6) b(7)];
        for i = 1:6
           if b(i+1) == 1
               nfixed = nfixed + 1;
           end    
        end
    end
end

%--------------------------------------------------------------------------
function nspring = readSpringStiff(fid,drv)
    % Read total number of nodes with spring supports
    nnsp = fscanf(fid,'%f',1);
    
    % Initialize number of springs
    nspring = 0;
    
    for n = 1:nnsp
        b = fscanf(fid,'%f',[1 7]);
        % b(1) = node id
        % b(2) = kdx
        % b(3) = kdy
        % b(4) = kdz
        % b(5) = krx
        % b(6) = kry
        % b(7) = krz
        drv.nodes(b(1)).springStiff = [b(2) b(3) b(4) b(5) b(6) b(7)];
        for i = 1:6
           if b(i+1) ~= 0
               nspring = nspring + 1;
           end    
        end    
    end
end

%--------------------------------------------------------------------------
function readMaterialIsotropic(fid,drv)
    % Read and store total number of materials
    nmat = fscanf(fid,'%f',1);
    drv.nmat = nmat;
    
    % Initialize vector of objects of the Material class
    materials(1,nmat) = Material();
    drv.materials = materials;
    
    for m = 1:nmat
        b = fscanf(fid,'%f',4);
        % b(1) = material id
        % b(2) = elasticity modulus [MPa]
        % b(3) = possion ratio
        % b(4) = thermal expansion coefficient  [/oC]
        drv.materials(m) = Material(b(1),1e3*b(2),b(3),b(4));
    end
end

%--------------------------------------------------------------------------
function readSectionProperty(fid,drv)
    % Read and store total number of cross-sections
    nsec = fscanf(fid,'%f',1);
    drv.nsec = nsec;
    
    % Initialize vector of objects of the Section class
    sections(1,nsec) = Section();
    drv.sections = sections;
    
    for s = 1:nsec
        b = fscanf(fid,'%f',[1 9]);
        % b(1) = Cross-section id
        % b(2) = Ax [cm2]
        % b(3) = Ay [cm2]
        % b(4) = Az [cm2]
        % b(5) = Ix [cm4]
        % b(6) = Iy [cm4]
        % b(7) = Iz [cm4]
        % b(8) = Hy [cm]
        % b(9) = Hz [cm]
        drv.sections(s).id = b(1);
        drv.sections(s).area_x = 1e-4*b(2);
        drv.sections(s).area_y = 1e-4*b(3);
        drv.sections(s).area_z = 1e-4*b(4);
        drv.sections(s).inertia_x = 1e-8*b(5);
        drv.sections(s).inertia_y = 1e-8*b(6);
        drv.sections(s).inertia_z = 1e-8*b(7);
        drv.sections(s).height_y = 1e-2*b(8);
        drv.sections(s).height_z = 1e-2*b(9);
    end
end

%--------------------------------------------------------------------------
function rotlib = readBeamEndLiberation(fid)
    include_constants;
    
    % Read number of beam end rotation liberation properties
    nrotlib = fscanf(fid,'%f',1);
    rotlib = zeros(2,nrotlib);
    
    % Read beam end rotation liberation (hinge) data
    for i = 1:nrotlib
        b = fscanf(fid,'%f',[1 13]);
        % b( 1) = rotation liberation property id (not used)
        % b( 2) = idx
        % b( 3) = idy
        % b( 4) = idz
        % b( 5) = irx
        % b( 6) = iry
        % b( 7) = irz
        % b( 8) = jdx
        % b( 9) = jdy
        % b(10) = jdz
        % b(11) = jrx
        % b(12) = jry
        % b(13) = jrz
        % If rotation liberation in any of the directions is free, it is
        % considered that it is free in all directions, thus it will receive a
        % hinged flag
        if (b(5)==0) || (b(6)==0) || (b(7)==0)
            rotlib(1,i) = HINGED_END;
        else
            rotlib(1,i) = CONTINUOUS_END;
        end
        
        if (b(11)==0) || (b(12)==0) || (b(13)==0)
            rotlib(2,i) = HINGED_END;
        else
            rotlib(2,i) = CONTINUOUS_END;
        end
    end
end

%--------------------------------------------------------------------------
function readElementBeam(fid,drv,rotlib)
    % Read and store total number of elements
    nel = fscanf(fid,'%f',1);
    drv.nel = nel;
    
    % Initialize vector of objects of the Elem class
    elems(1,nel) = Elem();
    drv.elems = elems;
    
    for e = 1:nel
        b = fscanf(fid,'%f',[1 10]);
        % b(1)  = element id
        % b(2)  = element type (0 = Navier, 1 = Timoshenko)
        % b(3)  = material id
        % b(4)  = section property id
        % b(5)  = end release id
        % b(6)  = init node id
        % b(7)  = final node id
        % b(8)  = vz_X
        % b(9)  = vz_Y
        % b(10) = vz_Z
        drv.elems(e) = Elem(b(2),...
                            drv.anm,...
                            drv.materials(b(3)),...
                            drv.sections(b(4)),...
                            [drv.nodes(b(6)), drv.nodes(b(7))],...
                            rotlib(1,b(5)), rotlib(2,b(5)),...
                            [b(8), b(9), b(10)],...
                            [],[],[],[],[],[],[],[]);
    end
end

%--------------------------------------------------------------------------
function readLoadCase(fid,drv)
    % Read total number of load cases
    drv.nlc = fscanf(fid,'%f',1);
    str0 = fgetl(fid); %#ok<NASGU>
    for i = 1:drv.nlc
       string(i) = {fgetl(fid)}; %#ok<AGROW>
    end
    drv.strLc = string';
end

%--------------------------------------------------------------------------
function nclc = readCurrentLoadCase(fid)
    % Read current load case id
    nclc = fscanf(fid,'%f',1);
     
end

%--------------------------------------------------------------------------
function readLoadComb(fid,drv)
     % Read total number of load case combinations
     drv.ncomb = fscanf(fid,'%f',1);
     for i = 1:drv.ncomb
        str0 = fgetl(fid); %#ok<NASGU>
        string(i) = {fgetl(fid)};  %#ok<AGROW>
        b = fscanf(fid,'%f',[1 drv.nlc]); 
        % b(1) = comb factor 1
        % b(2) = comb factor 2
        % (...)
        % b(nlc) = comb factor nlc
        if isempty(drv.loadComb) == 1
            drv.loadComb = zeros(drv.nlc,drv.ncomb);
        end
        drv.loadComb(:,i) = b';
        if isempty(drv.loadCombID) == 1
            drv.loadCombID = zeros(drv.nlc,drv.ncomb);
        end
        drv.loadCombID(:,i) = (~b' - 1) * -1;
     end
     drv.strComb = string';
end

%--------------------------------------------------------------------------
function readNodalLoadCase(fid,drv)
    % Read total number of nodes with applied load and/or prescribed
    % displacements in at least one load case
    nnlc = fscanf(fid,'%f',1);

    for i = 1:nnlc
        b = fscanf(fid,'%f',[1 14]);
        % b(1) = node id
        % b(2) = Fx [kN]
        % b(3) = Fy [kN]
        % b(4) = Fz [kN]
        % b(5) = Mx [kNm]
        % b(6) = My [kNm]
        % b(7) = Mz [kNm]
        % b(8) = Dx [mm]
        % b(9) = Dy [mm]
        % b(10) = Dz [mm]
        % b(11) = Rx [rad]
        % b(12) = Ry [rad]
        % b(13) = Rz [rad]
        % b(14) = number of load cases with loads in this specific node
        drv.nodes(b(1)).nodalLoadCase(:,1) = [b(2); b(3); b(4); b(5);...
                                              b(6); b(7); 1e-3*b(8);...
                                              1e-3*b(9); 1e-3*b(10);...
                                              b(11); b(12); b(13)];                                 
        if b(14) ~= 1
            for j = 2:b(14)
                c = fscanf(fid,'%f',[1 12]);
                % c(1) = Fx [kN]
                % c(2) = Fy [kN]
                % c(3) = Fz [kN]
                % c(4) = Mx [kNm]
                % c(5) = My [kNm]
                % c(6) = Mz [kNm]
                % c(7) = Dx [mm]
                % c(8) = Dy [mm]
                % c(9) = Dz [mm]
                % c(10) = Rx [rad]
                % c(11) = Ry [rad]
                % c(12) = Rz [rad]
                drv.nodes(b(1)).nodalLoadCase(:,j) = [c(1); c(2); c(3);...
                                                      c(4); c(5); c(6);...
                                                      1e-3*c(7); 1e-3*c(8);... 
                                                      1e-3*c(9); c(10);...
                                                      c(11); c(12)];
            end
        end
    end
end

%--------------------------------------------------------------------------
function readElemLoadCase(fid,drv)
    % Read total number of nodes with applied load and/or prescribed
    % displacements in at least one load case
    nelc = fscanf(fid,'%f',1);

    for i = 1:nelc
        b = fscanf(fid,'%f',[1 16]);
        % b(1) = elem id
        % b(2) = unifDir
        % b(3) = qX [kN/m]
        % b(4) = qY [kN/m]
        % b(5) = qZ [kN/m]
        % b(6) = linearDir
        % b(7) = qX1 [kN/m]
        % b(8) = qY1 [kN/m]
        % b(9) = qZ1 [kN/m]
        % b(10) = qX2 [kN/m]
        % b(11) = qY2 [kN/m]
        % b(12) = qZ2 [kN/m]
        % b(13) = dt_x [oC]
        % b(14) = dt_y [oC]
        % b(15) = dt_z [oC]
        % b(16) = number of load cases with loads in this specific node
        drv.elems(b(1)).load.elemLoadCase(:,1) = [b(2); b(3); b(4); b(5);...
                                                 b(6); b(7); b(8); b(9);...
                                                 b(10); b(11); b(12);...
                                                 b(13); b(14); b(15)];
        if b(16) ~= 1
            for j = 2:b(16)
                c = fscanf(fid,'%f',[1 14]);
                % c(1) = unifDir
                % c(2) = qX [kN/m]
                % c(3) = qY [kN/m]
                % c(4) = qZ [kN/m]
                % c(5) = linearDir
                % c(6) = qX1 [kN/m]
                % c(7) = qY1 [kN/m]
                % c(8) = qZ1 [kN/m]
                % c(9) = qX2 [kN/m]
                % c(10) = qY2 [kN/m]
                % c(11) = qZ2 [kN/m]
                % c(12) = dt_x [oC]
                % c(13) = dt_y [oC]
                % c(14) = dt_z [oC]
                drv.elems(b(1)).load.elemLoadCase(:,j) = [c(1); c(2); c(3);...
                                                         c(4); c(5); c(6);...
                                                         c(7); c(8); c(9);...
                                                         c(10); c(11); c(12);...
                                                         c(13); c(14)];
            end
        end
    end
end

%--------------------------------------------------------------------------
function setCurrentLoadCase(drv,lc)
% Get number of load cases
nlc = drv.nlc;

% Check if currebt load case is a case or a combination
if lc <= nlc
    % Allocate nodal loads and prescribed displacements in each Node object
    % obs.: nodalLoadCase(:,i) = [ fx        
    %                              fy
    %                              fz     (nodalLoadCase is a matrix, each
    %                              mx      column refers to one specific 
    %                              my      load case)
    %                              mz
    %                              dx
    %                              dy
    %                              dz
    %                              rx
    %                              ry
    %                              rz ]
    for n = 1:drv.nnp
        if isempty(drv.nodes(n).nodalLoadCase) == 0 && lc <= size(drv.nodes(n).nodalLoadCase,2)
            if all(drv.nodes(n).nodalLoadCase(1:6,lc) == 0) == 1
                drv.nodes(n).nodalLoad = []; 
            else %if there are any nodal loads, set them to nodalLoad
                drv.nodes(n).nodalLoad = drv.nodes(n).nodalLoadCase(1:6,lc);
            end
            if size(drv.nodes(n).nodalLoadCase,1) <= 6
                drv.nodes(n).prescDispl = [];
            elseif all(drv.nodes(n).nodalLoadCase(7:12,lc) == 0) == 1
                drv.nodes(n).prescDispl = [];
            else %if there are any prescribed displacements, set them to prescDispl
                drv.nodes(n).prescDispl = drv.nodes(n).nodalLoadCase(7:12,lc);
            end
        else
            drv.nodes(n).nodalLoad = []; 
            drv.nodes(n).prescDispl = [];
        end
    end    
    
    % Allocate element loads (distributed loads and thermal loads) in each
    % Lelem object
    % obs.: elemLoadCase(:,i) = [  unifDir
    %                                qx
    %                                qy
    %                                qz        (elemLoadCase is a matrix, 
    %                             linearDir     each column refers to one  
    %                                qx1        specific load case)
    %                                qy1
    %                                qz1
    %                                qx2
    %                                qy2
    %                                qz2
    %                                dtx
    %                                dty
    %                                dtz   ]
    for e = 1:drv.nel
        if isempty(drv.elems(e).load.elemLoadCase) == 0 && lc <= size(drv.elems(e).load.elemLoadCase,2)
            % Initialize uniform loads
            drv.elems(e).load.uniformGbl = [];    
            drv.elems(e).load.uniformLcl = [];
            if all(drv.elems(e).load.elemLoadCase(2:4,lc) == 0) == 1
                drv.elems(e).load.uniformDir = 0;  
            else  % if there are uniform loads, set their local and global components
                drv.elems(e).load.uniformDir = drv.elems(e).load.elemLoadCase(1,lc);
                drv.elems(e).load.setUnifLoad((drv.elems(e).load.elemLoadCase(2:4,lc))',drv.elems(e).load.elemLoadCase(1,lc));
            end
            % Initialize linear loads
            drv.elems(e).load.linearGbl = [];
            drv.elems(e).load.linearLcl = [];
            if size(drv.elems(e).load.elemLoadCase,1) <= 5
                drv.elems(e).load.linearDir = 0;
            elseif all(drv.elems(e).load.elemLoadCase(6:11,lc) == 0) == 1
                drv.elems(e).load.linearDir = 0;
            else  % if there are linear loads, set their local and global components
                drv.elems(e).load.linearDir = drv.elems(e).load.elemLoadCase(5,lc);
                drv.elems(e).load.setLinearLoad((drv.elems(e).load.elemLoadCase(6:11,lc))',drv.elems(e).load.elemLoadCase(5,lc));
            end
            % Set thermal loads
            if size(drv.elems(e).load.elemLoadCase,1) <= 11
                drv.elems(e).load.tempVar_X = 0;
                drv.elems(e).load.tempVar_Y = 0;  
                drv.elems(e).load.tempVar_Z = 0;
            else
                drv.elems(e).load.tempVar_X = drv.elems(e).load.elemLoadCase(12,lc);
                drv.elems(e).load.tempVar_Y = drv.elems(e).load.elemLoadCase(13,lc);
                drv.elems(e).load.tempVar_Z = drv.elems(e).load.elemLoadCase(14,lc);
            end
        else
            drv.elems(e).load.uniformDir = 0;      
            drv.elems(e).load.uniformGbl = [];    
            drv.elems(e).load.uniformLcl = [];    
            drv.elems(e).load.linearDir = 0;       
            drv.elems(e).load.linearGbl = [];
            drv.elems(e).load.linearLcl = [];
            drv.elems(e).load.tempVar_X = 0;
            drv.elems(e).load.tempVar_Y = 0;  
            drv.elems(e).load.tempVar_Z = 0; 
        end
    end
    
else  % if lc > nlc, the current load case is a combination
    
    % Consider load cases factors for the selected combination and allocate
    % the resulting nodal loads and prescribed displacements in each Node
    % object.
    % obs.: nodalLoadCase(:,i) = [ fx        
    %                              fy
    %                              fz     (nodalLoadCase is a matrix, each
    %                              mx      column refers to a specific load
    %                              my      case)
    %                              mz
    %                              dx
    %                              dy
    %                              dz
    %                              rx
    %                              ry
    %                              rz ]
    %
    % obs.2: loadComb is a matrix that holds combination factors for load
    % case combinations, where each column refers to a specific load comb.
    
    for n = 1:drv.nnp
        if isempty(drv.nodes(n).nodalLoadCase) == 0
            allLogic = all(drv.nodes(n).nodalLoadCase(1:6,:) == 0);
            if all(allLogic == 1) == 1
                drv.nodes(n).nodalLoad = []; 
            else  % if there are nodal loads, ,multiply them by their 
                  % proper comb factors, sum the results and set them to 
                  % nodalLoad.
                drv.nodes(n).nodalLoad(1) = drv.nodes(n).nodalLoadCase(1,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                drv.nodes(n).nodalLoad(2) = drv.nodes(n).nodalLoadCase(2,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                drv.nodes(n).nodalLoad(3) = drv.nodes(n).nodalLoadCase(3,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                drv.nodes(n).nodalLoad(4) = drv.nodes(n).nodalLoadCase(4,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                drv.nodes(n).nodalLoad(5) = drv.nodes(n).nodalLoadCase(5,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                drv.nodes(n).nodalLoad(6) = drv.nodes(n).nodalLoadCase(6,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                if all(drv.nodes(n).nodalLoad == 0) == 1
                    drv.nodes(n).nodalLoad = [];
                end    
            end
            if size(drv.nodes(n).nodalLoadCase,1) <= 6
                allLogic = 1;
            else
                allLogic = all(drv.nodes(n).nodalLoadCase(7:12,:) == 0);
            end
            if all(allLogic == 1) == 1
                drv.nodes(n).prescDispl = [];
            else  % if there are prescribed displacements, ,multiply them 
                  % by their proper comb factors, sum the results and set  
                  % them to prescDispl.
                drv.nodes(n).prescDispl(1) = drv.nodes(n).nodalLoadCase(7,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                drv.nodes(n).prescDispl(2) = drv.nodes(n).nodalLoadCase(8,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                drv.nodes(n).prescDispl(3) = drv.nodes(n).nodalLoadCase(9,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                drv.nodes(n).prescDispl(4) = drv.nodes(n).nodalLoadCase(10,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                drv.nodes(n).prescDispl(5) = drv.nodes(n).nodalLoadCase(11,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                drv.nodes(n).prescDispl(6) = drv.nodes(n).nodalLoadCase(12,:) * drv.loadComb(1:size(drv.nodes(n).nodalLoadCase,2),lc-nlc);
                if all(drv.nodes(n).prescDispl == 0) == 1
                    drv.nodes(n).prescDispl = [];
                end
            end
        else
            drv.nodes(n).nodalLoad = []; 
            drv.nodes(n).prescDispl = [];
        end
    end    

    % Consider load cases factors for the selected combination and allocate
    % the resulting element loads and prescribed displacements in each
    % Lelem object.
    % obs.: elemLoadCase(:,i) = [  unifDir
    %                                qx
    %                                qy
    %                                qz        (elemLoadCase is a matrix, 
    %                             linearDir     each column refers to a  
    %                                qx1        specific load case)
    %                                qy1
    %                                qz1
    %                                qx2
    %                                qy2
    %                                qz2
    %                                dtx
    %                                dty
    %                                dtz   ]
    %
    % obs.2: loadComb is a matrix that holds combination factors for load
    % case combinations, where each column refers to a specific load comb.
    
    for e = 1:drv.nel
        if isempty(drv.elems(e).load.elemLoadCase) == 0
            % Initilize uniform loads
            drv.elems(e).load.uniformGbl = [];    
            drv.elems(e).load.uniformLcl = [];
            allLogic = all(drv.elems(e).load.elemLoadCase(2:4,:) == 0);
            if all(allLogic == 1) == 1
                drv.elems(e).load.uniformDir = 0;
            else  % if there are uniform loads, set their local and global
                  % components and consider their comb factors
                unifLoad = zeros(size(drv.elems(e).load.elemLoadCase,2),3);
                for i = 1:size(drv.elems(e).load.elemLoadCase,2)
                    unifLoad(i,:) = drv.loadComb(i,lc-nlc) * drv.elems(e).load.elemLoadCase(2:4,i);
                    if all(unifLoad(i,:) == 0) == 0
                        drv.elems(e).load.setUnifLoad(unifLoad(i,:),drv.elems(e).load.elemLoadCase(1,i));
                    end    
                end
            end  
            
            % Initilize linear loads
            drv.elems(e).load.linearGbl = [];
            drv.elems(e).load.linearLcl = [];
            if size(drv.elems(e).load.elemLoadCase,1) <= 5
                allLogic = 1;
            else
                allLogic = all(drv.elems(e).load.elemLoadCase(6:11,:) == 0);
            end
            if all(allLogic == 1) == 1
                drv.elems(e).load.linearDir = 0;
            else  % if there are linear loads, set their local and global
                  % components and consider their comb factors    
                linLoad = zeros(size(drv.elems(e).load.elemLoadCase,2),6);
                for i = 1:size(drv.elems(e).load.elemLoadCase,2)
                    linLoad(i,:) = drv.loadComb(i,lc-nlc) * drv.elems(e).load.elemLoadCase(6:11,i);
                    if all(linLoad(i,:) == 0) == 0
                        drv.elems(e).load.setLinearLoad(linLoad(i,:),drv.elems(e).load.elemLoadCase(5,i));
                    end    
                end
            end
            % Set thermal loads
            if size(drv.elems(e).load.elemLoadCase,1) <= 11
                drv.elems(e).load.tempVar_X = 0;
                drv.elems(e).load.tempVar_Y = 0;  
                drv.elems(e).load.tempVar_Z = 0; 
            else
                drv.elems(e).load.tempVar_X = drv.elems(e).load.elemLoadCase(12,:) * drv.loadComb(1:size(drv.elems(e).load.elemLoadCase,2),lc-nlc);
                drv.elems(e).load.tempVar_Y = drv.elems(e).load.elemLoadCase(13,:) * drv.loadComb(1:size(drv.elems(e).load.elemLoadCase,2),lc-nlc);  
                drv.elems(e).load.tempVar_Z = drv.elems(e).load.elemLoadCase(14,:) * drv.loadComb(1:size(drv.elems(e).load.elemLoadCase,2),lc-nlc);
            end
        else
            drv.elems(e).load.uniformDir = 0;      
            drv.elems(e).load.uniformGbl = [];    
            drv.elems(e).load.uniformLcl = [];    
            drv.elems(e).load.linearDir = 0;       
            drv.elems(e).load.linearGbl = [];
            drv.elems(e).load.linearLcl = [];
            drv.elems(e).load.tempVar_X = 0;
            drv.elems(e).load.tempVar_Y = 0;  
            drv.elems(e).load.tempVar_Z = 0; 
        end
    end
end
end