%% Redraw function
%
% This file contains a function that works as a switch to call other
% functions. Redraws requested objects of the model.
%
%% Authors
% Luiz Fernando Martha, Rafael Lopez Rangel and Pedro Cortez Lopes
%
%% ------------------------------------------------------------------------
% Draw model based on the selected option in popupmenu_Results
% INPUT: 
% * mdata -> handle to GUI_Main
% * toBeRedrawn -> string identifier for what needs to be redrawn
function redraw(mdata,toBeRedrawn,fctnArgIn)
if nargin == 1
    resVal = get(mdata.popupmenu_Results,'Value');
    resStr = get(mdata.popupmenu_Results,'String');
    result = char(resStr(resVal,:));
    toBeRedrawn = result;
    fctnArgIn = [];
elseif nargin == 2
    fctnArgIn = [];
end

% Make sure that modifications will be made on canvas
axes(mdata.axes_Canvas);

switch toBeRedrawn
    case 'Model'
        drawModel(mdata);
    case 'Deformation'
        drawDeformation(mdata);
    case 'Axial Force'
        drawAxialForce(mdata);
    case 'Torsion Moment'
        drawTorsionMoment(mdata);
    case 'Shear Force Y'
        drawShearForceY(mdata);
    case 'Shear Force Z'
        drawShearForceZ(mdata);
    case 'Bending Moment Y'
        drawBendingMomentY(mdata);
    case 'Bending Moment Z'
        drawBendingMomentZ(mdata);
    case 'Nodes'
        if isempty(fctnArgIn)
            drawNodes(mdata);
        else
            drawNodes(mdata,fctnArgIn);
        end
    case 'Elements'
        drawElems(mdata);
    case 'Loads'
        drawLoads(mdata);
    case 'Nodal Loads'
        drawNodalLoads(mdata);
    case 'Element Loads'
        drawElemLoads(mdata);
    case 'Units'
        textUnits(mdata);
    case 'Decimal Precision'
        textDecPrec(mdata);
end

% Update mouse draw size flag
draw = getappdata(0,'draw');
mouse = getappdata(0,'mouse');
mouse.sizeFlag = draw.size;
setappdata(0,'mouse',mouse)
end

%% ------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws model on canvas
function drawModel(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Get analysis model
anm = get(mdata.popupmenu_Anm,'value');

% Clean canvas
axes(mdata.axes_Canvas);
cla reset

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Draw model
draw.setSize();
draw.elemLoadsScaleFactor();
draw.model();
draw.nodalLoads();
draw.elemLoads();
draw.thermalLoads();
draw.nodalPrescDispl();
if getappdata(0,'vis') == 1
    if (strcmp(get(mdata.nodeIDButton,'Checked'),'on') == 1) % Check if nodes ID is on
        draw.nodeID();
    end
    if (strcmp(get(mdata.elemIDButton,'Checked'),'on') == 1) % Check if elements ID is on
        draw.elementID();
    end
    if (strcmp(get(mdata.orientationButton,'Checked'),'on') == 1) % Check if elements orientation is on
        draw.elementOrientation();
    end
end
if get(mdata.checkbox_Reactions,'Value') == 1 % Turn reaction values on
    draw.reactions();
else
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end
axis equal
draw.setLimits();

% Set axes labels positions
if anm >= 4 || (anm == 3 && strcmp(get(mdata.togglebutton_2DView,'state'),'off'))
offset = ((max(xlim)-min(xlim))+(max(ylim)-min(ylim))+(max(zlim)-min(zlim)))/50;
xlabel('X', 'Position', [mean(xlim), min(ylim) - offset/3, min(zlim) - offset/2]);
ylabel('Y', 'Position', [min(xlim) - offset, mean(ylim), min(zlim) - offset/2]);
zlabel('Z', 'Position', [min(xlim) - offset, max(ylim) + offset, mean(zlim)], 'Rotation', pi/2);
end

% Adjust axes
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
elseif anm == 3 && strcmp(get(mdata.togglebutton_2DView,'state'),'on')
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Turn grid on/off
if strcmp(get(mdata.gridButton,'Checked'),'on') == 1
    grid on
else
    grid off
end

% Turn ruler on/off
if strcmp(get(mdata.rulerButton,'Checked'),'on') == 1
    mdata.axes_Canvas.XAxis.Visible = 'on';
    mdata.axes_Canvas.YAxis.Visible = 'on';
    mdata.axes_Canvas.ZAxis.Visible = 'on';
else
    mdata.axes_Canvas.XAxis.Visible = 'off';
    mdata.axes_Canvas.YAxis.Visible = 'off';
    mdata.axes_Canvas.ZAxis.Visible = 'off';
end

% Set plane of visualization
if (anm == 1) || (anm == 2)
    view(2)
    set(mdata.planeButton,'Enable','off');
elseif anm == 3 && strcmp(get(mdata.togglebutton_2DView,'state'),'on')
    view(2)
    set(mdata.planeButton,'Enable','on');
    set(mdata.plane3D,'Checked','off');
    set(mdata.planeXY,'Checked','on');
    set(mdata.planeXZ,'Checked','off','enable','off');
    set(mdata.planeYZ,'Checked','off','enable','off');
else
    view(3)
    set(mdata.planeButton,'Enable','on');
    set(mdata.plane3D,'Checked','on');
    set(mdata.planeXY,'Checked','off');
    set(mdata.planeXZ,'Checked','off');
    set(mdata.planeYZ,'Checked','off');
end

% Reinitialize object for mouse events and save it in root
mouse = getappdata(0,'mouse');
mouse.originalXLim = get(mdata.axes_Canvas,'XLim');
mouse.originalYLim = get(mdata.axes_Canvas,'YLim');
if anm == 3 && strcmp(get(mdata.togglebutton_2DView,'state'),'off')
    mouse.originalZLim = get(mdata.axes_Canvas,'ZLim');
elseif (anm == 4) || (anm == 5)
    mouse.originalZLim = get(mdata.axes_Canvas,'ZLim');
end
mouse.sizeFlag = draw.size;
setappdata(0,'mouse',mouse);
setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws deformation on canvas
function drawDeformation(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clean model
clearLoadAndResults()

% Enable scale indicator
set(mdata.edit_Scale,'Visible','on','Enable','on');

% Calculate scale value
dsf = getappdata(0,'deform_sf');
if isnan(str2double(get(mdata.edit_Scale,'String'))) == 1
    scale = dsf * get(mdata.slider_Scale,'Value');
    set(mdata.edit_Scale,'String',num2str(scale,3))
else    
    scale = str2double(get(mdata.edit_Scale,'String'));    
end

% Draw deformed configuration
draw.deformConfig(scale);
anm = get(mdata.popupmenu_Anm,'value');
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws Axial force diagram on canvas
function drawAxialForce(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clean model
clearLoadAndResults()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Calculate scale value
asf = getappdata(0,'axial_sf');
scale = asf * get(mdata.slider_Scale,'Value');

% Draw axial force diagram
draw.axialForce(scale);
anm = get(mdata.popupmenu_Anm,'value');
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws torsion moment diagram on canvas
function drawTorsionMoment(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clean model
clearLoadAndResults()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Draw torsion moment diagram
draw.torsionMoment();
anm = get(mdata.popupmenu_Anm,'value');
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws shear force Y diagram on canvas
function drawShearForceY(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clean model
clearLoadAndResults()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Calculate scale value
ssf = getappdata(0,'shearXY_sf');
scale = ssf * get(mdata.slider_Scale,'Value');

% Draw shear force diagram
draw.shearForce_XY(scale);
anm = get(mdata.popupmenu_Anm,'value');
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws shear force Z diagram on canvas
function drawShearForceZ(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clean model
clearLoadAndResults()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Calculate scale value
ssf = getappdata(0,'shearXZ_sf');
scale = ssf * get(mdata.slider_Scale,'Value');

% Draw shear force diagram
draw.shearForce_XZ(scale);
anm = get(mdata.popupmenu_Anm,'value');
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws bending moment Y diagram on canvas
function drawBendingMomentY(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clean model
clearLoadAndResults()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Calculate scale value
bsf = getappdata(0,'bendingXZ_sf');
scale = bsf * get(mdata.slider_Scale,'Value');

% Draw bending moment diagram
draw.bendingMoment_XZ(scale);
anm = get(mdata.popupmenu_Anm,'value');
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws bending moment Z diagram on canvas
function drawBendingMomentZ(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clean model
clearLoadAndResults()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Calculate scale value
bsf = getappdata(0,'bendingXY_sf');
scale = bsf * get(mdata.slider_Scale,'Value');

% Draw bending moment diagram
draw.bendingMoment_XY(scale);
anm = get(mdata.popupmenu_Anm,'value');
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws nodes and supports on canvas
function drawNodes(mdata,setLimitsFlag)
% Flag for need to reset axis limits
if nargin == 1
    setLimitsFlag = true;
end

% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Get 2D axis propotions
ax = mdata.axes_Canvas;
axProp_XY = diff(ax.XLim)/diff(ax.YLim);

% Clean canvas
clearNodesSpprts()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Draw nodes
draw.setSize();
draw.nodes();
if getappdata(0,'vis') == 1
    if (strcmp(get(mdata.nodeIDButton,'Checked'),'on') == 1) % Check if nodes ID is on
        draw.nodeID();
    end
    if (strcmp(get(mdata.gridButton,'Checked'),'on') == 1) % Check if grid is on
        grid on
    else
        grid off
    end
    if drv.nnp == 0 || drv.nnp == 1
        switch get(mdata.rulerButton,'Checked')
            case 'off'
                ax.XAxis.Visible = 'off';
                ax.YAxis.Visible = 'off';
                ax.ZAxis.Visible = 'off';
            case 'on'
                ax.XAxis.Visible = 'on';
                ax.YAxis.Visible = 'on';
                ax.ZAxis.Visible = 'on';
        end
    end
end
if drv.anm.analysis_type <= 1 % TRUSS_2D / FRAME_2D
    if setLimitsFlag == true
        draw.setLimits(axProp_XY);
    end
elseif drv.anm.analysis_type == 2 && strcmp(get(mdata.togglebutton_2DView,'state'),'on')  % GRILLAGE_2D
    if setLimitsFlag == true
        draw.setLimits(axProp_XY);
    end
elseif drv.anm.analysis_type == 2 % GRILLAGE_3D
    if setLimitsFlag == true
        axis equal
        draw.setLimits();
    end
else % TRUSS_3D / FRAME_3D
    if setLimitsFlag == true
        axis equal
        draw.setLimits();
    end
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

% Reinitialize object for mouse events and save it in root
mouse = getappdata(0,'mouse');
if setLimitsFlag == true
    mouse.originalXLim = get(mdata.axes_Canvas,'XLim');
    mouse.originalYLim = get(mdata.axes_Canvas,'YLim');
    anm = get(mdata.popupmenu_Anm,'value');
    if anm == 3 && strcmp(get(mdata.togglebutton_2DView,'state'),'off')
        mouse.originalZLim = get(mdata.axes_Canvas,'ZLim');
    elseif (anm == 4) || (anm == 5)
        mouse.originalZLim = get(mdata.axes_Canvas,'ZLim');
    end
end
mouse.sizeFlag = draw.size;
setappdata(0,'mouse',mouse);
setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws elements on canvas
function drawElems(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clean canvas
clearElems()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Draw model
draw.elements();
if getappdata(0,'vis') == 1
    if (strcmp(get(mdata.elemIDButton,'Checked'),'on') == 1) % Check if elements ID is on
        draw.elementID();
    end
    if (strcmp(get(mdata.orientationButton,'Checked'),'on') == 1) % Check if elements orientation is on
        draw.elementOrientation();
    end
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws loads on canvas
function drawLoads(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clear model
clearLoadAndResults()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Draw model
draw.elemLoadsScaleFactor();
draw.nodalLoads();
draw.elemLoads();
draw.thermalLoads();
draw.nodalPrescDispl();
anm = get(mdata.popupmenu_Anm,'value');
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws loads on canvas
function drawNodalLoads(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clear nodal loads
delete(findobj('tag','drawNodalLoads'))
delete(findobj('tag','textNodalLoads'))
delete(findobj('tag','textNodalMoments'))
delete(findobj('tag','drawPrescDispl'))
delete(findobj('tag','textPrescDispl'))
delete(findobj('tag','textPrescRot'))
clearResults()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Draw model
draw.nodalLoads();
draw.nodalPrescDispl();
anm = get(mdata.popupmenu_Anm,'value');
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Draws loads on canvas
function drawElemLoads(mdata)
% Get objects from root
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
draw.drv = drv;

% Clear elem loads
delete(findobj('tag','drawElemLoads'))
delete(findobj('tag','textElemLoads'))
delete(findobj('tag','drawThermalLoads'))
delete(findobj('tag','textThermalLoads'))
clearResults()

% Disable scale indicator
set(mdata.edit_Scale,'Visible','off','Enable','off');

% Draw model
draw.elemLoadsScaleFactor();
draw.elemLoads();
draw.thermalLoads();
anm = get(mdata.popupmenu_Anm,'value');
if anm == 1 || anm == 2
    ax = mdata.axes_Canvas;
    ax.Clipping = 'on';
else
    ax = mdata.axes_Canvas;
    ax.Clipping = 'off';
end

% Make sure not to display reactions if checkbox is unmarked
if get(mdata.checkbox_Reactions,'Value') == 0
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

setappdata(0,'drv',drv);
setappdata(0,'draw',draw);
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Writes/Deletes units from texts on canvas
function textUnits(mdata)
% Get units button state - on/off
check = get(mdata.unitsButton,'Checked');

% Get texts
springStiff = findobj('tag','textSprings');
rotSpringStiff = findobj('tag','textRotSprings');
nodalLoads = findobj('tag','textNodalLoads');
nodalMoments = findobj('tag','textNodalMoments');
prescDispl = findobj('tag','textPrescDispl');
prescRot = findobj('tag','textPrescRot');
elemLoads = findobj('tag','textElemLoads');
thermalLoads = findobj('tag','textThermalLoads');
axialForce = findobj('tag','textAxialForceDiagram');
torsionMoment = findobj('tag','textTorsionDiagram');
shearForceXY = findobj('tag','textShearForceXYDiagram');
shearForceXZ = findobj('tag','textShearForceXZDiagram');
bendMomentXY = findobj('tag','textBendMomentXYDiagram');
bendMomentXZ = findobj('tag','textBendMomentXZDiagram');
reactions = findobj('tag','textForceReactions');
momentReactions = findobj('tag','textMomentReactions');

% Update texts accordingly to current unitsButton 'Checked' state - on/off
switch check
    case 'on' % Units button has been turned on
        for i = 1:size(springStiff,1)
            springStiff(i).String = strcat(springStiff(i).String, ' kN/m');
        end
        for i = 1:size(rotSpringStiff,1)
            rotSpringStiff(i).String = strcat(rotSpringStiff(i).String, ' kNm/rad');
        end
        for i = 1:size(nodalLoads,1)
            nodalLoads(i).String = strcat(nodalLoads(i).String, ' kN');
        end
        for i = 1:size(nodalMoments,1)
            nodalMoments(i).String = strcat(nodalMoments(i).String, ' kNm');
        end
        for i = 1:size(prescDispl,1)
            prescDispl(i).String = strcat(prescDispl(i).String, ' mm');
        end
        for i = 1:size(prescRot,1)
            prescRot(i).String = strcat(prescRot(i).String, ' rad');
        end
        for i = 1:size(elemLoads,1)
            elemLoads(i).String = strcat(elemLoads(i).String, ' kN/m');
        end
        for i = 1:size(thermalLoads,1)
            thermalLoads(i).String = strcat(thermalLoads(i).String, ' C');
        end
        for i = 1:size(axialForce,1)
            axialForce(i).String = strcat(axialForce(i).String, ' kN');
        end
        for i = 1:size(torsionMoment,1)
            torsionMoment(i).String = strcat(torsionMoment(i).String, ' kNm');
        end
        for i = 1:size(shearForceXY,1)
            shearForceXY(i).String = strcat(shearForceXY(i).String, ' kN');
        end
        for i = 1:size(shearForceXZ,1)
            shearForceXZ(i).String = strcat(shearForceXZ(i).String, ' kN');
        end
        for i = 1:size(bendMomentXY,1)
            bendMomentXY(i).String = strcat(bendMomentXY(i).String, ' kNm');
        end
        for i = 1:size(bendMomentXZ,1)
            bendMomentXZ(i).String = strcat(bendMomentXZ(i).String, ' kNm');
        end
        for i = 1:size(reactions,1)
            reactions(i).String = strcat(reactions(i).String, ' kN');
        end
        for i = 1:size(momentReactions,1)
            momentReactions(i).String = strcat(momentReactions(i).String, ' kNm');
        end
        
    case 'off' % Units button has been turned off
        for i = 1:size(springStiff,1)
            springStiff(i).String = springStiff(i).String(1:end-5);
        end
        for i = 1:size(rotSpringStiff,1)
            rotSpringStiff(i).String = rotSpringStiff(i).String(1:end-8);
        end
        for i = 1:size(nodalLoads,1)
            nodalLoads(i).String = nodalLoads(i).String(1:end-3);
        end
        for i = 1:size(nodalMoments,1)
            nodalMoments(i).String = nodalMoments(i).String(1:end-4);
        end
        for i = 1:size(prescDispl,1)
            prescDispl(i).String = prescDispl(i).String(1:end-3);
        end
        for i = 1:size(prescRot,1)
            prescRot(i).String = prescRot(i).String(1:end-4);
        end
        for i = 1:size(elemLoads,1)
            elemLoads(i).String = elemLoads(i).String(1:end-5);
        end
        for i = 1:size(thermalLoads,1)
            thermalLoads(i).String = thermalLoads(i).String(1:end-3);
        end
        for i = 1:size(axialForce,1)
            axialForce(i).String = axialForce(i).String(1:end-3);
        end
        for i = 1:size(torsionMoment,1)
            torsionMoment(i).String = torsionMoment(i).String(1:end-4);
        end
        for i = 1:size(shearForceXY,1)
            shearForceXY(i).String = shearForceXY(i).String(1:end-3);
        end
        for i = 1:size(shearForceXZ,1)
            shearForceXZ(i).String = shearForceXZ(i).String(1:end-3);
        end
        for i = 1:size(bendMomentXY,1)
            bendMomentXY(i).String = bendMomentXY(i).String(1:end-4);
        end
        for i = 1:size(bendMomentXZ,1)
            bendMomentXZ(i).String = bendMomentXZ(i).String(1:end-4);
        end
        for i = 1:size(reactions,1)
            reactions(i).String = reactions(i).String(1:end-3);
        end
        for i = 1:size(momentReactions,1)
            momentReactions(i).String = momentReactions(i).String(1:end-4);
        end
end
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Resets all decimal precision of texts on canvas
function textDecPrec(mdata)
% Get decimal precision
dp = getappdata(0,'decPrec');

% Get texts
springStiff = findobj('tag','textSprings');
rotSpringStiff = findobj('tag','textRotSprings');
nodalLoads = findobj('tag','textNodalLoads');
nodalMoments = findobj('tag','textNodalMoments');
prescDispl = findobj('tag','textPrescDispl');
prescRot = findobj('tag','textPrescRot');
elemLoads = findobj('tag','textElemLoads');
thermalLoads = findobj('tag','textThermalLoads');
axialForce = findobj('tag','textAxialForceDiagram');
torsionMoment = findobj('tag','textTorsionDiagram');
shearForceXY = findobj('tag','textShearForceXYDiagram');
shearForceXZ = findobj('tag','textShearForceXZDiagram');
bendMomentXY = findobj('tag','textBendMomentXYDiagram');
bendMomentXZ = findobj('tag','textBendMomentXZDiagram');
reactions = findobj('tag','textForceReactions');
momentReactions = findobj('tag','textMomentReactions');

% Allocate all texts in vector of handles to text objects
springTexts = vertcat(springStiff,rotSpringStiff);
allTexts = vertcat(nodalLoads,nodalMoments,prescDispl,prescRot,elemLoads,...
                   axialForce,torsionMoment,shearForceXY,shearForceXZ,...
                   bendMomentXY,bendMomentXZ,reactions,momentReactions);

% Reset spring texts
for i = 1:size(springTexts,1)
    if springTexts(i).UserData >= 1000
        springTexts(i).String = sprintf('%.*e',dp,springTexts(i).UserData);
    else
        springTexts(i).String = sprintf('%.*f',dp,springTexts(i).UserData);
    end
end

% Reset thermal load texts
for i = 1:size(thermalLoads,1)
    str = thermalLoads(i).UserData{1};
    dt = thermalLoads(i).UserData{2};
    thermalLoads(i).String = sprintf('%s%.*f',str,dp,dt);
end

% Reset all other texts
for i = 1:size(allTexts,1)
    allTexts(i).String = sprintf('%.*f',dp,allTexts(i).UserData);
end

% Set units to texts, if necessary
if strcmp(get(mdata.unitsButton,'Checked'),'on') == true
    textUnits(mdata);
end
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Clears nodes and supports
function clearNodesSpprts()
    delete(findobj('tag','drawNodes'))
    delete(findobj('tag','drawSupports'))
    delete(findobj('tag','textSprings'))
    delete(findobj('tag','textRotSprings'))
    delete(findobj('tag','textNodeID'))
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Clears elements
function clearElems()
    delete(findobj('tag','drawElements'))
    delete(findobj('tag','textElemID'))
    delete(findobj('tag','drawElemOrient'))
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Clears loads/deformed configuration/diagrams
function clearLoadAndResults()
    delete(findobj('tag','drawNodalLoads'))
    delete(findobj('tag','textNodalLoads'))
    delete(findobj('tag','textNodalMoments'))
    delete(findobj('tag','drawPrescDispl'))
    delete(findobj('tag','textPrescDispl'))
    delete(findobj('tag','textPrescRot'))
    delete(findobj('tag','drawElemLoads'))
    delete(findobj('tag','textElemLoads'))
    delete(findobj('tag','drawThermalLoads'))
    delete(findobj('tag','textThermalLoads'))
    clearResults()
end

%--------------------------------------------------------------------------
% Auxiliary function to the redraw function
% Clears deformed configuration/diagrams
function clearResults()
    delete(findobj('tag','drawDeformConfig'))
    delete(findobj('tag','drawAxialForceDiagram'))
    delete(findobj('tag','textAxialForceDiagram'))
    delete(findobj('tag','textTorsionDiagram'))
    delete(findobj('tag','drawShearForceXYDiagram'))
    delete(findobj('tag','textShearForceXYDiagram'))
    delete(findobj('tag','drawShearForceXZDiagram'))
    delete(findobj('tag','textShearForceXZDiagram'))
    delete(findobj('tag','drawBendMomentXYDiagram'))
    delete(findobj('tag','textBendMomentXYDiagram'))
    delete(findobj('tag','drawBendMomentXZDiagram'))
    delete(findobj('tag','textBendMomentXZDiagram'))
end
