%% Draw_Truss2D class
%
% This is a sub-class, in the Object Oriented Programming (OOP) paradigm,
% of super-class <draw.html *Draw*> in the <main.html LESM (Linear Elements
% Structure Model)> program. This sub-class implements abstract methods,
% defined in super-class *Draw*, that deal with 2D truss analysis model
% of linear structure elements.
%
%% Authors
% Luiz Fernando Martha, Rafael Lopez Rangel and Pedro Cortez Lopes
%
%% Class definition
classdef Draw_Truss2D < Draw
    %% Constructor method
    methods
        %------------------------------------------------------------------
        function draw = Draw_Truss2D(drv)
            draw = draw@Draw(0,90);
            
            if (nargin > 0)
                draw.drv = drv;
            end
        end
    end
    
    %% Public methods
    % Implementation of the abstract methods declared in super-class <draw.html *Draw*>.
    methods
        %------------------------------------------------------------------
        % Sets viewpoint position and computes drawing size parameter
        % according to model dimensions.
        function draw = setSize(draw)
            
            x = zeros(draw.drv.nnp);
            y = zeros(draw.drv.nnp);
            z = zeros(draw.drv.nnp);
            
            for n = 1:draw.drv.nnp
                x(n) = draw.drv.nodes(n).coord(1);
                y(n) = draw.drv.nodes(n).coord(2);
                z(n) = draw.drv.nodes(n).coord(3);
            end
            
            xmin = min(x);
            xmax = max(x);
            ymin = min(y);
            ymax = max(y);
            zmin = min(z);
            zmax = max(z);
            
            dx = xmax - xmin;
            dy = ymax - ymin;
            dz = zmax - zmin;
            sz = max([dx,dy,dz]);
            if isempty(sz)
                sz = 0;
            end
            
            if sz == 0
                draw.size = 5;
            else
                draw.size = sz;
            end
        end
        
        %------------------------------------------------------------------
        % Sets axes limits according to model dimensions.
        function draw = setLimits(draw,axProp)
            if nargin == 2
                setBothAxis = true;
            else
                setBothAxis = false;
            end
            if draw.drv.nnp > 0
                x = zeros(draw.drv.nnp,1);
                y = zeros(draw.drv.nnp,1);
                
                for n = 1:draw.drv.nnp
                    x(n) = draw.drv.nodes(n).coord(1);
                    y(n) = draw.drv.nodes(n).coord(2);
                end
                
                xmin = min(x);
                xmax = max(x);
                ymin = min(y);
                ymax = max(y);
                
                dx = xmax - xmin;
                dy = ymax - ymin;
                
                if (dx == 0) && (dy == 0)
                    xlim([x - 5, x + 5])
                    if setBothAxis == true
                        ylim([y - 5/axProp, y + 5/axProp])
                    end
                elseif dx > dy
                    xlim([xmin - dx/1.7, xmax + dx/1.7])
                    if setBothAxis == true
                        aux_xlim = [xmin - dx/1.7, xmax + dx/1.7];
                        xlength = abs(diff(aux_xlim));
                        ylength = xlength/axProp;
                        ymean = mean([ymin ymax]);
                        ylim([ymean - ylength/2, ymean + ylength/2])
                    end
                else
                    if setBothAxis == true
                        aux_ylim = [ymin - dy/4, ymax + dy/4];
                        ylength = abs(diff(aux_ylim));
                        xlength = ylength * axProp;
                        xmean = mean([xmin xmax]);
                        xlim([xmean - xlength/2, xmean + xlength/2])
                    end
                    ylim([ymin - dy/4, ymax + dy/4])
                end
                
            else
                xlim([-5,5])
                if setBothAxis == true
                    ylim([-5/axProp, 5/axProp])
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws structure model with nodes, elements, supports and  hinges.
        function draw = model(draw)
            draw.nodes();
            hold on
            draw.elements();
        end
        
        %------------------------------------------------------------------
        % Draws elements with hinged ends.
        function draw = elements(draw)
            % Parameters
            r = draw.size/125;  % hinge symbol radius
            clr = [0,0,0];      % element color
            
            for e = 1:draw.drv.nel
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                
                % Get element orientation angle cosine with X and Y axes
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
                
                % Set element end coordinates
                xi = x1 + r * cx;
                yi = y1 + r * cy;
                xf = x2 - r * cx;
                yf = y2 - r * cy;
                
                % Connect element end coordinates
                X = [xi, xf];
                Y = [yi, yf];
                line(X, Y, 'Color', clr, 'tag', 'drawElements');
            end
        end
        
        %------------------------------------------------------------------
        % Draws nodal marks with support conditions.
        function draw = nodes(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Get flag for support visualization option status
            drawSupports =  get(mdata.viewSupportsButton,'Checked');
            
            include_constants;
            % Parameters
            r = draw.size/125;     % hinge symbol radius
            th = draw.size/35;     % translation constraint symbol (triangle height)
            tb = draw.size/50;     % translation constraint symbol (triangle base)
            sh = draw.size/20;     % displacement spring symbol (spring height)
            nclr = [0,0,0];        % node (hinge) color
            sclr = [0.6,0.6,0.6];  % support color
            sprclr = [0.6,0,0.4];  % spring color
            dc = getappdata(0,'decPrec');  % decimal precision
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
                
                % Draw hinge as a nodal point
                draw.circle(x, y, r, nclr,'drawNodes');
                hold on
                
                % Draw support conditions:
                % Check if there is a horizontal translation constraint
                if draw.drv.nodes(n).ebc(1) == FIXED_DOF && strcmp(drawSupports,'on')
                    draw.triangle(x - r, y, th, tb, pi, sclr,'drawSupports')
                    hold on
                end
                
                % Check if there is a vertical translation constraint
                if draw.drv.nodes(n).ebc(2) == FIXED_DOF && strcmp(drawSupports,'on')
                    draw.triangle(x, y - r, th, tb, 3 * pi/2, sclr,'drawSupports')
                    hold on
                end
                
                % Check if there is a displacement spring in the direction
                % of axis X and draw spring
                if draw.drv.nodes(n).ebc(1) == SPRING_DOF && strcmp(drawSupports,'on')  
                    draw.springX_2D(x - r, y, sh, sprclr,'drawSupports')
                    hold on
                    
                    % Get spring stiffness
                    kx = draw.drv.nodes(n).springStiff(1);
                    % Write spring stiffness value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        if kx >= 1000
                        value = sprintf('%.*e kN/m',dc,kx);
                        else
                        value = sprintf('%.*f kN/m',dc,kx);
                        end
                    else
                        if kx >= 1000
                        value = sprintf('%.*e',dc,kx);
                        else
                        value = sprintf('%.*f',dc,kx);
                        end
                    end
                    
                    text(x - 1.1*sh - r, y + sh*0.35, value, 'Color', sprclr,'Fontsize',8.5,'tag','textSprings','UserData',kx);
                end
                
                % Check if there is a displacement spring in the direction
                % of axis Y and draw spring
                if draw.drv.nodes(n).ebc(2) == SPRING_DOF && strcmp(drawSupports,'on')
                    draw.springY_2D(x, y - r, sh, sprclr,'drawSupports')
                    hold on

                    % Get spring stiffness
                    ky = draw.drv.nodes(n).springStiff(2);
                    % Write spring stiffness value
                   if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        if ky >= 1000
                        value = sprintf('%.*e kN/m',dc,ky);
                        else
                        value = sprintf('%.*f kN/m',dc,ky);
                        end
                    else
                        if ky >= 1000
                        value = sprintf('%.*e',dc,ky);
                        else
                        value = sprintf('%.*f',dc,ky);
                        end
                    end
                    
                    text(x + sh*0.25, y - sh - r, value, 'Color', sprclr,'Fontsize',8.5,'tag','textSprings','UserData',ky);
                end
            end
        end
        
        %------------------------------------------------------------------
        % Computes element loads scale factor.
        function scl = elemLoadsScaleFactor(draw)
            max_load = zeros(1,draw.drv.nel);
            
            for e = 1:draw.drv.nel
                % Initialize load values on element ends
                qi = 0;
                qf = 0;
                
                % Add uniform load contribtuion
                if isempty(draw.drv.elems(e).load.uniformLcl) == 0
                    qy = draw.drv.elems(e).load.uniformLcl(2);
                    qi = qi + qy;
                    qf = qf + qy;
                end
                
                % Add linear load contribtuion
                if isempty(draw.drv.elems(e).load.linearLcl) == 0
                    qyi = draw.drv.elems(e).load.linearLcl(2);
                    qyf = draw.drv.elems(e).load.linearLcl(5);
                    qi = qi + qyi;
                    qf = qf + qyf;
                end
                
                % Get maximum load value on current element
                max_load(e) = max(abs(qi),abs(qf));
            end
            
            % Get maximum load value on model
            max_val = max(max_load);
            
            % Calculate scale factor
            if max_val ~= 0
                scl = draw.size/(12*max_val);
            else
                scl = 0;
            end
            
            setappdata(0,'load_sf',scl);
            setappdata(0,'max_val',max_val);
        end
        
        %------------------------------------------------------------------
        % Draws element distributed loads (uniform and linear).
        function draw = elemLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if elem loads visualization is on
            if strcmp(get(mdata.viewDistribLoadsButton,'Checked'),'off')
                return
            end
            
            include_constants
            % Parameters
            r = draw.size/100;   % hinge symbol radius
            ah = draw.size/150;  % load symbol size (arrowhead height)
            ab = draw.size/150;  % load symbol size (arrowhead base)
            clr = [1,0,0];       % load symbol color
            scl = getappdata(0,'load_sf');                 % Load symbol scale
            max_val = getappdata(0,'max_val');
            rmappdata(0,'max_val')
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for e = 1:draw.drv.nel
                if ((isempty(draw.drv.elems(e).load.uniformGbl) == 0) &&...
                   (all(draw.drv.elems(e).load.uniformGbl == 0)==0)) ||...
                   ((isempty(draw.drv.elems(e).load.linearGbl) == 0) &&...
                   (all(draw.drv.elems(e).load.linearGbl == 0)==0))
               
                    % Get element length
                    L = draw.drv.elems(e).length;
                    
                    % Get element orientation angle
                    ang = acos(abs(draw.drv.elems(e).cosine_X));
                    
                    % Get element nodes IDs
                    n1 = draw.drv.elems(e).nodes(1).id;
                    n2 = draw.drv.elems(e).nodes(2).id;
                    
                    % Get nodal coordinates
                    x1 = draw.drv.nodes(n1).coord(1);
                    y1 = draw.drv.nodes(n1).coord(2);
                    x2 = draw.drv.nodes(n2).coord(1);
                    y2 = draw.drv.nodes(n2).coord(2);
                    
                    % Initialize load values on element ends
                    qxi = 0;
                    qyi = 0;
                    qxf = 0;
                    qyf = 0;
                    
                    % Add uniform load contribtuion
                    if isempty(draw.drv.elems(e).load.uniformLcl) == 0
                        qxi = qxi + draw.drv.elems(e).load.uniformLcl(1);
                        qyi = qyi + draw.drv.elems(e).load.uniformLcl(2);
                        qxf = qxi;
                        qyf = qyi;
                    end
                    
                    % Add linear load contribtuion
                    if isempty(draw.drv.elems(e).load.linearLcl) == 0
                        qxi = qxi + draw.drv.elems(e).load.linearLcl(1);
                        qyi = qyi + draw.drv.elems(e).load.linearLcl(2);
                        qxf = qxf + draw.drv.elems(e).load.linearLcl(4);
                        qyf = qyf + draw.drv.elems(e).load.linearLcl(5);
                    end
                    
                    % Axial load equation coefficients:
                    % p(x) = Ax + B
                    A = (qxf - qxi)/L;
                    B = qxi;
                    
                    % Transversal load equation coefficients:
                    % q(x) = Cx + D
                    C = (qyf - qyi)/L;
                    D = qyi;
                    
                    % Module of load parameters
                    Qyi = abs(scl * qyi);
                    Qyf = abs(scl * qyf);
                    
                    % Draw load symbol on a number cross-sections along
                    % element local axis X
                    step = (L-2*r) / round(20 * (L-2*r)/draw.size);
                    for x = r:step:(L-r) 
                        % Calculate load values on current cross-section
                        qx = A * x + B;
                        qy = C * x + D;
                        Qy = abs(scl * qy);
                        
                        % Calculate new element orientation angle and text
                        % rotation angle acording to element orientation
                        if (x1 <= x2) && (y1 <= y2)
                            alpha = ang;
                            rot = ang * 180/pi;
                        elseif (x1 >= x2) && (y1 >= y2)
                            alpha = ang + pi;
                            rot = ang * 180/pi;
                        elseif (x1 <= x2) && (y1 >= y2)
                            alpha = -ang;
                            rot = -ang * 180/pi;
                        elseif (x1 >= x2) && (y1 <= y2)
                            alpha = pi - ang;
                            rot = -ang * 180/pi;
                        end
                        
                        % Calculate current cross-section local coordinates
                        [xs,ys] = draw.coordTransf2D(x, 0, x1, y1, alpha);
                        [xe,ye] = draw.coordTransf2D(L/2, 3*r, x1, y1, alpha);
                        [xf,yf] = draw.coordTransf2D(3*r, 3*r, x1, y1, alpha);
                        [xg,yg] = draw.coordTransf2D(L-6*r, 3*r, x1, y1, alpha);
                        
                        % Get text distance factor
                        distTxt = (max_val/abs(qy));
                        if distTxt > 1.8
                            distTxt = 1.8;
                        elseif distTxt < 1.3
                            distTxt = 1.3;
                        end
                        
                        % Coordinates of transversal load symbol Qy
                        if qy > 0
                            [xa,ya] = draw.coordTransf2D(x, -r, x1, y1, alpha);
                            [xd,yd] = draw.coordTransf2D(L/2, -distTxt*(Qy+r), x1, y1, alpha);
                            if (alpha == ang)
                                dir_y = ang + 3 * pi/2;
                            elseif (alpha == ang + pi)
                                dir_y = ang + pi/2;
                            elseif (alpha == -ang)
                                dir_y = 3 * pi/2 - ang;
                            elseif (alpha == pi - ang)
                                dir_y = pi/2 - ang;
                            end
                        elseif qy < 0
                            [xa,ya] = draw.coordTransf2D(x, r, x1, y1, alpha);
                            [xd,yd] = draw.coordTransf2D(L/2, distTxt*(Qy+r), x1, y1, alpha);
                            [xe,ye] = draw.coordTransf2D(L/2, -3*r, x1, y1, alpha);
                            if (alpha == ang)
                                dir_y = ang + pi/2;
                            elseif (alpha == ang + pi)
                                dir_y = ang + 3 * pi/2;
                            elseif (alpha == -ang)
                                dir_y = pi/2 - ang;
                            elseif (alpha == pi - ang)
                                dir_y = 3 * pi/2 - ang;
                            end
                        end
                        
                        % Get text distance factor
                        distTxt = (max_val/abs(qyi));
                        if distTxt > 1.8
                            distTxt = 1.8;
                        elseif distTxt < 1.3
                            distTxt = 1.3;
                        end
                        
                        if qyi > 0
                            [xb,yb] = draw.coordTransf2D(r, -Qyi - r, x1, y1, alpha);
                            [xh,yh] = draw.coordTransf2D(0, -distTxt*(Qyi+r), x1, y1, alpha);
                        elseif qyi < 0
                            [xb,yb] = draw.coordTransf2D(r, Qyi + r, x1, y1, alpha);
                            [xh,yh] = draw.coordTransf2D(0, distTxt*(Qyi+r), x1, y1, alpha);
                            [xf,yf] = draw.coordTransf2D(3*r, -3*r, x1, y1, alpha);
                        elseif qyi == 0
                            if qyf > 0
                                [xb,yb] = draw.coordTransf2D(r, -Qyi - r, x1, y1, alpha);
                                [xh,yh] = draw.coordTransf2D(0, -distTxt*(Qyi+r), x1, y1, alpha);
                                [xf,yf] = draw.coordTransf2D(3*r, 3*r, x1, y1, alpha);
                            elseif qyf < 0
                                [xb,yb] = draw.coordTransf2D(r, Qyi + r, x1, y1, alpha);
                                [xh,yh] = draw.coordTransf2D(0, distTxt*(Qyi+r), y1, alpha);
                                [xf,yf] = draw.coordTransf2D(3*r, -3*r, x1, y1, alpha);
                            end
                        end
                        
                        % Get text distance factor
                        distTxt = (max_val/abs(qyf));
                        if distTxt > 1.8
                            distTxt = 1.8;
                        elseif distTxt < 1.3
                            distTxt = 1.3;
                        end
                        
                        if qyf > 0
                            [xc,yc] = draw.coordTransf2D(L-r, -Qyf - r, x1, y1, alpha);
                            [xj,yj] = draw.coordTransf2D(L, -distTxt*(Qyf+r), x1, y1, alpha);
                        elseif qyf < 0
                            [xc,yc] = draw.coordTransf2D(L-r, Qyf + r, x1, y1, alpha);
                            [xj,yj] = draw.coordTransf2D(L, distTxt*(Qyf+r), x1, y1, alpha);
                            [xg,yg] = draw.coordTransf2D(L-6*r, -3*r, x1, y1, alpha);
                        elseif qyf == 0
                            if qyi > 0
                                [xc,yc] = draw.coordTransf2D(L-r, -Qyf - r, x1, y1, alpha);
                                [xj,yj] = draw.coordTransf2D(L, -distTxt*(Qyf+r), x1, y1, alpha);
                                [xg,yg] = draw.coordTransf2D(L-6*r, 3*r, x1, y1, alpha);
                            elseif qyi < 0
                                [xc,yc] = draw.coordTransf2D(L-r, Qyf + r, x1, y1, alpha);
                                [xj,yj] = draw.coordTransf2D(L, distTxt*(Qyf+r), x1, y1, alpha);
                                [xg,yg] = draw.coordTransf2D(L-6*r, -3*r, x1, y1, alpha);
                            end
                        end
                        
                        % Coordinates of axial load symbol Qx
                        if qx > 0
                            if (alpha == ang)
                                dir_x = ang + pi;
                            elseif (alpha == ang + pi)
                                dir_x = ang;
                            elseif (alpha == -ang)
                                dir_x = pi - ang;
                            elseif (alpha == pi - ang)
                                dir_x = 2*pi - ang;
                            end
                        elseif qx < 0
                            if (alpha == ang)
                                dir_x = ang;
                            elseif (alpha == ang + pi)
                                dir_x = ang + pi;
                            elseif (alpha == -ang)
                                dir_x = 2*pi - ang;
                            elseif (alpha == pi - ang)
                                dir_x = pi - ang;
                            end
                        end
                        
                        % Draw axial load symbols
                        if ((x ~= r) && ((x+step) < L)) && ((qxi ~= 0) || (qxf ~= 0))
                            draw.triangle(xs, ys, 2*ah, 2*ab, dir_x, clr,'drawElemLoads');
                        end
                                                
                        % Draw transversal load symbol
                        if Qy >= ah
                            draw.arrow2D(xa, ya, Qy, ah, ab, dir_y, clr,'drawElemLoads');
                        elseif (abs(x-r) <= 10^-10) && ((Qyi ~=0) || (Qyf ~=0))
                            [x_ini,y_ini] = draw.coordTransf2D(r, 0, x1, y1, alpha); 
                            X = [xb, x_ini];
                            Y = [yb, y_ini];
                            line(X, Y, 'Color', clr,'tag','drawElemLoads');
                        elseif (abs(x-L+r) <= 10^-10) && ((Qyi ~=0) || (Qyf ~=0))
                            [x_end,y_end] = draw.coordTransf2D(L-r, 0, x1, y1, alpha);
                            X = [x_end, xc];
                            Y = [y_end, yc];
                            line(X, Y, 'Color', clr,'tag','drawElemLoads');
                        end
                    end
                    
                    % Connect diagram extremities
                    if  (qyi < 0) && (qyf > 0)
                        x0 = abs((Qyi*(L-2*r))/(Qyf+Qyi));
                        [xu,yu] = draw.coordTransf2D(x0, r, x1, y1, alpha);
                        [xl,yl] = draw.coordTransf2D(x0, -r, x1, y1, alpha);
                        
                        X = [xb, xu, xl, xc];
                        Y = [yb, yu, yl, yc];
                        line(X, Y, 'Color', clr,'tag','drawElemLoads');
                    elseif (qyi > 0) && (qyf < 0)
                        x0 = abs((Qyi*(L-2*r))/(Qyf+Qyi));
                        [xu,yu] = draw.coordTransf2D(x0, r, x1, y1, alpha);
                        [xl,yl] = draw.coordTransf2D(x0, -r, x1, y1, alpha);
                        
                        X = [xb, xl, xu, xc];
                        Y = [yb, yl, yu, yc];
                        line(X, Y, 'Color', clr,'tag','drawElemLoads');
                    elseif (qyi ~= 0) || (qyf ~= 0)
                        X = [xb, xc];
                        Y = [yb, yc];
                        line(X, Y, 'Color', clr,'tag','drawElemLoads');
                    end
                    
                    % Write load values:
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value_x1 = sprintf('%.*f kN/m',dc,abs(qxi));
                        value_x2 = sprintf('%.*f kN/m',dc,abs(qxf));
                        value_y1 = sprintf('%.*f kN/m',dc,abs(qyi));
                        value_y2 = sprintf('%.*f kN/m',dc,abs(qyf));
                    else
                        value_x1 = sprintf('%.*f',dc,abs(qxi));
                        value_x2 = sprintf('%.*f',dc,abs(qxf));
                        value_y1 = sprintf('%.*f',dc,abs(qyi));
                        value_y2 = sprintf('%.*f',dc,abs(qyf));
                    end

                    % Write axial load values
                    if (qxi ~= 0) || (qxf ~= 0)
                        if qxi == qxf
                            text(xe, ye, value_x1, 'Color', clr, 'Rotation', rot,'tag','textElemLoads','UserData',abs(qxi));
                        else
                            text(xf, yf, value_x1, 'Color', clr, 'Rotation', rot,'tag','textElemLoads','UserData',abs(qxi));
                            text(xg, yg, value_x2, 'Color', clr, 'Rotation', rot,'tag','textElemLoads','UserData',abs(qxf));
                        end
                    end
                    
                    % Write transversal load values
                    if (qyi ~= 0) || (qyf ~= 0)
                        if qyi == qyf
                            text(xd, yd, value_y1, 'Color', clr, 'Rotation', rot,'tag','textElemLoads','UserData',abs(qyi));
                        else
                            if qyi ~= 0
                                text(xh, yh, value_y1, 'Color', clr, 'Rotation', rot,'tag','textElemLoads','UserData',abs(qyi));
                            end
                            if qyf ~= 0
                                text(xj, yj, value_y2, 'Color', clr, 'Rotation', rot,'tag','textElemLoads','UserData',abs(qyf));
                            end
                        end
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws applied nodal loads.
        function draw = nodalLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if nodal loads visualization is on
            if strcmp(get(mdata.viewNodalLoadsButton,'Checked'),'off')
                return
            end
            
            % Parameters
            r = draw.size/100;   % hinge symbol radius
            al = draw.size/12;   % load symbol size (arrow length)
            ah = draw.size/60;   % load symbol size (arrowhead height)
            ab = draw.size/60;   % load symbol size (arrowhead base)
            t1 = draw.size/25;   % distance between text and nodal point
            t2 = draw.size/60;   % distance between text and nodal point
            t3 = draw.size/300;  % distance between text and nodal point
            clr = [1,0,0];       % load symbol color
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for n = 1:draw.drv.nnp
                % Check if current node has a nodal load
                if isempty(draw.drv.nodes(n).nodalLoad) == 0
                    % Get nodal coordinates
                    x = draw.drv.nodes(n).coord(1);
                    y = draw.drv.nodes(n).coord(2);
                
                    % Get nodal load components
                    fx = draw.drv.nodes(n).nodalLoad(1);
                    fy = draw.drv.nodes(n).nodalLoad(2);
                    
                    % Draw horizontal load component
                    if fx > 0
                        draw.arrow2D(x - r, y, al, ah, ab, pi, clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fx));
                        else
                            value = sprintf('%.*f',dc,abs(fx));
                        end
                        text(x - 2.5 * t1, y + t2, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fx));
                        
                    elseif fx < 0
                        draw.arrow2D(x + r, y, al, ah, ab, 0, clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fx));
                        else
                            value = sprintf('%.*f',dc,abs(fx));
                        end
                        text(x + t1, y + t2, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fx));
                    end
                    
                    % Draw vertical load component
                    if fy > 0
                        draw.arrow2D(x, y - r, al, ah, ab, 3 * pi/2, clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fy));
                        else
                            value = sprintf('%.*f',dc,abs(fy));
                        end
                        text(x + t3, y - 1.5 * t1, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fy));
                        
                    elseif fy < 0
                        draw.arrow2D(x, y + r, al, ah, ab, pi/2, clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fy));
                        else
                            value = sprintf('%.*f',dc,abs(fy));
                        end
                        text(x + t3, y + 1.5 * t1, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fy));
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws nodal prescribed displacement representation.
        function draw = nodalPrescDispl(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if presc displ visualization is on
            if strcmp(get(mdata.viewPrescDisplButton,'Checked'),'off')
                return
            end
            
            % Parameters
            shift = draw.size/100;  % distance between presc. displ. symbol and support
            r = draw.size/100;      % hinge symbol radius
            th = draw.size/35;      % translation constraint symbol (triangle height)
            al = draw.size/12;      % presc. displ. symbol size (arrow length)
            ah = draw.size/60;      % presc. displ. symbol size (arrowhead height)
            ab = draw.size/60;      % presc. displ. symbol size (arrowhead base)
            t1 = draw.size/200;     % distance between text and nodal point
            t2 = draw.size/40;      % distance between text and nodal point
            clr = [1,0,1];          % prec. displ. symbol color
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for n =1:draw.drv.nnp
                % Check if current node has a prescribed displacement
                if isempty(draw.drv.nodes(n).prescDispl) == 0
                    % Get nodal coordinates
                    x = draw.drv.nodes(n).coord(1);
                    y = draw.drv.nodes(n).coord(2);
                    
                    % Get prescribed displacement component values and
                    % convert it to milimeter
                    dx = 1000 * draw.drv.nodes(n).prescDispl(1);
                    dy = 1000 * draw.drv.nodes(n).prescDispl(2);
                    
                    % Check if horizontal translation is really fixed and
                    % draw prescribed displacement indication
                    if (draw.drv.nodes(n).ebc(1) == 1) && (dx ~= 0)
                        if dx > 0
                            draw.arrow2D(x - r - th - shift, y, al, ah, ab, pi, clr,'drawPrescDispl');
                        else
                            draw.arrow2D(x - r - th - shift - al, y, al, ah, ab, 0, clr,'drawPrescDispl');
                        end
                        
                        % Write displacement value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f mm',dc,abs(dx));
                        else
                            value = sprintf('%.*f',dc,abs(dx));
                        end
                        text(x - r - th - shift - al, y + t2, value, 'Color', clr,'tag','textPrescDispl','UserData',abs(dx));
                    end
                    
                    
                    % Check if vertical translation is really fixed and
                    % draw prescribed displacement indication
                    if (draw.drv.nodes(n).ebc(2) == 1) && (dy ~= 0)
                        if dy > 0
                            draw.arrow2D(x, y - r - th - shift, al, ah, ab, 3 * pi/2, clr,'drawPrescDispl');
                        else
                            draw.arrow2D(x, y - r - th - shift - al, al, ah, ab, pi/2, clr,'drawPrescDispl');
                        end
                        
                        % Write displacement value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f mm',dc,abs(dy));
                        else
                            value = sprintf('%.*f',dc,abs(dy));
                        end
                        text(x + t1, y - r - th - shift - al/2, value, 'Color', clr,'tag','textPrescDispl','UserData',abs(dy));
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws thermal load representation on elements.
        function draw = thermalLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if thermal loads visualization is on
            if strcmp(get(mdata.viewThermalLoadsButton,'Checked'),'off')
                return
            end
            
            % Parameters
            d = draw.size/150;     % distance between temperature grad symbol and element
            heatClr = [1,0,0];     % heat color
            coldClr = [0,0,1];     % cold color
            dc = getappdata(0,'decPrec');  % decimal precision
            
            for e = 1:draw.drv.nel
                if (draw.drv.elems(e).load.tempVar_X ~= 0) || ...
                   (draw.drv.elems(e).load.tempVar_Y ~= 0) || ...
                   (draw.drv.elems(e).load.tempVar_Z ~= 0)
                    % Get nodal coordinates
                    x1 = draw.drv.elems(e).nodes(1).coord(1);
                    y1 = draw.drv.elems(e).nodes(1).coord(2);
                    x2 = draw.drv.elems(e).nodes(2).coord(1);
                    y2 = draw.drv.elems(e).nodes(2).coord(2);

                    % Get element length
                    L = draw.drv.elems(e).length;

                    % Get element inclination
                    cx = draw.drv.elems(e).cosine_X;
                    cy = draw.drv.elems(e).cosine_Y;

                    % Get temperature gradient values
                    dtx = draw.drv.elems(e).load.tempVar_X;
                    dty = draw.drv.elems(e).load.tempVar_Y;

                    % Calculate element orientation angle and text
                    % rotation angle acording to element orientation
                    ang = acos(abs(draw.drv.elems(e).cosine_X));
                    if (x1 <= x2) && (y1 <= y2)
                        alpha = ang;
                        rot = ang * 180/pi;
                        Lx1 = 0.3*L;
                        Lx2 = 0.6*L;
                    elseif (x1 >= x2) && (y1 >= y2)
                        alpha = ang + pi;
                        rot = ang * 180/pi;
                        Lx1 = 0.4*L;
                        Lx2 = 0.7*L;
                    elseif (x1 <= x2) && (y1 >= y2)
                        alpha = -ang;
                        rot = -ang * 180/pi;
                        Lx1 = 0.3*L;
                        Lx2 = 0.6*L;
                    elseif (x1 >= x2) && (y1 <= y2)
                        alpha = pi - ang;
                        rot = -ang * 180/pi;
                        Lx1 = 0.4*L;
                        Lx2 = 0.7*L;
                    end

                    % Calculate transversal load value at the middle of the element
                    qy = 0;
                    if ~isempty(draw.drv.elems(e).load.uniformLcl)
                        qy = draw.drv.elems(e).load.uniformLcl(2);
                    end
                    if ~isempty(draw.drv.elems(e).load.linearLcl)
                        qy = qy + (draw.drv.elems(e).load.linearLcl(2)  + draw.drv.elems(e).load.linearLcl(5))/2;
                    end

                    % Calculate coordinates to write temp. var. values
                    if qy >= 0
                        [xdx,ydx] = draw.coordTransf2D(Lx1, 4*d, x1, y1, alpha);
                        [xdy,ydy] = draw.coordTransf2D(Lx2, 4*d, x1, y1, alpha);
                    elseif qy < 0
                        [xdx,ydx] = draw.coordTransf2D(Lx1, -4*d, x1, y1, alpha);
                        [xdy,ydy] = draw.coordTransf2D(Lx2, -4*d, x1, y1, alpha);
                    end
                    
                    % Check if units are enabled
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        unitsAreOn = true;
                    else
                        unitsAreOn = false;
                    end

                    % Draw temperature variation symbols
                    if dtx > 0
                        line([x1,x2], [y1,y2], 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text(xdx, ydx, sprintf('dTx = %.*f C',dc,dtx), 'Color', heatClr, 'Rotation', rot, 'tag', 'textThermalLoads','UserData',{'dTx = ',dtx});
                            case false
                                text(xdx, ydx, sprintf('dTx = %.*f',dc,dtx), 'Color', heatClr, 'Rotation', rot, 'tag', 'textThermalLoads','UserData',{'dTx = ',dtx});
                        end
                    elseif dtx < 0
                        line([x1,x2], [y1,y2], 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text(xdx, ydx, sprintf('dTx = %.*f C',dc,dtx), 'Color', coldClr, 'Rotation', rot, 'tag', 'textThermalLoads','UserData',{'dTx = ',dtx});
                            case false
                                text(xdx, ydx, sprintf('dTx = %.*f',dc,dtx), 'Color', coldClr, 'Rotation', rot, 'tag', 'textThermalLoads','UserData',{'dTx = ',dtx});
                        end
                    end

                    if dty > 0
                        line([x1-d*cy,x2-d*cy], [y1+d*cx,y2+d*cx], 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        line([x1+d*cy,x2+d*cy], [y1-d*cx,y2-d*cx], 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text(xdy, ydy, sprintf('dTy = %.*f C',dc,dty), 'Color', heatClr, 'Rotation', rot, 'tag', 'textThermalLoads','UserData',{'dTy = ',dty});
                            case false
                                text(xdy, ydy, sprintf('dTy = %.*f',dc,dty), 'Color', heatClr, 'Rotation', rot, 'tag', 'textThermalLoads','UserData',{'dTy = ',dty});
                        end
                    elseif dty < 0
                        line([x1-d*cy,x2-d*cy], [y1+d*cx,y2+d*cx], 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        line([x1+d*cy,x2+d*cy], [y1-d*cx,y2-d*cx], 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text(xdy, ydy, sprintf('dTy = %.*f C',dc,dty), 'Color', coldClr, 'Rotation', rot, 'tag', 'textThermalLoads','UserData',{'dTy = ',dty});
                            case false
                                text(xdy, ydy, sprintf('dTy = %.*f',dc,dty), 'Color', coldClr, 'Rotation', rot, 'tag', 'textThermalLoads','UserData',{'dTy = ',dty});
                        end
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Plots ID number of nodes.
        function draw = nodeID(draw)
            % Parameters
            tx = draw.size/90;  % distance between text and nodal point (horizontal)
            ty = draw.size/50;  % distance between text and nodal point (vertical)
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
                
                % Write ID number
                number = sprintf('%d',n);
                text(x + tx, y + ty, number, 'FontWeight', 'bold', 'tag', 'textNodeID');
            end
        end
        
        %------------------------------------------------------------------
        % Plots ID number of elements.
        function draw = elementID(draw)
            % Parameters
            tx = draw.size/45;  % distance between text and element (horizontal)
            ty = draw.size/45;  % distance between text and element (vertical)
            
            for e = 1:draw.drv.nel
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                
                % Write ID number
                number = sprintf('%d',e);
                text((x1+x2)/2 - tx, (y1+y2)/2 - ty, number, 'FontWeight', 'bold', 'tag', 'textElemID');
            end
        end
        
        %------------------------------------------------------------------
        % Draws element orientation indication from inital to final node.
        function draw = elementOrientation(draw)
            % Parameters
            t = 1.5;             % distance between text and symbol
            clr = [0,.7,0];      % orientation symbol color
            
            for e = 1:draw.drv.nel
                % Calculate spear length
                l = draw.size/20;
                
                % Get nodal coordinates
                xi = draw.drv.elems(e).nodes(1).coord(1);
                yi = draw.drv.elems(e).nodes(1).coord(2);
                xf = draw.drv.elems(e).nodes(2).coord(1);
                yf = draw.drv.elems(e).nodes(2).coord(2);
                
                % Calculate element local axis X orientation vector
                x = [xf-xi, yf-yi, 0];
                x = l * x / norm(x);
                
                % Calculate element local axis Y orientation vector
                y = cross([0,0,1],x);
                y = l * y / norm(y);
                
                % Draw orientation symbol
                xm = (xi + xf)/2;
                ym = (yi + yf)/2;
                
                X = [xm, xm + x(1)];
                Y = [ym, ym + x(2)];
                line(X, Y, 'Color', clr,'Linewidth',1.2, 'tag', 'drawElemOrient');
                text(xm + t * x(1), ym + t * x(2), 'x', 'Color', clr, 'FontWeight', 'bold', 'tag', 'drawElemOrient');
                
                X = [xm, xm + y(1)];
                Y = [ym, ym + y(2)];
                line(X, Y, 'Color', clr,'Linewidth',1.2, 'tag', 'drawElemOrient');
                text(xm + t * y(1), ym + t * y(2), 'y', 'Color', clr, 'FontWeight', 'bold', 'tag', 'drawElemOrient');
            end
        end
        
        %------------------------------------------------------------------
        % Computes deformed configuration scale factor.
        function draw = deformScaleFactor(draw)
            mdata = guidata(findobj('Tag','GUI_Main'));
            slider = get(mdata.slider_Scale,'Max');
            
            % Calculate maximum nodal displacement
            m = zeros(1,draw.drv.nnp);
            for n = 1:draw.drv.nnp
                dx = draw.drv.D(draw.drv.ID(1,n));
                dy = draw.drv.D(draw.drv.ID(2,n));
                d = [dx,dy];
                m(n) = max(abs(d));
            end
            max_ndisp = max(m);
            
            % Set adapted scale value
            dsf = draw.size/(4*slider*max_ndisp);
            if dsf == inf
                dsf = 0;
            end
            setappdata(0,'deform_sf',dsf);
        end
        
        %------------------------------------------------------------------
        % Draws structure deformed configuration on a given scale.
        % Input arguments:
        %  scale: deformed configuration scale factor
        function draw = deformConfig(draw,scale)
            % Parameters
            clr = [1,0,0];  % deformed configuration line color
            
            for e = 1:draw.drv.nel
                % Get element end nodes IDs
                n1 = draw.drv.elems(e).nodes(1).id;
                n2 = draw.drv.elems(e).nodes(2).id;
                
                % Get nodal coordinates
                x1 = draw.drv.nodes(n1).coord(1);
                y1 = draw.drv.nodes(n1).coord(2);
                x2 = draw.drv.nodes(n2).coord(1);
                y2 = draw.drv.nodes(n2).coord(2);
                
                % Get nodal displacements
                dx1 = draw.drv.D(draw.drv.ID(1,n1));
                dy1 = draw.drv.D(draw.drv.ID(2,n1));
                dx2 = draw.drv.D(draw.drv.ID(1,n2));
                dy2 = draw.drv.D(draw.drv.ID(2,n2));
                
                % Calculate displaced nodal coordinates
                xd1 = x1 + scale * dx1;
                yd1 = y1 + scale * dy1;
                xd2 = x2 + scale * dx2;
                yd2 = y2 + scale * dy2;
                
                % Connect displaced nodal coordinates
                X = [xd1, xd2];
                Y = [yd1, yd2];
                line(X, Y, 'Color', clr, 'tag', 'drawDeformConfig');
            end
        end
        
        %------------------------------------------------------------------
        % Computes axial force diagram scale factor value.
        % NOT USED
        function draw = axialScaleFactor(draw)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Draws resulting axial force diagram on a given scale.
        % Input arguments:
        %  scale: axial force diagram scale factor
        function draw = axialForce(draw,~)
            % Parameters
            shift = draw.size/50;  % distance between text and element
            clr = [1,0,0];         % diagram color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            % Get target element ID
            elem = get(mdata.popupmenu_ElementResults,'Value') - 1;
            if elem ~= 0
                i = elem;
                j = i;
            else
                i = 1;
                j = draw.drv.nel;
            end
            
            for e = i:j
                % Get element internal axial force value (always uniform
                % for 2D truss models) and convert it to string
                N = -draw.drv.elems(e).axial_force(1);
                if N ~= 0
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%+.*f kN',dc,N);
                    else
                        value = sprintf('%+.*f',dc,N);
                    end
                else
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%.*f kN',dc,abs(N));
                    else
                        value = sprintf('%.*f',dc,abs(N));
                    end
                    N = abs(N);
                end
                
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                
                % Get element inclination and element length
                alpha = acos(abs(draw.drv.elems(e).cosine_X));
                ang = alpha*180/pi;
                L = draw.drv.elems(e).length;
                
                % Write axial force value
                if ((x1 <= x2) && (y1 <= y2))
                    [xm,ym] = draw.coordTransf2D(L/2 - 1.30*shift, 0.85*shift, x1, y1, alpha);
                    text(xm, ym, value, 'Color', clr, 'Rotation', ang,'tag','textAxialForceDiagram','UserData',N);
                elseif ((x1 >= x2) && (y1 >= y2))
                    [xm,ym] = draw.coordTransf2D(L/2 + 1.10*shift, -0.85*shift, x1, y1, alpha + pi);
                    text(xm, ym, value, 'Color', clr, 'Rotation', ang,'tag','textAxialForceDiagram','UserData',N);
                elseif ((x1 <= x2) && (y1 >= y2))
                    [xm,ym] = draw.coordTransf2D(L/2 - 1.30*shift, 0.85*shift, x1, y1, -alpha);
                    text(xm, ym, value, 'Color', clr, 'Rotation', -ang,'tag','textAxialForceDiagram','UserData',N);
                elseif ((x1 >= x2) && (y1 <= y2))
                    [xm,ym] = draw.coordTransf2D(L/2 + 1.10*shift, -0.85*shift, x1, y1, pi - alpha);
                    text(xm, ym, value, 'Color', clr, 'Rotation', -ang,'tag','textAxialForceDiagram','UserData',N);
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws resulting torsion moment diagram.
        % NOT USED
        function draw = torsionMoment(draw)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Computes shear force diagram scale factor value in XY plane.
        % NOT USED
        function draw = shearScaleFactor_XY(draw)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Draws resulting shear force diagram in XY plane on a given scale.
        % NOT USED
        function draw = shearForce_XY(draw,~)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Computes shear force diagram scale factor value in XZ plane.
        % NOT USED
        function draw = shearScaleFactor_XZ(draw)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Draws resulting shear force diagram in XZ plane on a given scale.
        % NOT USED
        function draw = shearForce_XZ(draw,~)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Computes bending moment diagram scale factor value in XY plane.
        % NOT USED
        function draw = bendingMomentScaleFactor_XY(draw)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Draws resulting bending moment diagram in XY plane on a given scale.
        % NOT USED
        function draw = bendingMoment_XY(draw,~)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Computes bending moment diagram scale factor value in XZ plane.
        % NOT USED
        function draw = bendingMomentScaleFactor_XZ(draw)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Draws resulting bending moment diagram in XZ plane on a given scale.
        % NOT USED
        function draw = bendingMoment_XZ(draw,~)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Draws reactions indication next to nodal supports.
        function draw = reactions(draw)
            include_constants;
            % Parameters
            shift = draw.size/100;  % distance between reaction symbol and support
            r = draw.size/100;      % hinge symbol radius
            th = draw.size/35;      % translation constraint symbol (triangle height)
            sh = draw.size/15;      % displacement spring symbol (spring height)
            al = draw.size/12;      % reaction symbol size (arrow length)
            ah = draw.size/60;      % reaction symbol size (arrowhead height)
            ab = draw.size/60;      % reaction symbol size (arrowhead base)
            t1 = draw.size/200;     % distance between text and nodal point
            t2 = draw.size/50;      % distance between text and nodal point
            clr = [0,0,1];          % reaction symbol color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
                
                % Get reactions values
                rx = draw.drv.F(draw.drv.ID(1,n));
                ry = draw.drv.F(draw.drv.ID(2,n));
                
                % Support (or spring) height in the direction of axis X
                if draw.drv.nodes(n).ebc(1)== FIXED_DOF
                    hx = th;
                elseif draw.drv.nodes(n).ebc(1)== SPRING_DOF
                    hx = sh;
                end
                
                % Check if horizontal translation is fixed and draw
                % reaction indication
                if (draw.drv.nodes(n).ebc(1)== FIXED_DOF) || (draw.drv.nodes(n).ebc(1)== SPRING_DOF)
                    if rx >= 0
                        draw.arrow2D(x - r - hx - shift, y, al, ah, ab, pi, clr,'drawReactions');
                    else
                        draw.arrow2D(x - r - hx - shift - al, y, al, ah, ab, 0, clr,'drawReactions');
                    end
                    
                    % Write reaction value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%.*f kN',dc,abs(rx));
                    else
                        value = sprintf('%.*f',dc,abs(rx));
                    end
                    text(x - r - hx - shift - al, y + 1.2 * t2, value, 'Color', clr,'tag','textForceReactions','UserData',abs(rx));
                end
                
                % Support (or spring) height in the direction of axis Y
                if draw.drv.nodes(n).ebc(2)== FIXED_DOF
                    hy = th;
                elseif draw.drv.nodes(n).ebc(2)== SPRING_DOF
                    hy = sh;
                end 
                
                % Check if vertical translation is fixed and draw
                % reaction indication
                if (draw.drv.nodes(n).ebc(2) == FIXED_DOF) || (draw.drv.nodes(n).ebc(2) == SPRING_DOF)
                    if ry >= 0
                        draw.arrow2D(x, y - r - hy - shift, al, ah, ab, 3 * pi/2, clr,'drawReactions');
                    else
                        draw.arrow2D(x, y - r - hy - shift - al, al, ah, ab, pi/2, clr,'drawReactions');
                    end
                    
                    % Write reaction value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%.*f kN',dc,abs(ry));
                    else
                        value = sprintf('%.*f',dc,abs(ry));
                    end
                    text(x + t1, y - r - hy - shift - al/2, value, 'Color', clr,'tag','textForceReactions','UserData',abs(ry));
                end
            end
        end
    end
end