%% Draw_Grillage2D class
%
% This is a sub-class, in the Object Oriented Programming (OOP) paradigm,
% of super-class <draw.html *Draw*> in the <main.html LESM (Linear Elements
% Structure Model)> program. This sub-class implements abstract methods,
% defined in super-class *Draw*, that deal with 2D visualization of a
% Grillage model.
%
%% Authors
% Luiz Fernando Martha, Rafael Lopez Rangel and Pedro Cortez Lopes
%
%% Class definition
classdef Draw_Grillage2D < Draw
    %% Constructor method
    methods
        %------------------------------------------------------------------
        function draw = Draw_Grillage2D(drv)
            draw = draw@Draw(0,90);
            
            if (nargin > 0)
                draw.drv = drv;
            end
        end
    end
    
    %% Public methods
    % Implementation of the abstract methods declared in super-class <draw.html *Draw*>.
    methods
        %------------------------------------------------------------------
        % Sets viewpoint position and computes drawing size parameter
        % according to model dimensions.
        function draw = setSize(draw)
            
            x = zeros(draw.drv.nnp);
            y = zeros(draw.drv.nnp);
            z = zeros(draw.drv.nnp);
            
            for n = 1:draw.drv.nnp
                x(n) = draw.drv.nodes(n).coord(1);
                y(n) = draw.drv.nodes(n).coord(2);
                z(n) = draw.drv.nodes(n).coord(3);
            end
            
            xmin = min(x);
            xmax = max(x);
            ymin = min(y);
            ymax = max(y);
            zmin = min(z);
            zmax = max(z);
            
            dx = xmax - xmin;
            dy = ymax - ymin;
            dz = zmax - zmin;
            sz = max([dx,dy,dz]);
            if isempty(sz)
                sz = 0;
            end
            
            if sz == 0
                draw.size = 5;
            else
                draw.size = sz;
            end
        end
        
        %------------------------------------------------------------------
        % Sets axes limits according to model dimensions.
        function draw = setLimits(draw,axProp)
            if nargin == 2
                setBothAxis = true;
            else
                setBothAxis = false;
            end
            if draw.drv.nnp > 0
                x = zeros(draw.drv.nnp,1);
                y = zeros(draw.drv.nnp,1);
                
                for n = 1:draw.drv.nnp
                    x(n) = draw.drv.nodes(n).coord(1);
                    y(n) = draw.drv.nodes(n).coord(2);
                end
                
                xmin = min(x);
                xmax = max(x);
                ymin = min(y);
                ymax = max(y);
                
                dx = xmax - xmin;
                dy = ymax - ymin;
                
                if (dx == 0) && (dy == 0)
                    xlim([x - 5, x + 5])
                    if setBothAxis == true
                        ylim([y - 5/axProp, y + 5/axProp])
                    end
                elseif dx > dy
                    xlim([xmin - dx/1.7, xmax + dx/1.7])
                    if setBothAxis == true
                        aux_xlim = [xmin - dx/1.7, xmax + dx/1.7];
                        xlength = abs(diff(aux_xlim));
                        ylength = xlength/axProp;
                        ymean = mean([ymin ymax]);
                        ylim([ymean - ylength/2, ymean + ylength/2])
                    end
                else
                    if setBothAxis == true
                        aux_ylim = [ymin - dy/4, ymax + dy/4];
                        ylength = abs(diff(aux_ylim));
                        xlength = ylength * axProp;
                        xmean = mean([xmin xmax]);
                        xlim([xmean - xlength/2, xmean + xlength/2])
                    end
                    ylim([ymin - dy/4, ymax + dy/4])
                end
                
            else
                xlim([-5,5])
                if setBothAxis == true
                    ylim([-5/axProp, 5/axProp])
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws structure model with nodes, elements, supports and  hinges.
        function draw = model(draw)
            draw.nodes();
            hold on
            draw.elements();
        end
        
        %------------------------------------------------------------------
        % Draws elements with hinged or continuous ends.
        function draw = elements(draw)
            % Parameters
            r = draw.size/125;  % hinge symbol radius
            clr = [0,0,0];      % element color
            nclr = [0,0,0];        % node and hinge color
            
            for e = 1:draw.drv.nel
                % Get nodal IDs and coordinates
                n1 = draw.drv.elems(e).nodes(1).id;
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                
                n2 = draw.drv.elems(e).nodes(2).id;
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                
                % Get element orientation angle cosine with X and Y axes
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
                
                % Get element incidence information on nodes
                [toti,hei] = draw.drv.nodes(n1).elemsIncidence(draw.drv);
                [totf,hef] = draw.drv.nodes(n2).elemsIncidence(draw.drv);
                
                % Set element end coordinates
                xi = x1;
                yi = y1;
                xf = x2;
                yf = y2;
                
                % Set element initial coordinates by checking if there is a
                % hinge on nodal point position or on element end
                if (hei >= (toti-1)) && (toti >= 2) && (draw.drv.nodes(n1).ebc(6) == 0)
                    % Draw one hinge symbol representing the articulation
                    % between all elements
                    draw.circle(x1, y1, r, nclr,'drawNodes');
                    hold on
                    % One hinge on nodal point position
                    xi = x1 + r * cx;
                    yi = y1 + r * cy;
                    
                elseif draw.drv.elems(e).hingei == 0
                    % Hinge on element end
                    draw.circle(x1 + r * cx, y1 + r * cy, r, clr, 'drawElements');
                    xi = x1 + 2 * r * cx;
                    yi = y1 + 2 * r * cy;
                end
                
                % Set element final coordinates by checking if there is a
                % hinge on nodal point position or on element end
                if (hef >= (totf-1)) && (totf >= 2) && (draw.drv.nodes(n2).ebc(6) == 0)
                    % Draw one hinge symbol representing the articulation
                    % between all elements
                    draw.circle(x2, y2, r, nclr,'drawNodes');
                    hold on
                    % One hinge on nodal point position
                    xf = x2 - r * cx;
                    yf = y2 - r * cy;
                    
                elseif draw.drv.elems(e).hingef == 0
                    % Hinge on element end
                    draw.circle(x2 - r * cx, y2 - r * cy, r, clr, 'drawElements');
                    xf = x2 - 2 * r * cx;
                    yf = y2 - 2 * r * cy;
                end
                
                % Connect element end coordinates
                X = [xi, xf];
                Y = [yi, yf];
                line(X, Y, 'Color', clr, 'tag', 'drawElements');
                hold on
            end
        end
        
        %------------------------------------------------------------------
        % Draws nodal marks with support conditions.
        function draw = nodes(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Get flag for support visualization option status
            drawSupports =  get(mdata.viewSupportsButton,'Checked');
            
            include_constants;
            % Parameters
            nm = draw.size/200;    % node mark symbol (square side)
            tb = draw.size/50;     % translation constraint symbol (triangle base)
            ss = draw.size/35;     % rotation constraint symbol (square side)
            sh = draw.size/15;     % displacement spring symbol (spring height)
            sr = draw.size/115;    % rotational spring symbol radius
            nclr = [0,0,0];        % node and hinge color
            sclr = [0.6,0.6,0.6];  % support color
            sprclr = [0.6,0,0.4];  % spring color
            dc = getappdata(0,'decPrec');  % decimal precision
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
                
                % Get element incidence information
                [tot,hng] = draw.drv.nodes(n).elemsIncidence(draw.drv);
                
                if (hng >= (tot-1)) && (tot >= 2) && ...
                   ((draw.drv.nodes(n).ebc(4) == FREE_DOF) || (draw.drv.nodes(n).ebc(5) == FREE_DOF))
                    % Draw a nodal mark representing a free node
                    draw.square(x, y, nm, nclr,'drawNodes');
                    hold on
                    
                elseif  (draw.drv.nodes(n).ebc(4) == FIXED_DOF) && (draw.drv.nodes(n).ebc(5) == FIXED_DOF)
                    if strcmp(drawSupports,'on')
                        % Draw rotation constraint support
                        draw.square(x, y, ss, sclr,'drawSupports');
                        hold on
                    end
                    
                elseif (draw.drv.nodes(n).ebc(3) == FREE_DOF) &&...
                       (draw.drv.nodes(n).ebc(4) == FREE_DOF) &&...
                       (draw.drv.nodes(n).ebc(5) == FREE_DOF)
                    % Draw a nodal mark representing a free node
                    draw.square(x, y, nm, nclr,'drawNodes');
                    hold on
                end
                
                % Check if there is a translation constraint in the
                % direction of axis Z and draw support
                if draw.drv.nodes(n).ebc(3) == FIXED_DOF && strcmp(drawSupports,'on')
                    draw.square(x,y,tb,sclr,'drawSupports')
                    hold on
                    plot([x x-tb/2],[y y-tb/2],'color',[0 0 0],'tag','drawSupports')
                    plot([x x+tb/2],[y y-tb/2],'color',[0 0 0],'tag','drawSupports')
                    plot([x x-tb/2],[y y+tb/2],'color',[0 0 0],'tag','drawSupports')
                    plot([x x+tb/2],[y y+tb/2],'color',[0 0 0],'tag','drawSupports')
                end
                
                % Check if there is a displacement spring in the direction
                % of axis Z and draw spring
                if draw.drv.nodes(n).ebc(3) == SPRING_DOF && strcmp(drawSupports,'on') 
                    draw.square(x, y, tb, sclr,'drawSupports')
                    hold on
                    draw.circle(x, y, sh/13.75, sprclr,'drawSupports')
                    hold on
                    
                    % Get spring stiffness
                    kz = draw.drv.nodes(n).springStiff(3);
                    % Write spring stiffness value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        if kz >= 1000
                        value = sprintf('%.*e kN/m',dc,kz);
                        else
                        value = sprintf('%.*f kN/m',dc,kz);
                        end
                    else
                        if kz >= 1000
                        value = sprintf('%.*e',dc,kz);
                        else
                        value = sprintf('%.*f',dc,kz);
                        end
                    end
                    
                    text(x - sh, y + sh*0.3, value, 'Color', sprclr,'Fontsize',8.5,'tag','textSprings','UserData',kz);
                end
                
                % Check if there is a rotational srping about the direction
                % of axis X and Y and draw rotational spring
                if draw.drv.nodes(n).ebc(4) == SPRING_DOF && draw.drv.nodes(n).ebc(5) == SPRING_DOF...
                   && strcmp(drawSupports,'on')
                    circ = 0 : pi/50 : 2*pi;
                    xcirc = x + sr * cos(circ);
                    ycirc = y + sr * sin(circ);
                    fill(xcirc, ycirc, sprclr,'tag','drawSupports');
                    hold on
                    
                    % Get spring stiffness
                    krxy = draw.drv.nodes(n).springStiff(4);
                    % Write spring stiffness value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        if krxy >= 1000
                        value = sprintf('%.*e kNm/rad',dc,krxy);
                        else
                        value = sprintf('%.*f kNm/rad',dc,krxy);
                        end
                    else
                        if krxy >= 1000
                        value = sprintf('%.*e',dc,krxy);
                        else
                        value = sprintf('%.*f',dc,krxy);
                        end
                    end
                    
                    text(x - sh, y + sh*0.3, value, 'Color', sprclr,'Fontsize',8.5,'tag','textRotSprings','UserData',krxy);
                end    
            end
        end
        
        %------------------------------------------------------------------
        % Plots ID number of nodes.
        function draw = nodeID(draw)
            % Parameters
            tx = draw.size/90;  % distance between text and nodal point (horizontal)
            ty = draw.size/50;  % distance between text and nodal point (vertical)
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
                
                % Write ID number
                number = sprintf('%d',n);
                text(x + tx, y + ty, number, 'FontWeight', 'bold','tag','textNodeID');
            end
        end
        
        %------------------------------------------------------------------
        % Plots ID number of elements.
        function draw = elementID(draw)
            % Parameters
            tx = draw.size/45;  % distance between text and element (horizontal)
            ty = draw.size/45;  % distance between text and element (vertical)
            
            for e = 1:draw.drv.nel
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                
                % Write ID number
                number = sprintf('%d',e);
                text((x1+x2)/2 - tx, (y1+y2)/2 - ty, number, 'FontWeight', 'bold','tag','textElemID');
            end
        end
        
        %------------------------------------------------------------------
        % Draws element orientation indication from inital to final node.
        function draw = elementOrientation(draw)
            % Parameters
            t = 1.5;             % distance between text and symbol
            clr = [0,.7,0];      % orientation symbol color
            
            for e = 1:draw.drv.nel
                % Calculate spear length
                l = draw.size/20;
                
                % Get nodal coordinates
                xi = draw.drv.elems(e).nodes(1).coord(1);
                yi = draw.drv.elems(e).nodes(1).coord(2);
                xf = draw.drv.elems(e).nodes(2).coord(1);
                yf = draw.drv.elems(e).nodes(2).coord(2);
                
                % Calculate element local axis X orientation vector
                x = [xf-xi, yf-yi, 0];
                x = l * x / norm(x);
                
                % Calculate element local axis Y orientation vector
                y = cross([0,0,1],x);
                y = l * y / norm(y);
                
                % Draw orientation symbol
                xm = (xi + xf)/2;
                ym = (yi + yf)/2;
                
                X = [xm, xm + x(1)];
                Y = [ym, ym + x(2)];
                line(X, Y, 'Color', clr,'Linewidth',1.2, 'tag', 'drawElemOrient');
                text(xm + t * x(1), ym + t * x(2), 'x', 'Color', clr, 'FontWeight', 'bold', 'tag', 'drawElemOrient');
                
                X = [xm, xm + y(1)];
                Y = [ym, ym + y(2)];
                line(X, Y, 'Color', clr,'Linewidth',1.2, 'tag', 'drawElemOrient');
                text(xm + t * y(1), ym + t * y(2), 'y', 'Color', clr, 'FontWeight', 'bold', 'tag', 'drawElemOrient');
            end
        end
        
        %------------------------------------------------------------------
        % Computes element loads scale factor.
        function draw = elemLoadsScaleFactor(draw)
        end
        
        %------------------------------------------------------------------
        % Draws element distributed loads (uniform and linear).
        function draw = elemLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if elem loads visualization is on
            if strcmp(get(mdata.viewDistribLoadsButton,'Checked'),'off')
                return
            end
            
            % Parameters
            shift = draw.size/75;  % distance between load symbol and element
            pb = draw.size/125;     % load symbol size (pyramid base)
            clr = [1,0,0];          % load symbol color
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for e = 1:draw.drv.nel
                if ((isempty(draw.drv.elems(e).load.uniformGbl) == 0) &&...
                   (all(draw.drv.elems(e).load.uniformGbl == 0) == 0)) ||...
                   ((isempty(draw.drv.elems(e).load.linearGbl) == 0) &&...
                   (all(draw.drv.elems(e).load.linearGbl == 0) == 0))
               
                    % Get element length
                    L = draw.drv.elems(e).length;
                    
                    % Get element orientation angle cosine with axes X and Y
                    cx = draw.drv.elems(e).cosine_X;
                    cy = draw.drv.elems(e).cosine_Y;
                    
                    % Get element end nodes IDs
                    n1 = draw.drv.elems(e).nodes(1).id;
                    n2 = draw.drv.elems(e).nodes(2).id;
                    
                    % Get nodal coordinates
                    x1 = draw.drv.nodes(n1).coord(1);
                    y1 = draw.drv.nodes(n1).coord(2);
                    x2 = draw.drv.nodes(n2).coord(1);
                    y2 = draw.drv.nodes(n2).coord(2);
                    
                    % Initialize load values on element ends
                    qi = 0;
                    qf = 0;
                    
                    % Add uniform load contribtuion
                    if isempty(draw.drv.elems(e).load.uniformGbl) == 0
                        qz = draw.drv.elems(e).load.uniformGbl(3);
                        qi = qi + qz;
                        qf = qf + qz;
                    end
                    
                    % Add linear load contribtuion
                    if isempty(draw.drv.elems(e).load.linearGbl) == 0
                        qzi = draw.drv.elems(e).load.linearGbl(3);
                        qzf = draw.drv.elems(e).load.linearGbl(6);
                        qi = qi + qzi;
                        qf = qf + qzf;
                    end
                    
                    % Calculate load quation coefficients:
                    % q(x) = Ax + B
                    A = (qf - qi)/L;
                    B = qi;
                    
                    % Draw load symbol on a number cross-sections along
                    % element local axis X
                    step = L / round(20 * L/draw.size);
                    for x = 0:step:L
                        % Calculate current cross-section coordinates
                        xs = x1 + x * cx;
                        ys = y1 + x * cy;
                        
                        % Calculate load value on current cross-section
                        q = A * x + B;
                        
                        % Draw load symbol on current cross-section
                        if q > 0
                            dir = 'z+';
                        elseif q < 0
                            dir = 'z-';
                        end
                        
                        if q ~= 0
                            draw.arrow3DPlaneProj(xs,ys,pb,dir,clr,'drawElemLoads')
                            hold on
                        end
                    end
                    
                    % Plot line
                    plot([x1 x2],[y1 y2],':','color',clr,'LineWidth',1.35,'tag','drawElemLoads')

                    % Write load values:
                    
                    % If load is uniform, draw load value in the middle of
                    % the element
                    if qi == qf
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN/m',dc,qi);
                        else
                            value = sprintf('%.*f',dc,qi);
                        end
                        
                        if x1 == x2
                            text(x1 + shift, (y1+y2)/2, value, 'Color', clr,'tag','textElemLoads','UserData',qi);
                        elseif y1 == y2
                            text((x1+x2)/2, y1 + 1.5*shift, value, 'Color', clr,'tag','textElemLoads','UserData',qi);
                        else
                            text((x1+x2)/2 + shift, (y1+y2)/2 + shift, value, 'Color', clr,'tag','textElemLoads','UserData',qi);
                        end
                        
                        % If load is linear, draw initial and final load
                        % values
                    else
                        
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%.*f kN/m',dc,qi);
                            value2 = sprintf('%.*f kN/m',dc,qf);
                        else
                            value1 = sprintf('%.*f',dc,qi);
                            value2 = sprintf('%.*f',dc,qf);
                        end
                        
                        % Write initial value
                        if x1 == x2
                            text(x1 + shift, y1, value1, 'Color', clr,'tag','textElemLoads','UserData',qi);
                        elseif y1 == y2
                            text(x1, y1 + 1.5*shift, value1, 'Color', clr,'tag','textElemLoads','UserData',qi);
                        else
                            text(x1 + shift, y1 + shift, value1, 'Color', clr,'tag','textElemLoads','UserData',qi);
                        end
                        
                        % Write final value
                        if x1 == x2
                            text(x2 + shift, y2, value2, 'Color', clr,'tag','textElemLoads','UserData',qf);
                        elseif y1 == y2
                            text(x2, y2 + 1.5*shift, value2, 'Color', clr,'tag','textElemLoads','UserData',qf);
                        else
                            text(x2 + shift, y2 + shift, value2, 'Color', clr,'tag','textElemLoads','UserData',qf);
                        end
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws applied nodal loads and moments.
        function draw = nodalLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if nodal loads visualization is on
            if strcmp(get(mdata.viewNodalLoadsButton,'Checked'),'off')
                return
            end
            
            % Parameters
            shift = draw.size/100;  % distance between load symbol and nodal point
            al = draw.size/12;      % load symbol size (arrow lenght)
            ah = draw.size/60;      % load symbol size (arrowhead height)
            ab = draw.size/60;     % load symbol size (arrowhead base)
            tt = draw.size/200;     % distance between text and nodal point
            clr = [1,0,0];          % load color
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for n = 1:draw.drv.nnp
                % Check if current node has a nodal load
                if isempty(draw.drv.nodes(n).nodalLoad) == 0
                    % Get nodal coordinates
                    x = draw.drv.nodes(n).coord(1);
                    y = draw.drv.nodes(n).coord(2);
                    
                    % Get nodal load components
                    fz = draw.drv.nodes(n).nodalLoad(3);
                    mx = draw.drv.nodes(n).nodalLoad(4);
                    my = draw.drv.nodes(n).nodalLoad(5);
                    
                    % Draw load component in the Z direction
                    if fz > 0
                        draw.arrow3DPlaneProj(x,y,ab,'z+',clr,'drawNodalLoads')
                        hold on       
                    elseif fz < 0
                        draw.arrow3DPlaneProj(x,y,ab,'z-',clr,'drawNodalLoads')
                        hold on
                    end
                    
                    if fz ~= 0
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,fz);
                        else
                            value = sprintf('%.*f',dc,fz);
                        end
                        text(x + shift + tt, y + shift + tt, value, 'Color', clr,'tag','textNodalLoads','UserData',fz);
                    end
                    
                    % Draw moment component in the X direction 
                    if mx > 0
                        draw.arrow2D(x-shift,y,al,ah,ab,pi,clr,'drawNodalLoads');
                        hold on
                        draw.arrow2D(x-shift-0.25*al,y,0.75*al,ah,ab,pi,clr,'drawNodalLoads');
                        hold on
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mx));
                        else
                            value = sprintf('%.*f',dc,abs(mx));
                        end
                        text(x - shift - al - 6*tt, y + 3*tt, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(mx));
                        
                    elseif mx < 0
                        draw.arrow2D(x+shift,y,al,ah,ab,0,clr,'drawNodalLoads');
                        hold on
                        draw.arrow2D(x+shift+0.25*al,y,0.75*al,ah,ab,0,clr,'drawNodalLoads');
                        hold on
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mx));
                        else
                            value = sprintf('%.*f',dc,abs(mx));
                        end
                        text(x + shift + al, y + 3*tt, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(mx));
                    end
                    
                    % Draw moment component in the Y direction 
                    if my > 0
                        draw.arrow2D(x,y-shift,al,ah,ab,-pi/2,clr,'drawNodalLoads');
                        hold on
                        draw.arrow2D(x,y-shift-0.25*al,0.75*al,ah,ab,-pi/2,clr,'drawNodalLoads');
                        hold on
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(my));
                        else
                            value = sprintf('%.*f',dc,abs(my));
                        end
                        text(x + tt, y - shift - al - tt, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(my));
                        
                    elseif my < 0
                        draw.arrow2D(x,y+shift,al,ah,ab,pi/2,clr,'drawNodalLoads');
                        hold on
                        draw.arrow2D(x,y+shift+0.25*al,0.75*al,ah,ab,pi/2,clr,'drawNodalLoads');
                        hold on
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(my));
                        else
                            value = sprintf('%.*f',dc,abs(my));
                        end
                        text(x + tt, y + shift + al + tt, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(my));
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws nodal prescribed displacement representation.
        function draw = nodalPrescDispl(draw)
        end
        
        %------------------------------------------------------------------
        % Draws thermal load representation on elements.
        function draw = thermalLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if thermal loads visualization is on
            if strcmp(get(mdata.viewThermalLoadsButton,'Checked'),'off')
                return
            end
            
            % Parameters
            d = draw.size/150;     % distance between temperature grad symbol and element
            heatClr = [1,0,0];     % heat color
            coldClr = [0,0,1];     % cold color
            dc = getappdata(0,'decPrec');  % decimal precision
            
            for e = 1:draw.drv.nel
                if draw.drv.elems(e).load.tempVar_Z ~= 0
                    % Get element orientation angle cosine with axes X and Y
                    cx = draw.drv.elems(e).cosine_X;
                    cy = draw.drv.elems(e).cosine_Y;
                    
                    % Get nodal coordinates
                    x1 = draw.drv.elems(e).nodes(1).coord(1);
                    y1 = draw.drv.elems(e).nodes(1).coord(2);
                    x2 = draw.drv.elems(e).nodes(2).coord(1);
                    y2 = draw.drv.elems(e).nodes(2).coord(2);

                    % Get temperature variation values
                    dtz = draw.drv.elems(e).load.tempVar_Z;
                    
                    % Get thermal lines coords
                    X = [(x1 - d*(cx-cy)) (x1 - d*(cx+cy))...
                        (x2 + d*(cx-cy)) (x2 + d*(cx+cy))];
                    
                    Y = [(y1 - d*(cx+cy)) (y1 + d*(cx-cy))...
                        (y2 + d*(cx+cy)) (y2 - d*(cx-cy))];
                    
                    % Check if units are enabled
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        unitsAreOn = true;
                    else
                        unitsAreOn = false;
                    end
                    
                    % Draw temperature variation symbols
                    if dtz > 0
                        line(X([1 4]), Y([1 4]), 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        line(X([2 3]), Y([2 3]), 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text((x1+x2)/2 + d, (y1+y2)/2 + 2*d, sprintf('dTz = %.*f C',dc,dtz), 'Color', heatClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                            case false
                                text((x1+x2)/2 + d, (y1+y2)/2 + 2*d, sprintf('dTz = %.*f',dc,dtz), 'Color', heatClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                        end
                    elseif dtz < 0
                        line(X([1 4]), Y([1 4]), 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        line(X([2 3]), Y([2 3]), 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text((x1+x2)/2 + d, (y1+y2)/2 + 2*d, sprintf('dTz = %.*f C',dc,dtz), 'Color', coldClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                            case false
                                text((x1+x2)/2 + d, (y1+y2)/2 + 2*d, sprintf('dTz = %.*f',dc,dtz), 'Color', coldClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                        end
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Computes deformed configuration scale factor.
        function draw = deformScaleFactor(draw)
        end
        
        %------------------------------------------------------------------
        % Draws structure deformed configuration on a given scale.
        % Input arguments:
        %  scale: deformed configuration scale factor
        function draw = deformConfig(draw,~)
        end
        
        %------------------------------------------------------------------
        % Computes axial force diagram scale factor value.
        function draw = axialScaleFactor(draw)
        end
        
        %------------------------------------------------------------------
        % Draws resulting axial force diagram on a given scale.
        % Input arguments:
        %  scale: axial force diagram scale factor
        function draw = axialForce(draw,~)
        end
        
        %------------------------------------------------------------------
        % Draws resulting torsion moment diagram.
        function draw = torsionMoment(draw)
        end
        
        %------------------------------------------------------------------
        % Computes shear force diagram scale factor value in XY plane.
        function draw = shearScaleFactor_XY(draw)
        end
        
        %------------------------------------------------------------------
        % Draws resulting shear force diagram in XY plane on a given scale.
        % Input arguments:
        %  scale: shear force diagram scale factor
        function draw = shearForce_XY(draw,~)
        end
        
        %------------------------------------------------------------------
        % Computes shear force diagram scale factor value in XZ plane.
        function draw = shearScaleFactor_XZ(draw)
        end
        
        %------------------------------------------------------------------
        % Draws resulting shear force diagram in XZ plane on a given scale.
        % Input arguments:
        %  scale: shear force diagram scale factor
        function draw = shearForce_XZ(draw,~)
        end
        
        %------------------------------------------------------------------
        % Computes bending moment diagram scale factor value in XY plane.
        function draw = bendingMomentScaleFactor_XY(draw)
        end
        
        %------------------------------------------------------------------
        % Draws resulting bending moment diagram in XY plane on a given scale.
        % Input arguments:
        %  scale: bending moment diagram scale factor
        function draw = bendingMoment_XY(draw,~)
        end
        
        %------------------------------------------------------------------
        % Computes bending moment diagram scale factor value in XZ plane.
        function draw = bendingMomentScaleFactor_XZ(draw)
        end
        
        %------------------------------------------------------------------
        % Draws resulting bending moment diagram in XZ plane on a given scale.
        % Input arguments:
        %  scale: bending moment diagram scale factor
        function draw = bendingMoment_XZ(draw,~)
        end
        
        %------------------------------------------------------------------
        % Draws reactions indication next to nodal supports.
        function draw = reactions(draw)
        end
        
    end
end