%% Draw_Grillage class
%
% This is a sub-class, in the Object Oriented Programming (OOP) paradigm,
% of super-class <draw.html *Draw*> in the <main.html LESM (Linear Elements
% Structure Model)> program. This sub-class implements abstract methods,
% defined in super-class *Draw*, that deal with grillage analysis model
% of linear structure elements.
%
%% Authors
% Luiz Fernando Martha, Rafael Lopez Rangel and Pedro Cortez Lopes
%
%% Class definition
classdef Draw_Grillage < Draw
    %% Constructor method
    methods
        %------------------------------------------------------------------
        function draw = Draw_Grillage(drv)
            draw = draw@Draw(37.5,30);
            
            if (nargin > 0)
                draw.drv = drv;
            end
        end
    end
    
    %% Public methods
    % Implementation of the abstract methods declared in super-class <draw.html *Draw*>.
    methods
        %------------------------------------------------------------------
        % Sets viewpoint position and computes drawing size parameter
        % according to model dimensions.
        function draw = setSize(draw)
            
            x = zeros(draw.drv.nnp);
            y = zeros(draw.drv.nnp);
            z = zeros(draw.drv.nnp);
            
            for n = 1:draw.drv.nnp
                x(n) = draw.drv.nodes(n).coord(1);
                y(n) = draw.drv.nodes(n).coord(2);
                z(n) = draw.drv.nodes(n).coord(3);
            end
            
            xmin = min(x);
            xmax = max(x);
            ymin = min(y);
            ymax = max(y);
            zmin = min(z);
            zmax = max(z);
            
            dx = xmax - xmin;
            dy = ymax - ymin;
            dz = zmax - zmin;
            sz = max([dx,dy,dz]);
            if isempty(sz)
                sz = 0;
            end
            
            if sz == 0
                draw.size = 10;
            else
                draw.size = sz;
            end
        end
        
        %------------------------------------------------------------------
        % Sets axes limits according to model dimensions.
        function draw = setLimits(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Make sure that modifications will be made on canvas
            ax = mdata.axes_Canvas;
            ax.Clipping = 'off';
            
            if draw.drv.nnp > 0
                x = zeros(draw.drv.nnp,1);
                y = zeros(draw.drv.nnp,1);
                z = zeros(draw.drv.nnp,1);
                
                for n = 1:draw.drv.nnp
                    x(n) = draw.drv.nodes(n).coord(1);
                    y(n) = draw.drv.nodes(n).coord(2);
                    z(n) = draw.drv.nodes(n).coord(3);
                end
                
                xmin = min(x);
                xmax = max(x);
                ymin = min(y);
                ymax = max(y);
                zmin = min(z);
                zmax = max(z);
                
                dx = xmax - xmin;
                dy = ymax - ymin;
                dz = zmax - zmin;
                
                if (dx == 0) && (dy == 0) && (dz == 0)
                    xlim([x - 5, x + 5])
                    ylim([y - 5, y + 5])
                    zlim([0, 1])
                    
                elseif (dx >= dy) && (dx >= dz) ||...
                       (dy >= dx) && (dy >= dz) ||...
                       (dz >= dx) && (dz >= dy)
                    d = draw.size/5.2;
                    if xmax - xmin > 10
                        xlim([xmin - d, xmax + d])
                    else
                        xmean = (xmax + xmin)/2;
                        xlim([xmean - 5, xmean + 5])
                    end
                    if ymax - ymin > 10
                        ylim([ymin - d, ymax + d])
                    else
                        ymean = (ymax + ymin)/2;
                        ylim([ymean - 5, ymean + 5])
                    end
                    zlim([0, 1])
                    
                else
                    if xmax - xmin > 10
                        xlim([xmin - dx/5, xmax + dx/5])
                    else
                        xmean = (xmax + xmin)/2;
                        xlim([xmean - 5, xmean +5])
                    end
                    if ymax - ymin > 10
                        ylim([ymin - dy/5, ymax + dy/5])
                    else
                        ymean = (ymax + ymin)/2;
                        ylim([ymean - 5, ymean + 5])
                    end
                    zlim([0, 1])
                end
                
                offset = ((max(xlim)-min(xlim))+(max(ylim)-min(ylim))+(max(zlim)-min(zlim)))/50;
                xlabel('X', 'Position', [mean(xlim), min(ylim) - offset/3, min(zlim) - offset/2]);
                ylabel('Y', 'Position', [min(xlim) - offset, mean(ylim), min(zlim) - offset/2]);
                zlabel('Z', 'Position', [min(xlim) - offset, max(ylim) + offset, mean(zlim)], 'Rotation', pi/2);
                
            else
                xlim([-5,5])
                ylim([-5,5])
                zlim([0,1])
            end
        end
        
        %------------------------------------------------------------------
        % Draws structure model with nodes, elements, supports and  hinges.
        function draw = model(draw)
            draw.nodes();
            draw.elements();
        end
        
        %------------------------------------------------------------------
        % Draws elements with hinged or continuous ends.
        function draw = elements(draw)
            % Parameters
            r = draw.size/125;  % hinge symbol radius
            clr = [0,0,0];      % element color
            
            for e = 1:draw.drv.nel
                % Get nodes IDs and coordinates
                n1 = draw.drv.elems(e).nodes(1).id;
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                z1 = draw.drv.elems(e).nodes(1).coord(3);
                n2 = draw.drv.elems(e).nodes(2).id;
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Get element orientation angle cosine with X and Y axes
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
                
                % Get element incidence information on end nodes
                [tot_i,hng_i] = draw.drv.nodes(n1).elemsIncidence(draw.drv);
                [tot_f,hng_f] = draw.drv.nodes(n2).elemsIncidence(draw.drv);
                
                % Set element end coordinates
                xi = x1;
                yi = y1;
                xf = x2;
                yf = y2;
                
                % Set element initial coordinates by checking if there is a
                % hinge on nodal point position or on element end
                if (hng_i >= (tot_i - 1)) && (tot_i >= 2) &&...
                   ((draw.drv.nodes(n1).ebc(4) == 0) || (draw.drv.nodes(n1).ebc(5) == 0))
                    % Draw one hinge symbol representing the articulation
                    % between all elements
                    draw.sphere(x1, y1, z1, r, 'drawElements')
                    hold on
                    % One hinge on nodal point position:
                    xi = x1 + r * cx;
                    yi = y1 + r * cy;
                    
                elseif draw.drv.elems(e).hingei == 0
                    % Hinge on element end:
                    draw.sphere(x1 + r * cx, y1 + r * cy, z1, r, 'drawElements');
                    xi = x1 + 2 * r * cx;
                    yi = y1 + 2 * r * cy;
                end
                
                % Set element final coordinates by checking if there is a
                % hinge on nodal point position or on element end
                if (hng_f >= (tot_f - 1)) && (tot_f >= 2) &&...
                   ((draw.drv.nodes(n2).ebc(4) == 0) || (draw.drv.nodes(n2).ebc(5) == 0))
                    % Draw one hinge symbol representing the articulation
                    % between all elements
                    draw.sphere(x2, y2, z2, r, 'drawElements')
                    hold on
                    % One hinge on nodal point position:
                    xf = x2 - r * cx;
                    yf = y2 - r * cy;
                    
                else
                    if draw.drv.elems(e).hingef == 0
                        % Hinge on element end:
                        draw.sphere(x2 - r * cx, y2 - r * cy, z2, r, 'drawElements');
                        xf = x2 - 2 * r * cx;
                        yf = y2 - 2 * r * cy;
                    end
                end
                
                % Connect element end coordinates
                X = [xi, xf];
                Y = [yi, yf];
                Z = [z1, z2];
                plot3(X, Y, Z, 'Color', clr, 'tag', 'drawElements');
            end
        end
        
        %------------------------------------------------------------------
        % Draws nodal marks with support conditions.
        function draw = nodes(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Get flag for support visualization option status
            drawSupports =  get(mdata.viewSupportsButton,'Checked');
            
            include_constants;
            % Parameters
            nm = draw.size/230;    % node mark symbol (cube side)
            r = draw.size/125;     % hinge symbol radius
            ph = draw.size/40;     % translation constraint symbol (pyramid height)
            pb = draw.size/80;     % translation constraint symbol (pyramid base)
            cs = draw.size/75;     % rotation constraint symbol (cube side)
            sh = draw.size/20;     % spring symbol height
            sr = draw.size/105;    % rotational spring symbol radius
            nclr = [0,0,0];        % node and hinge color
            sclr = [0.6,0.6,0.6];  % support color
            sprclr = [0.6,0,0.4];  % spring color
            rotsclr = [0.7,0,0.5]; % rotational spring color
            dc = getappdata(0,'decPrec');  % decimal precision
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
                z = draw.drv.nodes(n).coord(3);
                
                % Get element incidence information
                [tot,hng] = draw.drv.nodes(n).elemsIncidence(draw.drv);
                
                % Distance between translation constraint support symbol
                % and nodal point.
                shift = 0;
                
                if (hng >= (tot - 1)) && (tot >= 2) &&...
                   ((draw.drv.nodes(n).ebc(4) == FREE_DOF) || (draw.drv.nodes(n).ebc(5) == FREE_DOF))
                    shift = r;
                    % Draw a nodal mark
                    draw.cube(x, y, z, nm, nclr,'drawNodes');
                    hold on
                    
                elseif (draw.drv.nodes(n).ebc(4) == FIXED_DOF) && (draw.drv.nodes(n).ebc(5) == FIXED_DOF)
                    if strcmp(drawSupports,'on')
                        % Draw rotation constraint support
                        draw.cube(x, y, z, cs, sclr,'drawSupports');
                        hold on
                        shift = cs/2;
                    end
                    
                elseif (draw.drv.nodes(n).ebc(4) == SPRING_DOF) && (draw.drv.nodes(n).ebc(5) == SPRING_DOF)
                    if strcmp(drawSupports,'on')
                        % Draw rotational spring
                        draw.rotSpring_3D(x,y,z,sr,rotsclr,'drawSupports');
                        hold on
                        shift = sr;

                        % Get spring stiffness
                        kr = draw.drv.nodes(n).springStiff(4);
                        % Write spring stiffness value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            if kr >= 1000
                                value = sprintf('%.*e kNm/rad',dc,kr);
                            else
                                value = sprintf('%.*f kNm/rad',dc,kr);
                            end
                        else
                            if kr >= 1000
                                value = sprintf('%.*e',dc,kr);
                            else
                                value = sprintf('%.*f',dc,kr);
                            end
                        end
                        
                        text(x + shift*0.8, y - shift*0.8, z + shift*1.1, value,...
                             'Color', sprclr,'Fontsize',8.5,'tag','textRotSprings','UserData',kr);
                    end
                elseif (draw.drv.nodes(n).ebc(3) == FREE_DOF) && ...
                       ((draw.drv.nodes(n).ebc(4) == FREE_DOF) || (draw.drv.nodes(n).ebc(5) == FREE_DOF))
                    % Draw a nodal mark representing a free node
                    draw.cube(x, y, z, nm, nclr,'drawNodes');
                    hold on
                end
                
                % Check if there is a translation constraint in the
                % direction of axis Z and draw support
                if draw.drv.nodes(n).ebc(3) == FIXED_DOF && strcmp(drawSupports,'on')
                    draw.pyramid(x, y, z - shift, ph, pb, 'z+', sclr,'drawSupports');
                end
                
                % Check if there is a displacement spring in the direction
                % of axis Z and draw spring
                if draw.drv.nodes(n).ebc(3) == SPRING_DOF && strcmp(drawSupports,'on')
                    draw.SpringZ_3D(x,y,z-shift,sh,sprclr,'drawSupports');
                    hold on

                    % Get spring stiffness
                    kz = draw.drv.nodes(n).springStiff(3);
                    % Write spring stiffness value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        if kz >= 1000
                        value = sprintf('%.*e kN/m',dc,kz);
                        else
                        value = sprintf('%.*f kN/m',dc,kz);
                        end
                    else
                        if kz >= 1000
                        value = sprintf('%.*e',dc,kz);
                        else
                        value = sprintf('%.*f',dc,kz);
                        end
                    end
                    
                    text(x + sh*0.1, y - sh*0.1, z - 1.2*sh - shift, value, 'Color', sprclr,'Fontsize',8.5,'tag','textSprings','UserData',kz);
                end    
            end
        end
        
        %------------------------------------------------------------------
        % Computes element loads scale factor.
        function scl = elemLoadsScaleFactor(draw)
            max_elem = zeros(1,draw.drv.nel);
            
            for e = 1:draw.drv.nel
                % Initialize load values on element ends
                qi = 0;
                qf = 0;
                
                % Add uniform load contribtuion
                if isempty(draw.drv.elems(e).load.uniformGbl) == 0
                    qz = draw.drv.elems(e).load.uniformGbl(3);
                    qi = qi + qz;
                    qf = qf + qz;
                end
                
                % Add linear load contribtuion
                if isempty(draw.drv.elems(e).load.linearGbl) == 0
                    qzi = draw.drv.elems(e).load.linearGbl(3);
                    qzf = draw.drv.elems(e).load.linearGbl(6);
                    qi = qi + qzi;
                    qf = qf + qzf;
                end
                
                % Get maximum load value on current element
                max_elem(e) = max(abs(qi),abs(qf));
            end
            
            % Get maximum load value on model
            max_val = max(max_elem);
            
            % Calculate scale factor
            if max_val ~= 0
                scl = draw.size/(12*max_val);
            else
                scl = 0;
            end
            
            setappdata(0,'load_sf',scl);
        end
        
        %------------------------------------------------------------------
        % Draws element distributed loads (uniform and linear).
        function draw = elemLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if elem loads visualization is on
            if strcmp(get(mdata.viewDistribLoadsButton,'Checked'),'off')
                return
            end
            
            % Parameters
            shift = draw.size/130;  % distance between load symbol and element
            ph = draw.size/200;     % load symbol size (pyramid height)
            pb = draw.size/300;     % load symbol size (pyramid base)
            t1 = draw.size/120;     % text position parameter
            t2 = draw.size/40;      % text position parameter
            t3 = draw.size/80;      % text position parameter
            clr = [1,0,0];          % load symbol color
            scl = getappdata(0,'load_sf');                 % Load symbol scale
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for e = 1:draw.drv.nel
                if ((isempty(draw.drv.elems(e).load.uniformGbl) == 0) &&...
                   (all(draw.drv.elems(e).load.uniformGbl == 0)==0)) ||...
                   ((isempty(draw.drv.elems(e).load.linearGbl) == 0) &&...
                   (all(draw.drv.elems(e).load.linearGbl == 0)==0))
               
                    % Get element length
                    L = draw.drv.elems(e).length;
                    
                    % Get element orientation angle cosine with axes X and Y
                    cx = draw.drv.elems(e).cosine_X;
                    cy = draw.drv.elems(e).cosine_Y;
                    
                    % Get element end nodes IDs
                    n1 = draw.drv.elems(e).nodes(1).id;
                    n2 = draw.drv.elems(e).nodes(2).id;
                    
                    % Get nodal coordinates
                    x1 = draw.drv.nodes(n1).coord(1);
                    y1 = draw.drv.nodes(n1).coord(2);
                    z1 = draw.drv.nodes(n1).coord(3);
                    x2 = draw.drv.nodes(n2).coord(1);
                    y2 = draw.drv.nodes(n2).coord(2);
                    z2 = draw.drv.nodes(n2).coord(3);
                    
                    % Initialize load values on element ends
                    qi = 0;
                    qf = 0;
                    
                    % Add uniform load contribtuion
                    if isempty(draw.drv.elems(e).load.uniformGbl) == 0
                        qz = draw.drv.elems(e).load.uniformGbl(3);
                        qi = qi + qz;
                        qf = qf + qz;
                    end
                    
                    % Add linear load contribtuion
                    if isempty(draw.drv.elems(e).load.linearGbl) == 0
                        qzi = draw.drv.elems(e).load.linearGbl(3);
                        qzf = draw.drv.elems(e).load.linearGbl(6);
                        qi = qi + qzi;
                        qf = qf + qzf;
                    end
                    
                    % Calculate load quation coefficients:
                    % q(x) = Ax + B
                    A = (qf - qi)/L;
                    B = qi;
                    
                    % Draw load symbol on a number cross-sections along
                    % element local axis X
                    step = L / round(20 * L/draw.size);
                    for x = 0:step:L
                        % Calculate current cross-section coordinates
                        xs = x1 + x * cx;
                        ys = y1 + x * cy;
                        zs = z1;
                        
                        % Calculate load value on current cross-section
                        q = A * x + B;
                        
                        % Draw load symbol on current cross-section
                        if abs(scl * q) >= ph
                            if q > 0
                                draw.arrow3D(draw, xs, ys, zs - shift, scl * q, ph, pb, 'z+', clr,'drawElemLoads');
                            elseif q < 0
                                draw.arrow3D(draw, xs, ys, zs + shift, -scl * q, ph, pb, 'z-', clr,'drawElemLoads');
                            end
                        end
                    end
                    
                    % Connect load symbols
                    if  (qi < 0) && (qf > 0)
                        x0 = (abs(qi)*(L))/(abs(qf)+abs(qi)); 
                        [xu,yu,zu] = draw.coordTransf3D(x0, 0, shift, x1, y1, z1, e);
                        [xd,yd,zd] = draw.coordTransf3D(x0, 0, -shift, x1, y1, z1, e);
                        
                        X = [x1, xu, xd, x2];
                        Y = [y1, yu, yd, y2];
                        Z = [z1 + shift - scl * qi, zu, zd, z2 - shift - scl * qf];
                        line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                    elseif (qi > 0) && (qf < 0)
                        x0 = (abs(qi)*(L))/(abs(qf)+abs(qi));
                        [xu,yu,zu] = draw.coordTransf3D(x0, 0, shift, x1, y1, z1, e);
                        [xd,yd,zd] = draw.coordTransf3D(x0, 0, -shift, x1, y1, z1, e);
                        
                        X = [x1, xd, xu, x2];
                        Y = [y1, yd, yu, y2];
                        Z = [z1 - shift - scl * qi, zd, zu, z2 + shift - scl * qf];
                        line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                    elseif (qi~=0) || (qf~=0)
                        X = [x1, x2];
                        Y = [y1, y2];
                        
                        if qi >= 0 && qf >= 0
                            Z = [z1 - shift - scl * qi, z2 - shift - scl * qf];
                        elseif qi <= 0 && qf <= 0
                            Z = [z1 + shift - scl * qi, z2 + shift - scl * qf];
                        end
                        
                        line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                    end
                    
                    % Write load values:
                    
                    % If load is uniform, draw load value in the middle of
                    % the element
                    if qi == qf
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN/m',dc,abs(qi));
                        else
                            value = sprintf('%.*f',dc,abs(qi));
                        end
                        
                        if qi > 0
                            text((x1+x2)/2, (y1+y2)/2, z1 - shift - scl * qi - t1, value, 'Color', clr,'tag','textElemLoads','UserData',abs(qi));
                        elseif qi < 0
                            text((x1+x2)/2, (y1+y2)/2, z1 + shift - scl * qi + 3 * t1, value, 'Color', clr,'tag','textElemLoads','UserData',abs(qi));
                        end
                        
                        % If load is linear, draw initial and final load
                        % values
                    else
                        % Get plotting coordinates on plane XY
                        xl1 = x1 + t2 * cx;
                        yl1 = y1 + t2 * cy;
                        xl2 = x2 - t2 * cx;
                        yl2 = y2 - t2 * cy;
                        
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%.*f kN/m',dc,abs(qi));
                            value2 = sprintf('%.*f kN/m',dc,abs(qf));
                        else
                            value1 = sprintf('%.*f',dc,abs(qi));
                            value2 = sprintf('%.*f',dc,abs(qf));
                        end
                        
                        % Write initial value
                        if qi > 0
                            text(xl1, yl1, z1 - scl * qi - t3, value1, 'Color', clr,'tag','textElemLoads','UserData',abs(qi));
                        elseif qi < 0
                            text(xl1, yl1, z1 - scl * qi + 2 * t3, value1, 'Color', clr,'tag','textElemLoads','UserData',abs(qi));
                        end
                        
                        % Write final value
                        if qf > 0
                            text(xl2, yl2, z2 - scl * qf - t3, value2, 'Color', clr,'tag','textElemLoads','UserData',abs(qf));
                        elseif qf < 0
                            text(xl2, yl2, z2 - scl * qf + 2 * t3, value2, 'Color', clr,'tag','textElemLoads','UserData',abs(qf));
                        end
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws applied nodal loads and moments.
        function draw = nodalLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if nodal loads visualization is on
            if strcmp(get(mdata.viewNodalLoadsButton,'Checked'),'off')
                return
            end
            
            % Parameters
            shift = draw.size/100;  % distance between load symbol and nodal point
            al = draw.size/20;      % load symbol size (arrow lenght)
            ah = draw.size/60;      % load symbol size (arrowhead height)
            ab = draw.size/100;     % load symbol size (arrowhead base)
            tt = draw.size/200;     % distance between text and nodal point
            clr = [1,0,0];          % load color
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for n = 1:draw.drv.nnp
                % Check if current node has a nodal load
                if isempty(draw.drv.nodes(n).nodalLoad) == 0
                    % Get nodal coordinates
                    x = draw.drv.nodes(n).coord(1);
                    y = draw.drv.nodes(n).coord(2);
                    z = draw.drv.nodes(n).coord(3);
                    
                    % Get nodal load components
                    fz = draw.drv.nodes(n).nodalLoad(3);
                    mx = draw.drv.nodes(n).nodalLoad(4);
                    my = draw.drv.nodes(n).nodalLoad(5);
                    
                    % Draw load component in the Z direction
                    if fz > 0
                        draw.arrow3D(draw, x, y, z - shift, 1.5*al, ah, ab, 'z+', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fz));
                        else
                            value = sprintf('%.*f',dc,abs(fz));
                        end
                        text(x, y, z - shift - 1.5*al - tt, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fz));
                        
                    elseif fz < 0
                        draw.arrow3D(draw, x, y, z + shift, 1.5*al, ah, ab, 'z-', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fz));
                        else
                            value = sprintf('%.*f',dc,abs(fz));
                        end
                        text(x, y, z + shift + 1.5*al + tt, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fz));
                    end
                    
                    % Draw moment component in the X direction 
                    if mx > 0
                        draw.moment3D(draw, x - shift, y, z, 1.5 * al, ah, ab, 'x+', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mx));
                        else
                            value = sprintf('%.*f',dc,abs(mx));
                        end
                        text(x - shift - al - 6 * tt, y, z + 3 * tt, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(mx));
                        
                    elseif mx < 0
                        draw.moment3D(draw, x + shift, y, z, 1.5 * al, ah, ab, 'x-', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mx));
                        else
                            value = sprintf('%.*f',dc,abs(mx));
                        end
                        text(x + shift + al + 6 * tt, y, z + 3 * tt, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(mx));
                    end
                    
                    % Draw moment component in the Y direction 
                    if my > 0
                        draw.moment3D(draw, x, y - shift, z, 1.5 * al, ah, ab, 'y+', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(my));
                        else
                            value = sprintf('%.*f',dc,abs(my));
                        end
                        text(x, y - shift - al - 6 * tt, z + 3 * tt, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(my));
                        
                    elseif my < 0
                        draw.moment3D(draw, x, y + shift, z, 1.5 * al, ah, ab, 'y-', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(my));
                        else
                            value = sprintf('%.*f',dc,abs(my));
                        end
                        text(x, y + shift + al + 6 * tt, z + 3 * tt, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(my));
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws nodal prescribed displacement representation.
        function draw = nodalPrescDispl(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if presc displ visualization is on
            if strcmp(get(mdata.viewPrescDisplButton,'Checked'),'off')
                return
            end
            
            % Parameters
            shift = draw.size/100;  % distance between presc. displ. symbol and support
            ph = draw.size/40;      % translation constraint symbol (pyramid height)
            al = draw.size/20;      % presc. displ. symbol size (arrow length)
            ah = draw.size/80;      % presc. displ. symbol size (pyramid height)
            ab = draw.size/130;     % presc. displ. symbol size (pyramid base)
            tt = draw.size/200;     % distance between text and nodal point
            clr = [1,0,1];          % presc. displ. symbol color
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for n =1:draw.drv.nnp
                % Check if current node has a prescribed displacement
                if isempty(draw.drv.nodes(n).prescDispl) == 0
                    % Get nodal coordinates
                    x = draw.drv.nodes(n).coord(1);
                    y = draw.drv.nodes(n).coord(2);
                    z = draw.drv.nodes(n).coord(3);
                    
                    % Get prescribed displacement component values and
                    % convert it to milimeter and radian
                    rx = draw.drv.nodes(n).prescDispl(4);
                    ry = draw.drv.nodes(n).prescDispl(5);
                    dz = 1000 * draw.drv.nodes(n).prescDispl(3);
                    
                    % Initialize rotation constraint symbol (cube side)
                    cs = 0;
                    
                    % Check if rotation is really fixed and draw prescribed
                    % displacement indication
                    if (draw.drv.nodes(n).ebc(4) == 1) && (draw.drv.nodes(n).ebc(5)== 1)
                        % Update rotation constraint symbol size value
                        cs = draw.size/100;
                        
                        if rx > 0
                            draw.moment3D(draw, x - cs - shift, y, z, 1.5 * al, ah, ab, 'x+', clr,'drawPrescDispl');
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f rad',dc,rx);
                            else
                                value = sprintf('%.*f',dc,rx);
                            end
                            text(x - cs - shift - al - 6 * tt, y, z + 3 * tt, value, 'Color', clr,'tag','textPrescRot','UserData',abs(rx));
                            
                        elseif rx < 0
                            draw.moment3D(draw, x + cs + shift, y, z, 1.5 * al, ah, ab, 'x-', clr,'drawPrescDispl');
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f rad',dc,abs(rx));
                            else
                                value = sprintf('%.*f',dc,abs(rx));
                            end
                            text(x + cs + shift + al + 6 * tt, y, z + 3 * tt, value, 'Color', clr,'tag','textPrescRot','UserData',abs(rx));
                        end
                        
                        if ry > 0
                            draw.moment3D(draw, x, y - cs - shift, z, 1.5 * al, ah, ab, 'y+', clr,'drawPrescDispl');
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f rad',dc,ry);
                            else
                                value = sprintf('%.*f',dc,ry);
                            end
                            text(x, y - cs - shift - al - 6 * tt, z + 3 * tt, value, 'Color', clr,'tag','textPrescRot','UserData',abs(ry));
                            
                        elseif ry < 0
                            draw.moment3D(draw, x, y + cs + shift, z, 1.5 * al, ah, ab, 'y-', clr,'drawPrescDispl');
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f rad',dc,abs(ry));
                            else
                                value = sprintf('%.*f',dc,abs(ry));
                            end
                            text(x, y + cs + shift + al + 6 * tt, z + 3 * tt, value, 'Color', clr,'tag','textPrescRot','UserData',abs(ry));
                        end
                    end
                    
                    % Check if translation in the Z axis is really fixed and
                    % draw prescribed displacement indication
                    if draw.drv.nodes(n).ebc(3) == 1
                        if dz ~= 0
                            if dz > 0
                                draw.arrow3D(draw, x, y, z - cs - shift - ph, al, ah, ab, 'z+', clr,'drawPrescDispl');
                            elseif dz < 0
                                draw.arrow3D(draw, x, y, z - cs - shift - ph - al, al, ah, ab, 'z-', clr,'drawPrescDispl');
                            end
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f mm',dc,abs(dz));
                            else
                                value = sprintf('%.*f',dc,abs(dz));
                            end
                            text(x, y, z - cs - shift - ph - al, value, 'Color', clr,'tag','textPrescDispl','UserData',abs(dz));
                        end
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws thermal load representation on elements.
        function draw = thermalLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if thermal loads visualization is on
            if strcmp(get(mdata.viewThermalLoadsButton,'Checked'),'off')
                return
            end
            
            % Parameters
            d = draw.size/150;     % distance between temperature grad symbol and element
            heatClr = [1,0,0];     % heat color
            coldClr = [0,0,1];     % cold color
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for e = 1:draw.drv.nel
                if draw.drv.elems(e).load.tempVar_Z ~= 0
                    % Get nodal coordinates
                    x1 = draw.drv.elems(e).nodes(1).coord(1);
                    y1 = draw.drv.elems(e).nodes(1).coord(2);
                    z1 = draw.drv.elems(e).nodes(1).coord(3);
                    x2 = draw.drv.elems(e).nodes(2).coord(1);
                    y2 = draw.drv.elems(e).nodes(2).coord(2);
                    z2 = draw.drv.elems(e).nodes(2).coord(3);

                    % Get temperature variation values
                    dtz = draw.drv.elems(e).load.tempVar_Z;
                    
                    % Check if units are enabled
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        unitsAreOn = true;
                    else
                        unitsAreOn = false;
                    end
                    
                    % Draw temperature variation symbols
                    if dtz > 0
                        line([x1,x2], [y1,y2], [z1+d,z2+d], 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        line([x1,x2], [y1,y2], [z1-d,z2-d], 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text((x1+x2)/2, (y1+y2)/2, d, sprintf('dTz = %.*f C',dc,dtz), 'Color', heatClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                            case false
                                text((x1+x2)/2, (y1+y2)/2, d, sprintf('dTz = %.*f',dc,dtz), 'Color', heatClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                        end
                    elseif dtz < 0
                        line([x1,x2], [y1,y2], [z1+d,z2+d], 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        line([x1,x2], [y1,y2], [z1-d,z2-d], 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text((x1+x2)/2, (y1+y2)/2, d, sprintf('dTz = %.*f C',dc,dtz), 'Color', coldClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                            case false
                                text((x1+x2)/2, (y1+y2)/2, d, sprintf('dTz = %.*f',dc,dtz), 'Color', coldClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                        end
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Plots ID number of nodes.
        function draw = nodeID(draw)
            % Parameters
            tx = draw.size/120;  % distance between text and nodal point (axis X)
            ty = draw.size/120;  % distance between text and nodal point (axis Y)
            tz = draw.size/80;   % distance between text and nodal point (axis Z)
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
                
                % Write ID number
                number = sprintf('%d',n);
                if (draw.drv.nodes(n).ebc(4) == 1) && (draw.drv.nodes(n).ebc(5) == 1)
                    text(x + tx, y + ty, 2 * tz, number, 'FontWeight', 'bold', 'tag', 'textNodeID');
                else
                    text(x + tx, y + ty, tz, number, 'FontWeight', 'bold', 'tag', 'textNodeID');
                end
            end
        end
        
        %------------------------------------------------------------------
        % Plots ID number of elements.
        function draw = elementID(draw)
            % Parameters
            te = draw.size/80;  % distance between text and element
            
            for e = 1:draw.drv.nel
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                
                % Write element ID number
                number = sprintf('%d',e);
                text((x1+x2)/2, (y1+y2)/2, te, number, 'FontWeight', 'bold', 'tag', 'textElemID');
            end
        end
        
        %------------------------------------------------------------------
        % Draws element orientation indication from inital to final node.
        function draw = elementOrientation(draw)
            % Parameters
            t = 1.3;             % distance between text and symbol
            clr = [0,.7,0];      % orientation symbol color
            
            for e = 1:draw.drv.nel
                % Calculate spear length
                l = draw.size/20;
                
                % Get nodal coordinates
                xi = draw.drv.elems(e).nodes(1).coord(1);
                yi = draw.drv.elems(e).nodes(1).coord(2);
                zi = draw.drv.elems(e).nodes(1).coord(3);
                xf = draw.drv.elems(e).nodes(2).coord(1);
                yf = draw.drv.elems(e).nodes(2).coord(2);
                zf = draw.drv.elems(e).nodes(2).coord(3);
                
                % Calculate element local axis X orientation vector
                x = [xf-xi, yf-yi, zf-zi];
                x = l * x / norm(x);
                
                % Get orientation vector on the local XZ plane
                z = [draw.drv.elems(e).vz(1), draw.drv.elems(e).vz(2), draw.drv.elems(e).vz(3)];
                
                % Calculate element local axis Y orientation vector
                y = cross(z,x);
                y = l * y / norm(y);
                
                % Calculate element local axis Z orientation vector
                z = cross(x,y);
                z = l * z/norm(z);
                
                % Draw orientation symbol
                xm = (xi + xf)/2;
                ym = (yi + yf)/2;
                zm = (zi + zf)/2;
                
                X = [xm, xm + x(1)];
                Y = [ym, ym + x(2)];
				Z = [zm, zm + x(3)];
                line(X, Y, Z, 'Color', clr, 'tag', 'drawElemOrient');
                text(xm + t * x(1), ym + t * x(2), zm + t * x(3), 'x', 'Color', clr, 'FontWeight', 'bold', 'tag', 'drawElemOrient');
                
                X = [xm, xm + y(1)];
                Y = [ym, ym + y(2)];
				Z = [zm, zm + y(3)];
                line(X, Y, Z, 'Color', clr, 'tag', 'drawElemOrient');
                text(xm + t * y(1), ym + t * y(2), zm + t * y(3), 'y', 'Color', clr, 'FontWeight', 'bold', 'tag', 'drawElemOrient');
                
                X = [xm, xm + z(1)];
                Y = [ym, ym + z(2)];
				Z = [zm, zm + z(3)];
                line(X, Y, Z, 'Color', clr, 'tag', 'drawElemOrient');
                text(xm + t * z(1), ym + t * z(2), zm + t * z(3), 'z', 'Color', clr, 'FontWeight', 'bold', 'tag', 'drawElemOrient');
            end
        end
        
        %------------------------------------------------------------------
        % Computes deformed configuration scale factor.
        function draw = deformScaleFactor(draw)
            mdata = guidata(findobj('Tag','GUI_Main'));
            sliderm = get(mdata.slider_Scale,'Max');
            
            % Estimate maximum displacement of each element
            % (nodal and internal)
            m = zeros(1,draw.drv.nel);
            for e = 1:draw.drv.nel
                % Get element end nodes IDs
                n1 = draw.drv.elems(e).nodes(1).id;
                n2 = draw.drv.elems(e).nodes(2).id;
                
                % Get maximum nodal displacements
                dz1 = draw.drv.D(draw.drv.ID(3,n1));
                dz2 = draw.drv.D(draw.drv.ID(3,n2));
                nodeDispl = [dz1,dz2];
                maxNode = max(abs(nodeDispl));
                
                % Get maximum estimated internal displacement
                maxInt = max(max(abs(draw.drv.elems(e).intDispl)));
                
                % Get maximum element displacement
                m(e) = max(maxNode,maxInt);
            end
            
            % Get maximum estimated model displacement
            max_disp = max(m);
            
            % Set adapted scale value
            dsf = draw.size/(5*sliderm*max_disp);
            if dsf == inf
                dsf = 0;
            end
            setappdata(0,'deform_sf',dsf);
        end
        
        %------------------------------------------------------------------
        % Draws structure deformed configuration on a given scale.
        % Input arguments:
        %  scale: deformed configuration scale factor
        function draw = deformConfig(draw,scale)
            % Parameters
            clr = [1,0,0];  % deformed configuration line color
            
            % Calculate deformed configuration coordinates of 50 cross-
            % sections along element local axis X and connect them
            for e = 1:draw.drv.nel
                % Get 50 cross-sections coordinates
                coords = draw.drv.elems(e).intCoords;
                
                % Get element axial and transversal internal displacements
                % in local system
                dl = draw.drv.elems(e).intDispl;
                
                % Deformed configuration global coordinates
                dfg = coords(3,:) + scale * dl(2,:);
                
                % Plot deformed configuration
                line(coords(1,:), coords(2,:), dfg, 'Color', clr, 'tag', 'drawDeformConfig');
            end
        end
        
        %------------------------------------------------------------------
        % Computes axial force diagram scale factor value.
        % NOT USED
        function draw = axialScaleFactor(draw)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Draws resulting axial force diagram on a given scale.
        % NOT USED
        function draw = axialForce(draw,~)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Anm.
        end
        
        %------------------------------------------------------------------
        % Draws resulting torsion moment diagram.
        function draw = torsionMoment(draw)
            % Parameters
            d = draw.size/40;  % distance between text and element
            clr = [1,0,0];         % diagram color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            % Get target element ID
            elem = get(mdata.popupmenu_ElementResults,'Value') - 1;
            if elem ~= 0
                i = elem;
                j = i;
            else
                i = 1;
                j = draw.drv.nel;
            end
            
            for e = i:j
                % Get element internal tortion moment value (always uniform
                % for grillage models) and convert it to string
                T = -draw.drv.elems(e).torsion_moment(1);
                if T ~= 0
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%+.*f kNm',dc,T);
                    else
                        value = sprintf('%+.*f',dc,T);
                    end
                else
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%.*f kNm',dc,abs(T));
                    else
                        value = sprintf('%.*f',dc,abs(T));
                    end
                    T = abs(T);
                end
                
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                
                % Write tortion moment value above the element
                text((x1+x2)/2, (y1+y2)/2, d, value, 'Color', clr, 'tag', 'textTorsionDiagram','UserData',T);
            end
        end
        
        %------------------------------------------------------------------
        % Computes shear force diagram scale factor value in XY plane.
        % NOT USED
        function draw = shearScaleFactor_XY(draw)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Draws resulting shear force diagram in XY plane on a given scale.
        % NOT USED
        function draw = shearForce_XY(draw,~)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Computes shear force diagram scale factor value in XZ plane.
        function draw = shearScaleFactor_XZ(draw)
            mdata = guidata(findobj('Tag','GUI_Main'));
            sliderm = get(mdata.slider_Scale,'Max');
            
            % Get maximum internal shear force value of each element
            max_elem = zeros(1,draw.drv.nel);
            for e = 1:draw.drv.nel
                % Get maximum value at element ends
                Q1_Z = draw.drv.elems(e).shear_force_Z(1);
                Q2_Z = draw.drv.elems(e).shear_force_Z(2);
                max_end = max(abs(Q1_Z),abs(Q2_Z));
                
                % Get maximum internal value
                if ~isempty(draw.drv.elems(e).maxShearForce_XZ)
                    max_int = abs(draw.drv.elems(e).maxShearForce_XZ(1));
                else
                    max_int = 0;
                end
                
                max_elem(e) = max(max_end,max_int);
            end
            
            % Get maximum shear internal force value of model
            max_val = max(max_elem);
            
            % Set adapted scale value
            if max_val == 0
                ssf = 0;
            else
                ssf = draw.size/(2.5*sliderm*max_val);
            end
            setappdata(0,'shearXZ_sf',ssf);
        end
        
        %------------------------------------------------------------------
        % Draws resulting shear force diagram in XZ plane on a given scale.
        % Input arguments:
        %  scale: shear force diagram scale factor
        function draw = shearForce_XZ(draw,scale)
            include_constants;
            
            % Parameters
            d = draw.size/80;  % distance between force value and diagram
            clr = [1,0,0];     % diagram line color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            % Get target element ID
            elem = get(mdata.popupmenu_ElementResults,'Value') - 1;
            if elem ~= 0
                i = elem;
                j = i;
            else
                i = 1;
                j = draw.drv.nel;
            end
            
            for e = i:j
                % Get element length
                L = draw.drv.elems(e).length;
                
                % Get element orientation angle cosine with axes X and Y
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
                
                % Get element internal shear force value at both ends
                Q1 = draw.drv.elems(e).shear_force_Z(1);
                Q2 = -draw.drv.elems(e).shear_force_Z(2);
                
                % Avoid plotting garbage
                if abs(Q1) < 1e-10
                    Q1 = 0;
                end
                if abs(Q2) < 1e-10
                    Q2 = 0;
                end
                
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                z1 = draw.drv.elems(e).nodes(1).coord(3);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Calculate diagram beggining coordinates
                zd1 = z1 + scale * Q1;
                zd2 = z2 + scale * Q2;
                
                % Adjust axes limits to fit diagram extreme values
                lim = max(abs(zd1),abs(zd2));
                if lim > 5
                    zlim([-lim lim]);
                end
                
                % Draw diagram extremities
                Xi = [x1, x1];
                Yi = [y1, y1];
                Zi = [z1, zd1];
                line(Xi, Yi, Zi, 'Color', clr, 'tag', 'drawShearForceXZDiagram');
                
                Xf = [x2, x2];
                Yf = [y2, y2];
                Zf = [z2, zd2];
                line(Xf, Yf, Zf, 'Color', clr, 'tag', 'drawShearForceXZDiagram');
                
                % Write force values
                if (abs(Q1-Q2) < 10e-10) && (isempty(draw.drv.elems(e).load.linearLcl) == 1)
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%+.*f kN',dc,Q1);
                    else
                        value = sprintf('%+.*f',dc,Q1);
                    end
                    
                    if Q1 > 10e-10
                        text((x1 + x2)/2, (y1 + y2)/2, zd1 + d, value, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Q1);
                    elseif Q1 < -10e-10
                        text((x1 + x2)/2, (y1 + y2)/2, zd1 - d, value, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Q1);
                    end
                    
                else
                    if Q1 > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%+.*f kN',dc,Q1);
                        else
                            value1 = sprintf('%+.*f',dc,Q1);
                        end
                        text(x1, y1, zd1 + d, value1, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Q1);
                        
                    elseif Q1 < -10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%+.*f kN',dc,Q1);
                        else
                            value1 = sprintf('%+.*f',dc,Q1);
                        end
                        text(x1, y1, zd1 - d, value1, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Q1);
                    end
                    
                    if Q2 > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value2 = sprintf('%+.*f kN',dc,Q2);
                        else
                            value2 = sprintf('%+.*f',dc,Q2);
                        end
                        text(x2, y2, zd2 + d, value2, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Q2);
                        
                    elseif Q2 < -10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value2 = sprintf('%+.*f kN',dc,Q2);
                        else
                            value2 = sprintf('%+.*f',dc,Q2);
                        end
                        text(x2, y2, zd2 - d, value2, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Q2);
                    end
                end
                
                % Connect diagram extremities:
                % Check if element has distributed load.
                % -If so, calculate shear force value along element length
                % -If not, connect both ends with a straight line
                if (isempty(draw.drv.elems(e).load.uniformLcl) == 0) || ...
                   (isempty(draw.drv.elems(e).load.linearLcl) == 0)
               
                    % Get element 50 point division coords and internal
                    % stress values
                    coords = draw.drv.elems(e).intCoords;
                    Q = draw.drv.elems(e).intStresses(1,:);
                    Qmax = draw.drv.elems(e).maxShearForce_XZ;
                    
                    % Avoid plotting numeric garbage
                    if ~all(abs(Q) < 10^-10)
                    
                        % Get element basis transformation matrix
                        rot = draw.drv.elems(e).T;

                        % Compute diagram coordinates
                        diagramCoords = rot' * [zeros(1,size(Q,2)); zeros(1,size(Q,2)); scale*Q] + coords;

                        % Plot diagram
                        X = diagramCoords(1,:);
                        Y = diagramCoords(2,:);
                        Z = diagramCoords(3,:);
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawShearForceXZDiagram');

                        % Check if there is a maximum value within the diagram
                        if ~isempty(Qmax)
                            % Compute maximum stress value position, in global
                            % coordinates
                            QmaxGblCoords = [x1 + Qmax(2) * cx;
                                             y1 + Qmax(2) * cy;
                                             z1 + Qmax(2) * cz];

                            % Compute max point in diagram and text coordinates
                            maxPointCoords = rot' * [0; 0; scale*Qmax(1)] + QmaxGblCoords;
                            maxTextCoords = rot' * [0; 0; scale*Qmax(1) + d*(Qmax(1)/abs(Qmax(1)))] + QmaxGblCoords;

                            % Plot point indicating maximum value
                            xp = maxPointCoords(1);
                            yp = maxPointCoords(2);
                            zp = maxPointCoords(3);
                            scatter3(xp, yp, zp, 50, clr, '.', 'tag', 'drawShearForceXZDiagram')

                            % Plot maximum value in text
                            xe = maxTextCoords(1);
                            ye = maxTextCoords(2);
                            ze = maxTextCoords(3);
                            % Avoid plotting text too close to element end
                            if abs(Qmax(2) - L) >= L/25 && Qmax(2) >= L/25
                                if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                    value = sprintf('%+.*f kN',dc,Qmax(1));
                                else
                                    value = sprintf('%+.*f',dc,Qmax(1));
                                end
                                text(xe, ye, ze, value, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Qmax(1));
                            end
                        end
                    else
                        % Connect both ends with a straight line
                        X = [xd1, xd2];
                        Y = [yd1, yd2];
                        Z = [zd1, zd2];
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawShearForceXZDiagram');
                    end
                    
                else
                    % Connect both ends with a straight line
                    X = [x1, x2];
                    Y = [y1, y2];
                    Z = [zd1, zd2];
                    line(X, Y, Z, 'Color', clr, 'tag', 'drawShearForceXZDiagram');
                end
            end
        end
        
        %------------------------------------------------------------------
        % Computes bending moment diagram scale factor value in XY plane.
        % NOT USED
        function draw = bendingMomentScaleFactor_XY(draw)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Draws resulting bending moment diagram in XY plane on a given scale.
        % NOT USED
        function draw = bendingMoment_XY(draw,~)
            % MATLAB requires implementation of all abstract methods
            % declared on superclass Draw.
        end
        
        %------------------------------------------------------------------
        % Computes bending moment diagram scale factor value in XZ plane.
        function draw = bendingMomentScaleFactor_XZ(draw)
            mdata = guidata(findobj('Tag','GUI_Main'));
            sliderm = get(mdata.slider_Scale,'Max');
            
            % Get maximum internal bending moment value of each element
            max_elem = zeros(1,draw.drv.nel);
            for e = 1:draw.drv.nel
                % Get maximum value at element ends
                M1_Y = draw.drv.elems(e).bending_moment_Y(1);
                M2_Y = draw.drv.elems(e).bending_moment_Y(2);
                max_end = max(abs(M1_Y), abs(M2_Y));
                
                % Get maximum internal value
                if ~isempty(draw.drv.elems(e).maxBendMoment_XZ)
                    max_int = max(abs(draw.drv.elems(e).maxBendMoment_XZ(:,1)));
                else
                    max_int = 0;
                end
                
                max_elem(e) = max(max_end,max_int);
            end
            
            % Get maximum axial internal force value of model
            max_val = max(max_elem);
            
            % Set adapted scale value
            if max_val == 0
                bsf = 0;
            else
                bsf = draw.size/(2.5*sliderm*max_val);
            end
            setappdata(0,'bendingXZ_sf',bsf);
        end
        
        %------------------------------------------------------------------
        % Draws resulting bending moment diagram in XY plane on a given scale.
        % Input arguments:
        %  scale: bending moment diagram scale factor
        function draw = bendingMoment_XZ(draw,scale)
            include_constants
            
            % Parameters
            d = draw.size/80;  % distance between force value and diagram
            clr = [1,0,0];     % diagram line color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            % Get target element ID
            elem = get(mdata.popupmenu_ElementResults,'Value') - 1;
            if elem ~= 0
                i = elem;
                j = i;
            else
                i = 1;
                j = draw.drv.nel;
            end
            
            for e = i:j
                % Get element length
                L = draw.drv.elems(e).length;
                
                % Get element orientation angle cosine with axes X and Y
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
                cz = draw.drv.elems(e).cosine_Z;
                
                % Get element internal force values ate both ends
                M1 = draw.drv.elems(e).bending_moment_Y(1);
                M2 = -draw.drv.elems(e).bending_moment_Y(2);
                
                % Avoid plotting garbage
                if abs(M1) < 1e-10
                    M1 = 0;
                end
                if abs(M2) < 1e-10
                    M2 = 0;
                end
                
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                z1 = draw.drv.elems(e).nodes(1).coord(3);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Calculate diagram beginning coordinates
                zd1 = z1 - scale * M1;
                zd2 = z2 - scale * M2;
                
                % Draw diagram extremities
                Xi = [x1, x1];
                Yi = [y1, y1];
                Zi = [z1, zd1];
                line(Xi, Yi, Zi, 'Color', clr, 'tag', 'drawBendMomentXZDiagram');
                
                Xf = [x2, x2];
                Yf = [y2, y2];
                Zf = [z2, zd2];
                line(Xf, Yf, Zf, 'Color', clr, 'tag', 'drawBendMomentXZDiagram');
                
                % Write bending moment values
                if (abs(M1-M2) < 10e-10) &&...
                   (isempty(draw.drv.elems(e).load.uniformGbl) == 1) &&...
                   (isempty(draw.drv.elems(e).load.linearGbl) == 1)
               
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%+.*f kNm',dc,M1);
                    else
                        value = sprintf('%+.*f',dc,M1);
                    end
                    
                    if M1 > 10e-10
                        text((x1 + x2)/2, (y1 + y2)/2, zd1 + d, value, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',M1);
                    elseif M1 < 10e-10
                        text((x1 + x2)/2, (y1 + y2)/2, zd1 - d, value, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',M1);
                    end
                    
                else
                    if M1 > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%+.*f kNm',dc,M1);
                        else
                            value1 = sprintf('%+.*f',dc,M1);
                        end
                        text(x1, y1, zd1 - d, value1, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',M1);
                        
                    elseif M1 < -10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%+.*f kNm',dc,M1);
                        else
                            value1 = sprintf('%+.*f',dc,M1);
                        end
                        text(x1, y1, zd1 + d, value1, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',M1);
                    end
                    
                    if M2 > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value2 = sprintf('%+.*f kNm',dc,M2);
                        else
                            value2 = sprintf('%+.*f',dc,M2);
                        end
                        text(x2, y2, zd2 - d, value2, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',M2);
                        
                    elseif M2 < -10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value2 = sprintf('%+.*f kNm',dc,M2);
                        else
                            value2 = sprintf('%+.*f',dc,M2);
                        end
                        text(x2, y2, zd2 + d, value2, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',M2);
                    end
                end
                
                % Connect diagram extremities:
                % Check if element has distributed load.
                % -If so, calculate shear force value along element length
                % -If not, connect both ends with a straight line
                if (isempty(draw.drv.elems(e).load.uniformLcl) == 0) || ...
                   (isempty(draw.drv.elems(e).load.linearLcl) == 0)
               
                    % Get element 50 point division coords and internal
                    % stress values
                    coords = draw.drv.elems(e).intCoords;
                    M = draw.drv.elems(e).intStresses(2,:);
                    Mmax = draw.drv.elems(e).maxBendMoment_XZ;
                    
                    % Avoid plotting numeric garbage
                    if ~all(abs(M) < 10^-10)
                    
                        % Get element basis transformation matrix
                        rot = draw.drv.elems(e).T;

                        % Compute diagram coordinates
                        diagramCoords = rot' * [zeros(1,size(M,2)); zeros(1,size(M,2)); -scale*M] + coords;

                        % Plot diagram
                        X = diagramCoords(1,:);
                        Y = diagramCoords(2,:);
                        Z = diagramCoords(3,:);
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawBendMomentXZDiagram');

                        % Check if there is a maximum value within the diagram
                        if ~isempty(Mmax)
                            % Compute maximum stress value position, in global
                            % coordinates
                            MmaxGblCoords = [x1 + (Mmax(:,2))' * cx;
                                             y1 + (Mmax(:,2))' * cy;
                                             z1 + (Mmax(:,2))' * cz];

                            % Get maximum stress values
                            Mmax_val = (Mmax(:,1))';

                            % Get number of maximum values within diagram
                            nm = size(Mmax_val,2);

                            % Compute max point in diagram and text coordinates
                            maxPointCoords = rot' * [zeros(1,nm); zeros(1,nm); -scale*Mmax_val] + MmaxGblCoords;
                            maxTextCoords = rot' * [zeros(1,nm); zeros(1,nm); -scale*Mmax_val - d*(Mmax_val./abs(Mmax_val))] + MmaxGblCoords;

                            % Plot point indicating maximum value
                            xp = maxPointCoords(1,:);
                            yp = maxPointCoords(2,:);
                            zp = maxPointCoords(3,:);
                            scatter3(xp, yp, zp, 50, clr, '.', 'tag', 'drawBendMomentXZDiagram')

                            % Plot maximum value in text
                            xe = maxTextCoords(1,:);
                            ye = maxTextCoords(2,:);
                            ze = maxTextCoords(3,:);
                            for np = 1:nm % 2 rounds max
                                % Avoid plotting text too close to element end
                                if abs(Mmax(np,2) - L) >= L/25 && Mmax(np,2) >= L/25
                                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                        value = sprintf('%+.*f kNm',dc,Mmax_val(np));
                                    else
                                        value = sprintf('%+.*f',dc,Mmax_val(np));
                                    end
                                    text(xe(np), ye(np), ze(np), value, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',Mmax_val(np));
                                end
                            end
                        end
                    else
                        % Connect both ends with a straight line
                        X = [xd1, xd2];
                        Y = [yd1, yd2];
                        Z = [zd1, zd2];
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawBendMomentXZDiagram');
                    end
                    
                else
                    % Connect both ends with a straight line
                    X = [x1, x2];
                    Y = [y1, y2];
                    Z = [zd1, zd2];
                    line(X, Y, Z, 'Color', clr, 'tag', 'drawBendMomentXZDiagram');
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws reactions indication next to nodal supports.
        function draw = reactions(draw)
            include_constants;
            % Parameters
            shift = draw.size/100;  % distance between reaction symbol and support
            ph = draw.size/40;      % translation constraint symbol (pyramid height)
            sh = draw.size/20;      % spring symbol height
            al = draw.size/20;      % reaction symbol size (arrow length)
            ah = draw.size/70;      % reaction symbol size (pyramid height)
            ab = draw.size/120;     % reaction symbol size (pyramid base)
            tt = draw.size/50;      % distance between text and nodal point
            clr = [0,0,1];          % reaction symbol color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
                z = draw.drv.nodes(n).coord(3);
                
                % Get reactions values
                rx = draw.drv.F(draw.drv.ID(1,n));
                ry = draw.drv.F(draw.drv.ID(2,n));
                rz = draw.drv.F(draw.drv.ID(3,n));
                
                % Rotation constraint symbol (cube side)
                cs = 0;
                
                if (draw.drv.nodes(n).ebc(4) == FIXED_DOF) && (draw.drv.nodes(n).ebc(5)== FIXED_DOF)
                    % Update rotation constraint symbol size value
                    cs = draw.size/100;
                elseif (draw.drv.nodes(n).ebc(4) == SPRING_DOF) && (draw.drv.nodes(n).ebc(5) == SPRING_DOF)
                    cs = draw.size/70;
                end    
                
                % Check if rotation is fixed and draw reaction indication
                if ((draw.drv.nodes(n).ebc(4) == FIXED_DOF) && (draw.drv.nodes(n).ebc(5)== FIXED_DOF))...
                   || ((draw.drv.nodes(n).ebc(4) == SPRING_DOF) && (draw.drv.nodes(n).ebc(5)== SPRING_DOF))     
                    if rx >= 0
                        draw.moment3D(draw, x - cs - shift, y, z, 1.5 * al, ah, ab, 'x+', clr,'drawReactions');
                        
                        % Write reaction value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(rx));
                        else
                            value = sprintf('%.*f',dc,abs(rx));
                        end
                        text(x - cs - shift - al - tt, y, z + tt, value, 'Color', clr,'tag','textMomentReactions','UserData',abs(rx));
                        
                    else
                        draw.moment3D(draw, x + cs + shift, y, z, 1.5 * al, ah, ab, 'x-', clr,'drawReactions');
                        
                        % Write reaction value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(rx));
                        else
                            value = sprintf('%.*f',dc,abs(rx));
                        end
                        text(x + cs + shift + al + tt, y, z + tt, value, 'Color', clr,'tag','textMomentReactions','UserData',abs(rx));
                    end
                    
                    if ry >= 0
                        draw.moment3D(draw, x, y - cs - shift, z, 1.5 * al, ah, ab, 'y+', clr,'drawReactions');
                        
                        % Write reaction value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(ry));
                        else
                            value = sprintf('%.*f',dc,abs(ry));
                        end
                        text(x, y - cs - shift - al - tt, z + tt, value, 'Color', clr,'tag','textMomentReactions','UserData',abs(ry));
                        
                    else
                        draw.moment3D(draw, x, y + cs + shift, z, 1.5 * al, ah, ab, 'y-', clr,'drawReactions');
                        
                        % Write reaction value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(ry));
                        else
                            value = sprintf('%.*f',dc,abs(ry));
                        end
                        text(x, y + cs + shift + al + tt, z + tt, value, 'Color', clr,'tag','textMomentReactions','UserData',abs(ry));
                    end
                end
                
                % Support (or spring) height in the direction of axis Z
                if draw.drv.nodes(n).ebc(3)== FIXED_DOF
                    hz = ph;
                elseif draw.drv.nodes(n).ebc(3)== SPRING_DOF
                    hz = sh;
                end 
                
                % Check if translation in the Z axis is fixed and draw
                % reaction indication
                if (draw.drv.nodes(n).ebc(3) == FIXED_DOF) || (draw.drv.nodes(n).ebc(3) == SPRING_DOF)
                    if rz >= 0
                        draw.arrow3D(draw, x, y, z - cs - shift - hz, al, ah, ab, 'z+', clr,'drawReactions');
                    else
                        draw.arrow3D(draw, x, y, z - cs - shift - hz - al, al, ah, ab, 'z-', clr,'drawReactions');
                    end
                    
                    % Write reaction value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%.*f kN',dc,abs(rz));
                    else
                        value = sprintf('%.*f',dc,abs(rz));
                    end
                    text(x, y, z - cs - shift - hz - 1.2 * al, value, 'Color', clr,'tag','textForceReactions','UserData',abs(rz));
                end
            end
        end
    end
end