%% Draw_Frame3D class
%
% This is a sub-class, in the Object Oriented Programming (OOP) paradigm,
% of super-class <draw.html *Draw*> in the <main.html LESM (Linear Elements
% Structure Model)> program. This sub-class implements abstract methods,
% defined in super-class *Draw*, that deal with 3D frame analysis model
% of linear structure elements.
%
%% Authors
% Luiz Fernando Martha, Rafael Lopez Rangel and Pedro Cortez Lopes
%
%% Class definition
classdef Draw_Frame3D < Draw
    %% Constructor method
    methods
        %------------------------------------------------------------------
        function draw = Draw_Frame3D(drv)
            draw = draw@Draw(37.5,30);
            
            if (nargin > 0)
                draw.drv = drv;
            end
        end
    end
    
    %% Public methods
    % Implementation of the abstract methods declared in super-class <draw.html *Draw*>.
    methods
        %------------------------------------------------------------------
        % Sets viewpoint position and computes drawing size parameter
        % according to model dimensions.
        function draw = setSize(draw)
            
            x = zeros(draw.drv.nnp);
            y = zeros(draw.drv.nnp);
            z = zeros(draw.drv.nnp);
            
            for n = 1:draw.drv.nnp
                x(n) = draw.drv.nodes(n).coord(1);
                y(n) = draw.drv.nodes(n).coord(2);
                z(n) = draw.drv.nodes(n).coord(3);
            end
            
            xmin = min(x);
            xmax = max(x);
            ymin = min(y);
            ymax = max(y);
            zmin = min(z);
            zmax = max(z);
            
            dx = xmax - xmin;
            dy = ymax - ymin;
            dz = zmax - zmin;
            sz = max([dx,dy,dz]);
            if isempty(sz)
                sz = 0;
            end
            
            if sz == 0
                draw.size = 5;
            else
                draw.size = sz;
            end
        end
        
        %------------------------------------------------------------------
        % Sets axes limits according to model dimensions.
        function draw = setLimits(draw)
            if draw.drv.nnp > 0
                x = zeros(draw.drv.nnp,1);
                y = zeros(draw.drv.nnp,1);
                z = zeros(draw.drv.nnp,1);
                
                for n = 1:draw.drv.nnp
                    x(n) = draw.drv.nodes(n).coord(1);
                    y(n) = draw.drv.nodes(n).coord(2);
                    z(n) = draw.drv.nodes(n).coord(3);
                end
                
                xmin = min(x);
                xmax = max(x);
                ymin = min(y);
                ymax = max(y);
                zmin = min(z);
                zmax = max(z);
                
                dx = xmax - xmin;
                dy = ymax - ymin;
                dz = zmax - zmin;
                
                if (dx == 0) && (dy == 0) && (dz == 0)
                    xlim([x - 2, x + 2])
                    ylim([y - 2, y + 2])
                    zlim([z - 1, z + 1])
                    
                elseif (dx >= dy) && (dx >= dz) ||...
                       (dy >= dx) && (dy >= dz) ||...
                       (dz >= dx) && (dz >= dy)
                    d = draw.size/5.2;
                    xlim([xmin - d, xmax + d])
                    ylim([ymin - d, ymax + d])
                    zlim([zmin - d, zmax + d])
                    
                else
                    xlim([xmin - dx/5, xmax + dx/5])
                    ylim([ymin - dy/5, ymax + dy/5])
                    zlim([zmin - dz/5, zmax + dz/5])
                end
                
                offset = ((max(xlim)-min(xlim))+(max(ylim)-min(ylim))+(max(zlim)-min(zlim)))/50;
                xlabel('X', 'Position', [mean(xlim), min(ylim) - offset/3, min(zlim) - offset/2]);
                ylabel('Y', 'Position', [min(xlim) - offset, mean(ylim), min(zlim) - offset/2]);
                zlabel('Z', 'Position', [min(xlim) - offset, max(ylim) + offset, mean(zlim)], 'Rotation', pi/2);
                
            else
                xlim([-5,5])
                ylim([-5,5])
                zlim([-5,5])
            end
        end
        
        %------------------------------------------------------------------
        % Draws structure model with nodes, elements, supports and  hinges.
        function draw = model(draw)
            draw.nodes();
            hold on
            draw.elements();
        end
        
        %------------------------------------------------------------------
        % Draws elements with hinged or continuous ends.
        function draw = elements(draw)
            % Parameters
            r = draw.size/125;  % hinge symbol radius
            clr = [0,0,0];      % element color
            
            for e = 1:draw.drv.nel
                % Get nodal IDs and coordinates
                n1 = draw.drv.elems(e).nodes(1).id;
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
				z1 = draw.drv.elems(e).nodes(1).coord(3);
                
                n2 = draw.drv.elems(e).nodes(2).id;
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
				z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Get element orientation angle cosine with X and Y axes
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
				cz = draw.drv.elems(e).cosine_Z;					
                
                % Get element incidence information on nodes
                [toti,hei] = draw.drv.nodes(n1).elemsIncidence(draw.drv);
                [totf,hef] = draw.drv.nodes(n2).elemsIncidence(draw.drv);
                
                % Set element end coordinates
                xi = x1;
                yi = y1;
				zi = z1;
                xf = x2;
                yf = y2;
				zf = z2;
                
                % Set element initial coordinates by checking if there is a
                % hinge on nodal point position or on element end
                if (hei >= (toti-1)) && (toti >= 2) &&...
                   ((draw.drv.nodes(n1).ebc(4) == 0) ||...
                   (draw.drv.nodes(n1).ebc(5) == 0) ||...
                   (draw.drv.nodes(n1).ebc(6) == 0))
                    % Draw one hinge symbol representing the articulation
                    % between all elements
                    draw.sphere(x1, y1, z1, r, 'drawNodes');
                    hold on
                    % One hinge on nodal point position
                    xi = x1 + r * cx;
                    yi = y1 + r * cy;
                    zi = z1 + r * cz;
                    
                elseif draw.drv.elems(e).hingei == 0
                    % Hinge on element end
                    draw.sphere(x1 + r * cx, y1 + r * cy, z1 + r * cz, r, 'drawElements');
                    xi = x1 + 2 * r * cx;
                    yi = y1 + 2 * r * cy;
                    zi = z1 + 2 * r * cz;
                end
                
                % Set element final coordinates by checking if there is a
                % hinge on nodal point position or on element end
                if (hef >= (totf-1)) && (totf >= 2) &&...
                   ((draw.drv.nodes(n2).ebc(4) == 0) ||...
                   (draw.drv.nodes(n2).ebc(5) == 0) ||...
                   (draw.drv.nodes(n2).ebc(6) == 0))
                    % Draw one hinge symbol representing the articulation
                    % between all elements
                    draw.sphere(x2, y2, z2, r, 'drawNodes');
                    hold on
                    % One hinge on nodal point position
                    xf = x2 - r * cx;
                    yf = y2 - r * cy;
                    zf = z2 - r * cz;
                    
                elseif draw.drv.elems(e).hingef == 0
                    % Hinge on element end
                    draw.sphere(x2 - r * cx, y2 - r * cy, z2 - r * cz, r, 'drawElements');
                    xf = x2 - 2 * r * cx;
                    yf = y2 - 2 * r * cy;
                    zf = z2 - 2 * r * cz;
                end
                
                % Connect element end coordinates
                X = [xi, xf];
                Y = [yi, yf];
                Z = [zi, zf];
                plot3(X, Y, Z, 'Color', clr, 'tag', 'drawElements');
                hold on
            end
        end
        
        %------------------------------------------------------------------
        % Draws nodal marks with support conditions.
        function draw = nodes(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Get flag for support visualization option status
            drawSupports =  get(mdata.viewSupportsButton,'Checked');
            
            include_constants;
            % Parameters
            nm = draw.size/200;    % node mark symbol (cube side)
            r = draw.size/125;     % hinge symbol radius
            ph = draw.size/30;     % translation constraint symbol (pyramid height)
            pb = draw.size/65;     % translation constraint symbol (pyramid base)
            cs = draw.size/75;     % rotation constraint symbol (cube side)
            sh = draw.size/20;     % spring symbol height
            sr = draw.size/105;     % rotational spring symbol radius
            nclr = [0,0,0];        % node and hinge color
            sclr = [0.6,0.6,0.6];  % support color
            sprclr = [0.6,0,0.4];  % spring color
            rotsclr = [0.7,0,0.5]; % rotational spring color
            dc = getappdata(0,'decPrec');  % decimal precision
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
                z = draw.drv.nodes(n).coord(3);
                
                % Get element incidence information
                [tot,hng] = draw.drv.nodes(n).elemsIncidence(draw.drv);
                
                % Distance between translation constraint support symbol
                % and nodal point.
                shift = 0;
                
                if (hng >= (tot-1)) && (tot >= 2) &&...
                   ((draw.drv.nodes(n).ebc(4) == FREE_DOF) ||...
                    (draw.drv.nodes(n).ebc(5) == FREE_DOF) ||...
                    (draw.drv.nodes(n).ebc(6) == FREE_DOF))
                    shift = r;
                    % Draw a nodal mark representing a free node
                    draw.cube(x, y, z, nm, nclr,'drawNodes');
                    hold on
                    
                elseif (draw.drv.nodes(n).ebc(4) == FIXED_DOF) &&...
                       (draw.drv.nodes(n).ebc(5) == FIXED_DOF) &&...
                       (draw.drv.nodes(n).ebc(6) == FIXED_DOF)
                    if strcmp(drawSupports,'on')
                        % Draw rotation constraint support
                        draw.cube(x, y, z, cs, sclr,'drawSupports');
                        shift = cs/2;
                        hold on
                    end
                
                elseif (draw.drv.nodes(n).ebc(4) == SPRING_DOF) &&...
                       (draw.drv.nodes(n).ebc(5) == SPRING_DOF) &&...
                       (draw.drv.nodes(n).ebc(6) == SPRING_DOF)
                   if strcmp(drawSupports,'on')
                       % Draw rotational spring
                       draw.rotSpring_3D(x,y,z,sr,rotsclr,'drawSupports');
                       hold on
                       shift = sr;
                       
                       % decimal precision
                       dc = getappdata(0,'decPrec');
                       % Get spring stiffness
                       kr = draw.drv.nodes(n).springStiff(4);
                       % Write spring stiffness value
                       if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                           if kr >= 1000
                               value = sprintf('%.*e kNm/rad',dc,kr);
                           else
                               value = sprintf('%.*f kNm/rad',dc,kr);
                           end
                       else
                           if kr >= 1000
                               value = sprintf('%.*e',dc,kr);
                           else
                               value = sprintf('%.*f',dc,kr);
                           end
                       end
                       
                       text(x + shift*0.8, y - shift*0.8, z + shift*1.1, value,...
                           'Color', sprclr,'Fontsize',8.5,'tag','textRotSprings','UserData',kr);
                   end
                    
                elseif (draw.drv.nodes(n).ebc(1) == FREE_DOF) &&...
                       (draw.drv.nodes(n).ebc(2) == FREE_DOF) &&...
                       (draw.drv.nodes(n).ebc(3) == FREE_DOF) &&...
                       (draw.drv.nodes(n).ebc(4) == FREE_DOF) &&...
                       (draw.drv.nodes(n).ebc(5) == FREE_DOF) &&...
                       (draw.drv.nodes(n).ebc(6) == FREE_DOF)
                    % Draw a nodal mark representing a free node
                    draw.cube(x, y, z, nm, nclr,'drawNodes');
                    hold on
                end
                
                % Check if there is a translation constraint in the
                % direction of axis X and draw support
                if draw.drv.nodes(n).ebc(1) == FIXED_DOF && strcmp(drawSupports,'on')
                    draw.pyramid(x - shift, y, z, ph, pb, 'x+', sclr,'drawSupports');
                    hold on
                end
                
                % Check if there is a translation constraint in the
                % direction of axis Y and draw support
                if draw.drv.nodes(n).ebc(2) == FIXED_DOF && strcmp(drawSupports,'on')
                    draw.pyramid(x, y - shift, z, ph, pb, 'y+', sclr,'drawSupports');
                    hold on
                end
                
                % Check if there is a translation constraint in the
                % direction of axis Z and draw support
                if draw.drv.nodes(n).ebc(3) == FIXED_DOF && strcmp(drawSupports,'on')
                    draw.pyramid(x, y, z - shift, ph, pb, 'z+', sclr,'drawSupports');
                    hold on
                end
                
                % Check if there is a displacement spring in the direction
                % of axis X and draw spring
                if draw.drv.nodes(n).ebc(1) == SPRING_DOF && strcmp(drawSupports,'on')   
                    draw.SpringX_3D(x-shift,y,z,sh,sprclr,'drawSupports');
                    hold on
                    
                    % Get spring stiffness
                    kx = draw.drv.nodes(n).springStiff(1);
                    % Write spring stiffness value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        if kx >= 1000
                        value = sprintf('%.*e kN/m',dc,kx);
                        else
                        value = sprintf('%.*f kN/m',dc,kx);
                        end
                    else
                        if kx >= 1000
                        value = sprintf('%.*e',dc,kx);
                        else
                        value = sprintf('%.*f',dc,kx);
                        end
                    end
                    
                    text(x - 1.2*sh - shift, y + sh*0.1, z + sh*0.5, value, 'Color', sprclr,'Fontsize',8.5,'tag','textSprings','UserData',kx);
                end
                
                % Check if there is a displacement spring in the direction
                % of axis Z and draw spring
                if draw.drv.nodes(n).ebc(2) == SPRING_DOF && strcmp(drawSupports,'on')
                    draw.SpringY_3D(x,y-shift,z,sh,sprclr,'drawSupports');
                    hold on

                    % Get spring stiffness
                    ky = draw.drv.nodes(n).springStiff(2);
                    % Write spring stiffness value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        if ky >= 1000
                        value = sprintf('%.*e kN/m',dc,ky);
                        else
                        value = sprintf('%.*f kN/m',dc,ky);
                        end
                    else
                        if ky >= 1000
                        value = sprintf('%.*e',dc,ky);
                        else
                        value = sprintf('%.*f',dc,ky);
                        end
                    end
                    
                    text(x + sh*0.1, y - 1.2*sh - shift, z + sh*0.1, value, 'Color', sprclr,'Fontsize',8.5,'tag','textSprings','UserData',ky);
                end
                
                % Check if there is a displacement spring in the direction
                % of axis Z and draw spring
                if draw.drv.nodes(n).ebc(3) == SPRING_DOF && strcmp(drawSupports,'on')
                    draw.SpringZ_3D(x,y,z-shift,sh,sprclr,'drawSupports');
                    hold on

                    % Get spring stiffness
                    kz = draw.drv.nodes(n).springStiff(3);
                    % Write spring stiffness value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        if kz >= 1000
                        value = sprintf('%.*e kN/m',dc,kz);
                        else
                        value = sprintf('%.*f kN/m',dc,kz);
                        end
                    else
                        if kz >= 1000
                        value = sprintf('%.*e',dc,kz);
                        else
                        value = sprintf('%.*f',dc,kz);
                        end
                    end
                    
                    text(x + sh*0.1, y - sh*0.1, z - 1.2*sh - shift, value, 'Color', sprclr,'Fontsize',8.5,'tag','textSprings','UserData',kz);
                end
            end
        end
        
        %------------------------------------------------------------------
        % Computes element loads scale factor.
        function scl = elemLoadsScaleFactor(draw)
            max_load_y = zeros(1,draw.drv.nel);
            max_load_z = zeros(1,draw.drv.nel);
            
            for e = 1:draw.drv.nel
                % Initialize load values on element ends
                qyi = 0;
                qyf = 0;
				qzi = 0;
				qzf = 0;
                
                % Add uniform load contribtuion
                if isempty(draw.drv.elems(e).load.uniformLcl) == 0
                    qy = draw.drv.elems(e).load.uniformLcl(2);
					qz = draw.drv.elems(e).load.uniformLcl(3);
                    qyi = qyi + qy;
                    qyf = qyf + qy;
					qzi = qzi + qz;
					qzf = qzf + qz;
                end
                
                % Add linear load contribtuion
                if isempty(draw.drv.elems(e).load.linearLcl) == 0
                    qyi = draw.drv.elems(e).load.linearLcl(2);
                    qyf = draw.drv.elems(e).load.linearLcl(5);
					qzi = draw.drv.elems(e).load.linearLcl(3);
                    qzf = draw.drv.elems(e).load.linearLcl(6);
                    qyi = qyi + qyi;
                    qyf = qyf + qyf;
                    qzi = qzi + qzi;
                    qzf = qzf + qzf;
                end
                
                % Get maximum load value on current element
                max_load_y(e) = max(abs(qyi), abs(qyf));
                max_load_z(e) = max(abs(qzi), abs(qzf));
            end
            
            % Get maximum load value on model
            max_val_y = max(max_load_y);
            max_val_z = max(max_load_z);
            max_val = max(max_val_y, max_val_z);
            
            % Calculate scale factor
            if max_val ~= 0
                scl = draw.size/(10*max_val);
            else
                scl = 0;
            end
            
            setappdata(0,'load_sf',scl);
            setappdata(0,'max_val',max_val);
        end
        
        %------------------------------------------------------------------
        % Draws element distributed loads (uniform and linear).
        function draw = elemLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if elem loads visualization is on
            if strcmp(get(mdata.viewDistribLoadsButton,'Checked'),'off')
                return
            end
            
            include_constants
            % Parameters
            r = draw.size/100;   % hinge symbol radius
            ah = draw.size/200;  % load symbol size (arrowhead height)
            ab = draw.size/300;  % load symbol size (arrowhead base)
            clr = [1,0,0];       % load symbol color
            scl = getappdata(0,'load_sf');                 % Load symbol scale
            max_val = getappdata(0,'max_val');
            rmappdata(0,'max_val')
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for e = 1:draw.drv.nel
                if ((isempty(draw.drv.elems(e).load.uniformGbl) == 0) &&...
                   (all(draw.drv.elems(e).load.uniformGbl == 0)==0)) ||...
                   ((isempty(draw.drv.elems(e).load.linearGbl) == 0) &&...
                   (all(draw.drv.elems(e).load.linearGbl == 0)==0))
               
                    % Get element length
                    L = draw.drv.elems(e).length;
                    
                    % Get initial node ID
                    n1 = draw.drv.elems(e).nodes(1).id;
                    
                    % Get initial nodal coordinates
                    x1 = draw.drv.nodes(n1).coord(1);
                    y1 = draw.drv.nodes(n1).coord(2);
					z1 = draw.drv.nodes(n1).coord(3);
                    
                    % Initialize load values on element ends
                    qxi = 0;
                    qyi = 0;
					qzi = 0;
                    qxf = 0;
                    qyf = 0;
					qzf = 0;
                    
                    % Add uniform load contribtuion
                    if isempty(draw.drv.elems(e).load.uniformLcl) == 0
                        qxi = qxi + draw.drv.elems(e).load.uniformLcl(1);
                        qyi = qyi + draw.drv.elems(e).load.uniformLcl(2);
						qzi = qzi + draw.drv.elems(e).load.uniformLcl(3);
                        qxf = qxi;
                        qyf = qyi;
						qzf = qzi;
                    end
                    
                    % Add linear load contribtuion
                    if isempty(draw.drv.elems(e).load.linearLcl) == 0
                        qxi = qxi + draw.drv.elems(e).load.linearLcl(1);
                        qyi = qyi + draw.drv.elems(e).load.linearLcl(2);
						qzi = qzi + draw.drv.elems(e).load.linearLcl(3);
                        qxf = qxf + draw.drv.elems(e).load.linearLcl(4);
                        qyf = qyf + draw.drv.elems(e).load.linearLcl(5);
						qzf = qzf + draw.drv.elems(e).load.linearLcl(6);
                    end
                    
                    % Axial load equation coefficients:
                    % p(x) = Ax + B
                    A = (qxf - qxi)/L;
                    B = qxi;
                    
                    % Transversal load equation coefficients in local axis Y:
                    % q(x) = Cx + D
                    C = (qyf - qyi)/L;
                    D = qyi;
					
					
                    % Transversal load equation coefficients in local axis Z:
                    % q(x) = Ex + F
                    E = (qzf - qzi)/L;
                    F = qzi;
                    
                    % Module of load parameters
                    Qyi = abs(scl * qyi);
                    Qyf = abs(scl * qyf);
					Qzi = abs(scl * qzi);
					Qzf = abs(scl * qzf);
                    
                    % Draw load symbol on a number cross-sections along
                    % element local axis X
                    step = (L-2*r) / round(20 * (L-2*r)/draw.size);
                    for x = r:step:(L-r)
                        % Calculate load values on current cross-section
                        qx = A * x + B;
                        qy = C * x + D;
						qz = E * x + F;
                        Qy = abs(scl * qy);
						Qz = abs(scl * qz);
                        
                        % Calculate current cross-section local coordinates
                        [xs,ys,zs] = draw.coordTransf3D(x, 0, 0, x1, y1, z1, e);
                        [xe,ye,ze] = draw.coordTransf3D(L/2, 3*r, 0, x1, y1, z1, e);
                        [xf,yf,zf] = draw.coordTransf3D(3*r, 3*r, 0, x1, y1, z1, e);
                        [xg,yg,zg] = draw.coordTransf3D(L-6*r, 3*r, 0, x1, y1, z1, e);
                        
                        % Get text distance factor
                        distTxt = (max_val/abs(qy));
                        if distTxt > 1.8
                            distTxt = 1.8;
                        elseif distTxt < 1.2
                            distTxt = 1.2;
                        end
                        
                        % Coordinates of transversal load symbol Qy
                        if qy > 0
                            [xa_y,ya_y,za_y] = draw.coordTransf3D(x, -r, 0, x1, y1, z1, e);
                            [xd_y,yd_y,zd_y] = draw.coordTransf3D(L/2, -distTxt*(Qy+r), 0, x1, y1, z1, e);
                        elseif qy < 0
                            [xa_y,ya_y,za_y] = draw.coordTransf3D(x, r, 0, x1, y1, z1, e);
                            [xd_y,yd_y,zd_y] = draw.coordTransf3D(L/2, distTxt*(Qy+r), 0, x1, y1, z1, e);
                            [xe,ye,ze] = draw.coordTransf3D(L/2, -3*r, 0, x1, y1, z1, e);
                        end
                        
                        % Get text distance factor
                        distTxt = (max_val/abs(qyi));
                        if distTxt > 1.8
                            distTxt = 1.8;
                        elseif distTxt < 1.2
                            distTxt = 1.2;
                        end
                        
                        if qyi > 0
                            [xb_y,yb_y,zb_y] = draw.coordTransf3D(r, -Qyi - r, 0, x1, y1, z1, e);
                            [xh_y,yh_y,zh_y] = draw.coordTransf3D(0, -distTxt*(Qyi+r), 0, x1, y1, z1, e);
                        elseif qyi < 0
                            [xb_y,yb_y,zb_y] = draw.coordTransf3D(r, Qyi + r, 0, x1, y1, z1, e);
                            [xh_y,yh_y,zh_y] = draw.coordTransf3D(0, distTxt*(Qyi+r), 0, x1, y1, z1, e);
                            [xf,yf,zf] = draw.coordTransf3D(3*r, -3*r, 0, x1, y1, z1, e);
                        elseif qyi == 0
                            if qyf > 0
                                [xb_y,yb_y,zb_y] = draw.coordTransf3D(r, -Qyi - r, 0, x1, y1, z1, e);
                                [xh_y,yh_y,zh_y] = draw.coordTransf3D(0, -distTxt*(Qyi+r), 0, x1, y1, z1, e);
                                [xf,yf,zf] = draw.coordTransf3D(3*r, 3*r, x1, 0, y1, z1, e);
                            elseif qyf < 0
                                [xb_y,yb_y,zb_y] = draw.coordTransf3D(r, Qyi + r, 0, x1, y1, z1, e);
                                [xh_y,yh_y,zh_y] = draw.coordTransf3D(0, distTxt*(Qyi+r), 0, x1, y1, z1, e);
                                [xf,yf,zf] = draw.coordTransf3D(3*r, -3*r, 0, x1, y1, z1, e);
                            end
                        end
                        
                        % Get text distance factor
                        distTxt = (max_val/abs(qyf));
                        if distTxt > 1.8
                            distTxt = 1.8;
                        elseif distTxt < 1.2
                            distTxt = 1.2;
                        end
                        
                        if qyf > 0
                            [xc_y,yc_y,zc_y] = draw.coordTransf3D(L-r, -Qyf - r, 0, x1, y1, z1, e);
                            [xj_y,yj_y,zj_y] = draw.coordTransf3D(L-3*r, -distTxt*(Qyf+r), 0, x1, y1, z1, e);
                        elseif qyf < 0
                            [xc_y,yc_y,zc_y] = draw.coordTransf3D(L-r, Qyf + r, 0, x1, y1, z1, e);
                            [xj_y,yj_y,zj_y] = draw.coordTransf3D(L-3*r, distTxt*(Qyf+r), 0, x1, y1, z1, e);
                            [xg,yg,zg] = draw.coordTransf3D(L-6*r, -3*r, 0, x1, y1, z1, e);
                        elseif qyf == 0
                            if qyi > 0
                                [xc_y,yc_y,zc_y] = draw.coordTransf3D(L-r, -Qyf - r, 0, x1, y1, z1, e);
                                [xj_y,yj_y,zj_y] = draw.coordTransf3D(L-3*r, -distTxt*(Qyf+r), 0, x1, y1, z1, e);
                                [xg,yg,zg] = draw.coordTransf3D(L-6*r, 3*r, 0, x1, y1, z1, e);
                            elseif qyi < 0
                                [xc_y,yc_y,zc_y] = draw.coordTransf3D(L-r, Qyf + r, 0, x1, y1, z1, e);
                                [xj_y,yj_y,zj_y] = draw.coordTransf3D(L-3*r, distTxt*(Qyf+r), 0, x1, y1, z1, e);
                                [xg,yg,zg] = draw.coordTransf3D(L-6*r, -3*r, 0, x1, y1, z1, e);
                            end
                        end
                        
                        % Get text distance factor
                        distTxt = (max_val/abs(qz));
                        if distTxt > 1.8
                            distTxt = 1.8;
                        elseif distTxt < 1.2
                            distTxt = 1.2;
                        end
						
                        % Coordinates of transversal load symbol Qz
                        if qz > 0
                            [xa_z,ya_z,za_z] = draw.coordTransf3D(x, 0, -r, x1, y1, z1, e);
                            [xd_z,yd_z,zd_z] = draw.coordTransf3D(L/2, 0, -distTxt*(Qz+r), x1, y1, z1, e);
                        elseif qz < 0
                            [xa_z,ya_z,za_z] = draw.coordTransf3D(x, 0, r, x1, y1, z1, e);
                            [xd_z,yd_z,zd_z] = draw.coordTransf3D(L/2, 0, distTxt*(Qz+r), x1, y1, z1, e);
                        end
                        
                        % Get text distance factor
                        distTxt = (max_val/abs(qzi));
                        if distTxt > 1.8
                            distTxt = 1.8;
                        elseif distTxt < 1.2
                            distTxt = 1.2;
                        end
                        
                        if qzi > 0
                            [xb_z,yb_z,zb_z] = draw.coordTransf3D(r, 0, -Qzi - r, x1, y1, z1, e);
                            [xh_z,yh_z,zh_z] = draw.coordTransf3D(0, 0, -distTxt*(Qzi+r), x1, y1, z1, e);
                        elseif qzi < 0
                            [xb_z,yb_z,zb_z] = draw.coordTransf3D(r, 0, Qzi + r, x1, y1, z1, e);
                            [xh_z,yh_z,zh_z] = draw.coordTransf3D(0, 0, distTxt*(Qzi+r), x1, y1, z1, e);
                        elseif qzi == 0
                            if qzf > 0
                                [xb_z,yb_z,zb_z] = draw.coordTransf3D(r, 0, -Qzi - r, x1, y1, z1, e);
                                [xh_z,yh_z,zh_z] = draw.coordTransf3D(0, 0, -distTxt*(Qzi+r), x1, y1, z1, e);
                            elseif qzf < 0
                                [xb_z,yb_z,zb_z] = draw.coordTransf3D(r, 0, Qzi + r, x1, y1, z1, e);
                                [xh_z,yh_z,zh_z] = draw.coordTransf3D(0, 0, distTxt*(Qzi+r), x1, y1, z1, e);
                            end
                        end
                        
                        % Get text distance factor
                        distTxt = (max_val/abs(qzf));
                        if distTxt > 1.8
                            distTxt = 1.8;
                        elseif distTxt < 1.2
                            distTxt = 1.2;
                        end
                        
                        if qzf > 0
                            [xc_z,yc_z,zc_z] = draw.coordTransf3D(L-r, 0, -Qzf - r, x1, y1, z1, e);
                            [xj_z,yj_z,zj_z] = draw.coordTransf3D(L-3*r, 0, -distTxt*(Qzf+r), x1, y1, z1, e);
                        elseif qzf < 0
                            [xc_z,yc_z,zc_z] = draw.coordTransf3D(L-r, 0, Qzf + r, x1, y1, z1, e);
                            [xj_z,yj_z,zj_z] = draw.coordTransf3D(L-3*r, 0, distTxt*(Qzf+r), x1, y1, z1, e);
                            [xg,yg,zg] = draw.coordTransf3D(L-6*r, 0, -3*r, x1, y1, z1, e);
                        elseif qzf == 0
                            if qzi > 0
                                [xc_z,yc_z,zc_z] = draw.coordTransf3D(L-r, 0, -Qzf - r, x1, y1, z1, e);
                                [xj_z,yj_z,zj_z] = draw.coordTransf3D(L-3*r, 0, -distTxt*(Qzf+r), x1, y1, z1, e);
                                [xg,yg,zg] = draw.coordTransf3D(L-6*r, 0, 3*r, x1, y1, z1, e);
                            elseif qzi < 0
                                [xc_z,yc_z,zc_z] = draw.coordTransf3D(L-r, 0, Qzf + r, x1, y1, z1, e);
                                [xj_z,yj_z,zj_z] = draw.coordTransf3D(L-3*r, 0, distTxt*(Qzf+r), x1, y1, z1, e);
                                [xg,yg,zg] = draw.coordTransf3D(L-6*r, 0, -3*r, x1, y1, z1, e);
                            end
                        end
                        
                        % Draw axial load symbols
                        if ((x ~= r) && ((x+step) < L)) && ((qxi ~= 0) || (qxf ~= 0))
                            if qx > 0
                                draw.spear3D(xs, ys, zs, 0, 3*ah, 1.5*ab, 'x+', clr, e, draw,'drawElemLoads');
                            elseif qx < 0
                                draw.spear3D(xs, ys, zs, 0, 3*ah, 1.5*ab, 'x-', clr, e, draw,'drawElemLoads');
                            end
                        end
                                                
                        % Draw transversal load symbol in local axis Y
                        if Qy >= ah
                            if qy > 0
                                draw.spear3D(xa_y, ya_y, za_y, Qy, ah, ab, 'y+', clr, e, draw,'drawElemLoads');
                            else
                                draw.spear3D(xa_y, ya_y, za_y, Qy, ah, ab, 'y-', clr, e, draw,'drawElemLoads');
                            end
                        elseif (x == r) && ((Qyi ~=0) || (Qyf ~=0))
                            [xi,yi,zi] = draw.coordTransf3D(r, 0, 0, x1, y1, z1, e);
                            X = [xi, xb_y];
                            Y = [yi, yb_y];
                            Z = [zi, zb_y];
                            line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                        elseif (x == L-r) && ((Qyi ~=0) || (Qyf ~=0))
                            [xf,yf,zf] = draw.coordTransf3D(L-r, 0, 0, x1, y1, z1, e);
                            X = [xf, xc_y];
                            Y = [yf, yc_y];
                            Z = [zf, zc_y];
                            line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                        end
                        
                        % Draw transversal load symbol in local axis Z
                        if Qz >= ah
                            if qz > 0
                                draw.spear3D(xa_z, ya_z, za_z, Qz, ah, ab, 'z+', clr, e, draw,'drawElemLoads');
                            else
                                draw.spear3D(xa_z, ya_z, za_z, Qz, ah, ab, 'z-', clr, e, draw,'drawElemLoads');
                            end
                        elseif (x == r) && ((Qzi ~=0) || (Qzf ~=0))
                            [xi,yi,zi] = draw.coordTransf3D(r, 0, 0, x1, y1, z1, e);
                            X = [xi, xb_z];
                            Y = [yi, yb_z];
                            Z = [zi, zb_z];
                            line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                        elseif (x == L-r) && ((Qzi ~=0) || (Qzf ~=0))
                            [xf,yf,zf] = draw.coordTransf3D(L-r, 0, 0, x1, y1, z1, e);
                            X = [xf, xc_z];
                            Y = [yf, yc_z];
                            Z = [zf, zc_z];
                            line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                        end
                    end
                    
                    % Connect diagram extremities in local axis Y
                    if  (qyi < 0) && (qyf > 0)
                        x0 = abs((Qyi*(L-2*r))/(Qyf+Qyi));
                        [xu,yu,zu] = draw.coordTransf3D(x0, r, 0, x1, y1, z1, e);
                        [xd,yd,zd] = draw.coordTransf3D(x0, -r, 0, x1, y1, z1, e);
                        
                        X = [xb_y, xu, xd, xc_y];
                        Y = [yb_y, yu, yd, yc_y];
                        Z = [zb_y, zu, zd, zc_y];
                        line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                    elseif (qyi > 0) && (qyf < 0)
                        x0 = abs((Qyi*(L-2*r))/(Qyf+Qyi));
                        [xu,yu,zu] = draw.coordTransf3D(x0, r, 0, x1, y1, z1, e);
                        [xd,yd,zd] = draw.coordTransf3D(x0, -r, 0, x1, y1, z1, e);
                        
                        X = [xb_y, xd, xu, xc_y];
                        Y = [yb_y, yd, yu, yc_y];
                        Z = [zb_y, zd, zu, zc_y];
                        line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                    elseif (qyi ~= 0) || (qyf ~= 0)
                        X = [xb_y, xc_y];
                        Y = [yb_y, yc_y];
						Z = [zb_y, zc_y];
                        line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                    end
					
                    % Connect diagram extremitiesin in local axis Z
                    if  (qzi < 0) && (qzf > 0)
                        x0 = abs((Qzi*(L-2*r))/(Qzf+Qzi));
                        [xu,yu,zu] = draw.coordTransf3D(x0, 0, r, x1, y1, z1, e);
                        [xd,yd,zd] = draw.coordTransf3D(x0, 0, -r, x1, y1, z1, e);
                        
                        X = [xb_z, xu, xd, xc_z];
                        Y = [yb_z, yu, yd, yc_z];
                        Z = [zb_z, zu, zd, zc_z];
                        line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                    elseif (qzi > 0) && (qzf < 0)
                        x0 = abs((Qzi*(L-2*r))/(Qzf+Qzi));
                        [xu,yu,zu] = draw.coordTransf3D(x0, 0, r, x1, y1, z1, e);
                        [xd,yd,zd] = draw.coordTransf3D(x0, 0, -r, x1, y1, z1, e);
                        
                        X = [xb_z, xd, xu, xc_z];
                        Y = [yb_z, yd, yu, yc_z];
                        Z = [zb_z, zd, zu, zc_z];
                        line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                    elseif (qzi ~= 0) || (qzf ~= 0)
                        X = [xb_z, xc_z];
                        Y = [yb_z, yc_z];
						Z = [zb_z, zc_z];
                        line(X, Y, Z, 'Color', clr,'tag','drawElemLoads');
                    end
                    
                    % Write load values:
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value_x1 = sprintf('%.*f kN/m',dc,abs(qxi));
                        value_x2 = sprintf('%.*f kN/m',dc,abs(qxf));
                        value_y1 = sprintf('%.*f kN/m',dc,abs(qyi));
                        value_y2 = sprintf('%.*f kN/m',dc,abs(qyf));
						value_z1 = sprintf('%.*f kN/m',dc,abs(qzi));
                        value_z2 = sprintf('%.*f kN/m',dc,abs(qzf));
                    else
                        value_x1 = sprintf('%.*f',dc,abs(qxi));
                        value_x2 = sprintf('%.*f',dc,abs(qxf));
                        value_y1 = sprintf('%.*f',dc,abs(qyi));
                        value_y2 = sprintf('%.*f',dc,abs(qyf));
						value_z1 = sprintf('%.*f',dc,abs(qzi));
                        value_z2 = sprintf('%.*f',dc,abs(qzf));
                    end

                    % Write axial load values
                    if (qxi ~= 0) || (qxf ~= 0)
                        if qxi == qxf
                            text(xe, ye, ze, value_x1, 'Color', clr,'tag','textElemLoads','UserData',abs(qxi));
                        else
                            text(xf, yf, zf, value_x1, 'Color', clr,'tag','textElemLoads','UserData',abs(qxi));
                            text(xg, yg, zg, value_x2, 'Color', clr,'tag','textElemLoads','UserData',abs(qxf));
                        end
                    end
                    
                    % Write transversal load values in local axis Y
                    if (qyi ~= 0) || (qyf ~= 0)
                        if qyi == qyf
                            text(xd_y, yd_y, zd_y, value_y1, 'Color', clr,'tag','textElemLoads','UserData',abs(qyi));
                        else
                            if qyi ~= 0
                                text(xh_y, yh_y, zh_y, value_y1, 'Color', clr,'tag','textElemLoads','UserData',abs(qyi));
                            end
                            if qyf ~= 0
                                text(xj_y, yj_y, zj_y, value_y2, 'Color', clr,'tag','textElemLoads','UserData',abs(qyf));
                            end
                        end
                    end
					
					% Write transversal load values in local axis Z
                    if (qzi ~= 0) || (qzf ~= 0)
                        if qzi == qzf
                            text(xd_z, yd_z, zd_z, value_z1, 'Color', clr,'tag','textElemLoads','UserData',abs(qzi));
                        else
                            if qzi ~= 0
                                text(xh_z, yh_z, zh_z, value_z1, 'Color', clr,'tag','textElemLoads','UserData',abs(qzi));
                            end
                            if qzf ~= 0
                                text(xj_z, yj_z, zj_z, value_z2, 'Color', clr,'tag','textElemLoads','UserData',abs(qzf));
                            end
                        end
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws applied nodal loads and moments.
        function draw = nodalLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if nodal loads visualization is on
            if strcmp(get(mdata.viewNodalLoadsButton,'Checked'),'off')
                return
            end
            
            % Parameters
            lshift = draw.size/100;   % distance between load symbol and nodal point
            al = draw.size/10;        % load symbol size (arrow length)
            ah = draw.size/30;        % load symbol size (arrowhead height)
            ab = draw.size/70;        % load symbol size (arrowhead base)
            t1 = draw.size/9;         % distance between text and nodal point
            t2 = draw.size/45;        % distance between text and nodal point
            t3 = draw.size/300;       % distance between text and nodal point
            clr = [1,0,0];            % load color
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for n = 1:draw.drv.nnp
                % Check if current node has a nodal load
                if isempty(draw.drv.nodes(n).nodalLoad) == 0
                    % Set distance between moment symbol and nodal point
                    mshift_x1 = draw.size/100;  
                    mshift_x2 = draw.size/100;
                    mshift_y1 = draw.size/100;
                    mshift_y2 = draw.size/100;
                    mshift_z1 = draw.size/100;
                    mshift_z2 = draw.size/100;
                    
                    % Get nodal coordinates
                    x = draw.drv.nodes(n).coord(1);
                    y = draw.drv.nodes(n).coord(2);
                    z = draw.drv.nodes(n).coord(3);
                    
                    % Get nodal load components
                    fx = draw.drv.nodes(n).nodalLoad(1);
                    fy = draw.drv.nodes(n).nodalLoad(2);
                    fz = draw.drv.nodes(n).nodalLoad(3);
                    mx = draw.drv.nodes(n).nodalLoad(4);
                    my = draw.drv.nodes(n).nodalLoad(5);
                    mz = draw.drv.nodes(n).nodalLoad(6);
                    
                    % Draw load component in X axis
                    if fx > 0
                        draw.arrow3D(draw, x - lshift, y, z, al, ah, ab, 'x+', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fx));
                        else
                            value = sprintf('%.*f',dc,abs(fx));
                        end
                        text(x - t1, y + t2, z + t2, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fx));
                        
                        mshift_x1 = mshift_x1 + lshift + al;
                        
                    elseif fx < 0
                        draw.arrow3D(draw, x + lshift, y, z, al, ah, ab, 'x-', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fx));
                        else
                            value = sprintf('%.*f',dc,abs(fx));
                        end
                        text(x + t1, y + t2, z + t2, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fx));
                        
                        mshift_x2 = mshift_x2 + lshift + al;
                        
                    end
                    
                    % Draw load component in Y axis
                    if fy > 0
                        draw.arrow3D(draw, x, y - lshift, z, al, ah, ab, 'y+', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fy));
                        else
                            value = sprintf('%.*f',dc,abs(fy));
                        end
                        text(x + t2, y - t1, z + t2, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fy));
                        
                        mshift_y1 = mshift_y1 + lshift + al;
                        
                    elseif fy < 0
                        draw.arrow3D(draw, x, y + lshift, z, al, ah, ab, 'y-', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fy));
                        else
                            value = sprintf('%.*f',dc,abs(fy));
                        end
                        text(x + t2, y + t1, z + t2, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fy));
                        
                        mshift_y2 = mshift_y2 + lshift + al;
                        
                    end
                    
                    % Draw load component in Z axis
                    if fz > 0
                        draw.arrow3D(draw, x, y, z - lshift, al, ah, ab, 'z+', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fz));
                        else
                            value = sprintf('%.*f',dc,abs(fz));
                        end
                        text(x + t3, y + t3, z - t1, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fz));
                        
                        mshift_z1 = mshift_z1 + lshift + al;
                        
                    elseif fz < 0
                        draw.arrow3D(draw, x, y, z + lshift, al, ah, ab, 'z-', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kN',dc,abs(fz));
                        else
                            value = sprintf('%.*f',dc,abs(fz));
                        end
                        text(x + t3, y + t3, z + t1, value, 'Color', clr,'tag','textNodalLoads','UserData',abs(fz));
                        
                        mshift_z2 = mshift_z2 + lshift + al;
                        
                    end
                    
                    % Draw applied moment component in the X direction 
                    if mx > 0
                        draw.moment3D(draw, x - mshift_x1, y, z, al, ah/1.4, ab, 'x+', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mx));
                        else
                            value = sprintf('%.*f',dc,abs(mx));
                        end
                        text(x - mshift_x1 - al, y + t2, z + t2, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(mx));
                        
                    elseif mx < 0
                        draw.moment3D(draw, x + mshift_x2, y, z, al, ah/1.4, ab, 'x-', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mx));
                        else
                            value = sprintf('%.*f',dc,abs(mx));
                        end
                        text(x + mshift_x2 + al, y + t2, z + t2, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(mx));
                    end
                    
                    % Draw applied moment component in the Y direction 
                    if my > 0
                        draw.moment3D(draw, x, y - mshift_y1, z, al, ah/1.4, ab, 'y+', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(my));
                        else
                            value = sprintf('%.*f',dc,abs(my));
                        end
                        text(x + t2, y - mshift_y1 - al, z + t2, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(my));
                        
                    elseif my < 0
                        draw.moment3D(draw, x, y + mshift_y2, z, al, ah/1.4, ab, 'y-', clr,'drawNodalLoads');
                        
                        % Write nodal load value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(my));
                        else
                            value = sprintf('%.*f',dc,abs(my));
                        end
                        text(x + t2, y + mshift_y2 + al, z + t2, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(my));
                    end
                    
                    % Draw applied moment component in the Z direction 
                    if mz > 0
                        draw.moment3D(draw, x, y, z - mshift_z1, al, ah/1.4, ab, 'z+', clr,'drawNodalLoads');
                        
                        % Write moment value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mz));
                        else
                            value = sprintf('%.*f',dc,abs(mz));
                        end
                        text(x + t2, y + t2, z - mshift_z1 - al, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(mz));
                        
                    elseif mz < 0
                        draw.moment3D(draw, x, y, z + mshift_z2, al, ah/1.4, ab, 'z-', clr,'drawNodalLoads');
                        
                        % Write moment value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mz));
                        else
                            value = sprintf('%.*f',dc,abs(mz));
                        end
                        text(x + t2, y + t2, z + mshift_z2 + al, value, 'Color', clr,'tag','textNodalMoments','UserData',abs(mz));
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws nodal prescribed displacement representation.
        function draw = nodalPrescDispl(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if presc displ visualization is on
            if strcmp(get(mdata.viewPrescDisplButton,'Checked'),'off')
                return
            end
            
            % Parameters
            shift = draw.size/100;   % distance between presc. displ. symbol and support
            ph = draw.size/30;       % translation constraint symbol (pyramid height)
            al = draw.size/10;       % presc. displ. symbol size (arrow length)
            ah = draw.size/30;       % presc. displ. symbol size (pyramid height)
            ab = draw.size/70;       % presc. displ. symbol size (pyramid base)
            t = draw.size/45;        % distance between text and nodal point
            clr = [1,0,1];           % presc. displ. symbol color
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for n =1:draw.drv.nnp
                % Check if current node has a prescribed displacement
                if isempty(draw.drv.nodes(n).prescDispl) == 0
                    % Get nodal coordinates
                    x = draw.drv.nodes(n).coord(1);
                    y = draw.drv.nodes(n).coord(2);
                    z = draw.drv.nodes(n).coord(3);
                    
                    % Get prescribed displacement component values and
                    % convert it to milimeter and radian
                    dx = 1000 * draw.drv.nodes(n).prescDispl(1);
                    dy = 1000 * draw.drv.nodes(n).prescDispl(2);
                    dz = 1000 * draw.drv.nodes(n).prescDispl(3);
                    rx = draw.drv.nodes(n).prescDispl(4);
                    ry = draw.drv.nodes(n).prescDispl(5);
                    rz = draw.drv.nodes(n).prescDispl(6);
                    
                    % Initialize rotation constraint symbol (cube side)
                    cs = 0;
                    
                    % Check if rotation is really fixed and draw prescribed
                    % displacement indication
                    if (draw.drv.nodes(n).ebc(4) == 1) &&...
                       (draw.drv.nodes(n).ebc(5) == 1) &&...
                       (draw.drv.nodes(n).ebc(6) == 1)
                        % Update rotation constraint symbol size value
                        cs = draw.size/100;
                        
                        if rx > 0
                            
                            draw.moment3D(draw, x + cs + shift + al, y, z, al, ah/1.4, ab, 'x+', clr,'drawPrescDispl');
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f rad',dc,abs(rx));
                            else
                                value = sprintf('%.*f',dc,abs(rx));
                            end
                            text(x + cs + shift + al, y + t, z + t, value, 'Color', clr,'tag','textPrescRot','UserData',abs(rx));
                            
                        elseif rx < 0
                            
                            draw.moment3D(draw, x + cs + shift, y, z, al, ah/1.4, ab, 'x-', clr,'drawPrescDispl');
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f rad',dc,abs(rx));
                            else
                                value = sprintf('%.*f',dc,abs(rx));
                            end
                            text(x + cs + shift + al, y + t, z + t, value, 'Color', clr,'tag','textPrescRot','UserData',abs(rx));
                        end
                        
                        if ry > 0
                            
                            draw.moment3D(draw, x, y + cs + shift + al, z, al, ah/1.4, ab, 'y+', clr,'drawPrescDispl');
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f rad',dc,abs(ry));
                            else
                                value = sprintf('%.*f',dc,abs(ry));
                            end
                            text(x + t, y + cs + shift + al, z + t, value, 'Color', clr,'tag','textPrescRot','UserData',abs(ry));
                            
                        elseif ry < 0
                            
                            draw.moment3D(draw, x, y + cs + shift, z, al, ah/1.4, ab, 'y-', clr,'drawPrescDispl');
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f rad',dc,abs(ry));
                            else
                                value = sprintf('%.*f',dc,abs(ry));
                            end
                            text(x + t, y + cs + shift + al, z + t, value, 'Color', clr,'tag','textPrescRot','UserData',abs(ry));
                        end
                        
                        if rz > 0
                            
                            draw.moment3D(draw, x, y, z + cs + shift + al, al, ah/1.4, ab, 'z+', clr,'drawPrescDispl');
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f rad',dc,abs(rz));
                            else
                                value = sprintf('%.*f',dc,abs(rz));
                            end
                            text(x + t, y + t, z + cs + shift + al, value, 'Color', clr,'tag','textPrescRot','UserData',abs(rz));
                            
                        elseif rz < 0
                            
                            draw.moment3D(draw, x, y, z + cs + shift, al, ah/1.4, ab, 'z-', clr,'drawPrescDispl');
                            
                            % Write displacement value
                            if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                value = sprintf('%.*f rad',dc,abs(rz));
                            else
                                value = sprintf('%.*f',dc,abs(rz));
                            end
                            text(x + t, y + t, z + cs + shift + al, value, 'Color', clr,'tag','textPrescRot','UserData',abs(rz));
                        end
                    end
                    
                    % Check if translation is really fixed in X direction
                    % and  draw prescribed displacement indication
                    if (draw.drv.nodes(n).ebc(1) == 1) && (dx ~= 0)
                        if dx > 0
                            draw.arrow3D(draw, x - cs - shift - ph, y, z, al, ah, ab, 'x+', clr,'drawPrescDispl');
                        else
                            draw.arrow3D(draw, x - cs - shift - ph - al, y, z, al, ah, ab, 'x-', clr,'drawPrescDispl');
                        end
                        
                        % Write displacement value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f mm',dc,abs(dx));
                        else
                            value = sprintf('%.*f',dc,abs(dx));
                        end
                        text(x - cs - shift - ph - al, y + t, z + t, value, 'Color', clr,'tag','textPrescDispl','UserData',abs(dx));
                    end
                    
                    % Check if translation is really fixed in Y direction
                    % and  draw prescribed displacement indication
                    if (draw.drv.nodes(n).ebc(2) == 1) && (dy ~= 0)
                        if dy > 0
                            draw.arrow3D(draw, x, y - cs - shift - ph, z, al, ah, ab, 'y+', clr,'drawPrescDispl');
                        else
                            draw.arrow3D(draw, x, y - cs - shift - ph - al, z, al, ah, ab, 'y-', clr,'drawPrescDispl');
                        end
                        
                        % Write displacement value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f mm',dc,abs(dy));
                        else
                            value = sprintf('%.*f',dc,abs(dy));
                        end
                        text(x + t, y - cs - shift - ph - al, z + t, value, 'Color', clr,'tag','textPrescDispl','UserData',abs(dy));
                    end
                    
                    % Check if translation is really fixed in Z direction
                    % and  draw prescribed displacement indication
                    if (draw.drv.nodes(n).ebc(3) == 1) && (dz ~= 0)
                        if dz > 0
                            draw.arrow3D(draw, x, y, z - cs - shift - ph, al, ah, ab, 'z+', clr,'drawPrescDispl');
                        else
                            draw.arrow3D(draw, x, y, z - cs - shift - ph - al, al, ah, ab, 'z-', clr,'drawPrescDispl');
                        end

                        % Write displacement value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f mm',dc,abs(dz));
                        else
                            value = sprintf('%.*f',dc,abs(dz));
                        end
                        text(x + t, y + t, z - cs - shift - ph - al, value, 'Color', clr,'tag','textPrescDispl','UserData',abs(dz));    
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws thermal load representation on elements.
        function draw = thermalLoads(draw)
            % Get handle to GUI_Main
            mdata = guidata(findobj('Tag','GUI_Main'));
            
            % Check if thermal loads visualization is on
            if strcmp(get(mdata.viewThermalLoadsButton,'Checked'),'off')
                return
            end
            
            % Parameters
            d = draw.size/80;      % distance between temperature grad symbol and element
            heatClr = [1,0,0];     % heat color
            coldClr = [0,0,1];     % cold color
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for e = 1:draw.drv.nel
                if (draw.drv.elems(e).load.tempVar_X ~= 0) || ...
                   (draw.drv.elems(e).load.tempVar_Y ~= 0) || ...
                   (draw.drv.elems(e).load.tempVar_Z ~= 0)
                    % Get nodal coordinates
                    x1 = draw.drv.elems(e).nodes(1).coord(1);
                    y1 = draw.drv.elems(e).nodes(1).coord(2);
                    z1 = draw.drv.elems(e).nodes(1).coord(3);
                    x2 = draw.drv.elems(e).nodes(2).coord(1);
                    y2 = draw.drv.elems(e).nodes(2).coord(2);
                    z2 = draw.drv.elems(e).nodes(2).coord(3);

                    % Get temperature variation values
                    dtx = draw.drv.elems(e).load.tempVar_X;
                    dty = draw.drv.elems(e).load.tempVar_Y;
                    dtz = draw.drv.elems(e).load.tempVar_Z;

                    % New coordinates of the temperature variation
                    [xi_ty1,yi_ty1,zi_ty1] = draw.coordTransf3D(0, d, 0, x1, y1, z1, e);
                    [xf_ty1,yf_ty1,zf_ty1] = draw.coordTransf3D(0, d, 0, x2, y2, z2, e);
                    [xi_ty2,yi_ty2,zi_ty2] = draw.coordTransf3D(0, -d, 0, x1, y1, z1, e);
                    [xf_ty2,yf_ty2,zf_ty2] = draw.coordTransf3D(0, -d, 0, x2, y2, z2, e);

                    [xi_tz1,yi_tz1,zi_tz1] = draw.coordTransf3D(0, 0, d, x1, y1, z1, e);
                    [xf_tz1,yf_tz1,zf_tz1] = draw.coordTransf3D(0, 0, d, x2, y2, z2, e);
                    [xi_tz2,yi_tz2,zi_tz2] = draw.coordTransf3D(0, 0, -d, x1, y1, z1, e);
                    [xf_tz2,yf_tz2,zf_tz2] = draw.coordTransf3D(0, 0, -d, x2, y2, z2, e);
                    
                    % Check if units are enabled
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        unitsAreOn = true;
                    else
                        unitsAreOn = false;
                    end

                    % Draw temperature variation symbols
                    if dtx > 0
                        line([x1,x2], [y1,y2], [z1,z2], 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2+2*d, sprintf('dTx = %.*f C',dc,dtx), 'Color', heatClr, 'tag', 'textThermalLoads','UserData',{'dTx = ',dtx});
                            case false
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2+2*d, sprintf('dTx = %.*f',dc,dtx), 'Color', heatClr, 'tag', 'textThermalLoads','UserData',{'dTx = ',dtx});
                        end
                    elseif dtx < 0
                        line([x1,x2], [y1,y2], [z1,z2], 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2+2*d, sprintf('dTx = %.*f C',dc,dtx), 'Color', coldClr, 'tag', 'textThermalLoads','UserData',{'dTx = ',dtx});
                            case false
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2+2*d, sprintf('dTx = %.*f',dc,dtx), 'Color', coldClr, 'tag', 'textThermalLoads','UserData',{'dTx = ',dtx});
                        end
                    end

                    if dty > 0
                        plot3([xi_ty1,xf_ty1], [yi_ty1,yf_ty1], [zi_ty1,zf_ty1], 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        plot3([xi_ty2,xf_ty2], [yi_ty2,yf_ty2], [zi_ty2,zf_ty2], 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2, sprintf('dTy = %.*f C',dc,dty), 'Color', heatClr, 'tag', 'textThermalLoads','UserData',{'dTy = ',dty});
                            case false
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2, sprintf('dTy = %.*f',dc,dty), 'Color', heatClr, 'tag', 'textThermalLoads','UserData',{'dTy = ',dty});
                        end
                    elseif dty < 0
                        plot3([xi_ty1,xf_ty1], [yi_ty1,yf_ty1], [zi_ty1,zf_ty1], 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        plot3([xi_ty2,xf_ty2], [yi_ty2,yf_ty2], [zi_ty2,zf_ty2], 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2, sprintf('dTy = %.*f C',dc,dty), 'Color', coldClr, 'tag', 'textThermalLoads','UserData',{'dTy = ',dty});
                            case false
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2, sprintf('dTy = %.*f',dc,dty), 'Color', coldClr, 'tag', 'textThermalLoads','UserData',{'dTy = ',dty});
                        end
                    end

                    if dtz > 0
                        plot3([xi_tz1,xf_tz1], [yi_tz1,yf_tz1], [zi_tz1,zf_tz1], 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        plot3([xi_tz2,xf_tz2], [yi_tz2,yf_tz2], [zi_tz2,zf_tz2], 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2-2*d, sprintf('dTz = %.*f C',dc,dtz), 'Color', heatClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                            case false
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2-2*d, sprintf('dTz = %.*f',dc,dtz), 'Color', heatClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                        end
                    elseif dtz < 0
                        plot3([xi_tz1,xf_tz1], [yi_tz1,yf_tz1], [zi_tz1,zf_tz1], 'Color', heatClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        plot3([xi_tz2,xf_tz2], [yi_tz2,yf_tz2], [zi_tz2,zf_tz2], 'Color', coldClr, 'LineStyle', '-.', 'tag', 'drawThermalLoads');
                        switch unitsAreOn
                            case true
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2-2*d, sprintf('dTz = %.*f C',dc,dtz), 'Color', coldClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                            case false
                                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2-2*d, sprintf('dTz = %.*f',dc,dtz), 'Color', coldClr, 'tag', 'textThermalLoads','UserData',{'dTz = ',dtz});
                        end
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Plots ID number of nodes.
        function draw = nodeID(draw)
            % Parameters
            tx = draw.size/90;  % distance between text and nodal point in X axis
            ty = draw.size/50;  % distance between text and nodal point in Y axis
			tz = draw.size/50;  % distance between text and nodal point in Z axis
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
				z = draw.drv.nodes(n).coord(3);
                
                % Write ID number
                number = sprintf('%d',n);
                text(x + tx, y + ty, z + tz, number, 'FontWeight', 'bold', 'tag', 'textNodeID');
            end
        end
        
        %------------------------------------------------------------------
        % Plots ID number of elements.
        function draw = elementID(draw)
            % Parameters
            tx = draw.size/45;  % distance between text and element in X axis
            ty = draw.size/45;  % distance between text and element in Y axis
			tz = draw.size/45;  % distance between text and element in Z axis
            
            for e = 1:draw.drv.nel
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
				z1 = draw.drv.elems(e).nodes(1).coord(3);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
				z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Write ID number
                number = sprintf('%d',e);
                text((x1+x2)/2 - tx, (y1+y2)/2 - ty, (z1+z2)/2 - tz, number, 'FontWeight', 'bold', 'tag', 'textElemID');
            end
        end
        
        %------------------------------------------------------------------
        % Draws element orientation indication from inital to final node.
        function draw = elementOrientation(draw)
            % Parameters
            t = 1.3;             % distance between text and symbol
            clr = [0,.7,0];      % orientation symbol color
            
            for e = 1:draw.drv.nel
                % Calculate spear length
                l = draw.size/20;
                
                % Get nodal coordinates
                xi = draw.drv.elems(e).nodes(1).coord(1);
                yi = draw.drv.elems(e).nodes(1).coord(2);
                zi = draw.drv.elems(e).nodes(1).coord(3);
                xf = draw.drv.elems(e).nodes(2).coord(1);
                yf = draw.drv.elems(e).nodes(2).coord(2);
                zf = draw.drv.elems(e).nodes(2).coord(3);
                
                % Calculate element local axis X orientation vector
                x = [xf-xi, yf-yi, zf-zi];
                x = l * x / norm(x);
                
                % Get orientation vector on the local XZ plane
                z = [draw.drv.elems(e).vz(1), draw.drv.elems(e).vz(2), draw.drv.elems(e).vz(3)];
                
                % Calculate element local axis Y orientation vector
                y = cross(z,x);
                y = l * y / norm(y);
                
                % Calculate element local axis Z orientation vector
                z = cross(x,y);
                z = l * z/norm(z);
                
                % Draw orientation symbol
                xm = (xi + xf)/2;
                ym = (yi + yf)/2;
                zm = (zi + zf)/2;
                
                X = [xm, xm + x(1)];
                Y = [ym, ym + x(2)];
				Z = [zm, zm + x(3)];
                line(X, Y, Z, 'Color', clr, 'tag', 'drawElemOrient');
                text(xm + t * x(1), ym + t * x(2), zm + t * x(3), 'x', 'Color', clr, 'FontWeight', 'bold', 'tag', 'drawElemOrient');
                
                X = [xm, xm + y(1)];
                Y = [ym, ym + y(2)];
				Z = [zm, zm + y(3)];
                line(X, Y, Z, 'Color', clr, 'tag', 'drawElemOrient');
                text(xm + t * y(1), ym + t * y(2), zm + t * y(3), 'y', 'Color', clr, 'FontWeight', 'bold', 'tag', 'drawElemOrient');
                
                X = [xm, xm + z(1)];
                Y = [ym, ym + z(2)];
				Z = [zm, zm + z(3)];
                line(X, Y, Z, 'Color', clr, 'tag', 'drawElemOrient');
                text(xm + t * z(1), ym + t * z(2), zm + t * z(3), 'z', 'Color', clr, 'FontWeight', 'bold', 'tag', 'drawElemOrient');
            end
        end
        
        %------------------------------------------------------------------
        % Computes deformed configuration scale factor.
        function draw = deformScaleFactor(draw)
            mdata = guidata(findobj('Tag','GUI_Main'));
            sliderm = get(mdata.slider_Scale,'Max');
            
            % Estimate maximum displacement of each element
            % (nodal and internal)
            m = zeros(1,draw.drv.nel);
            for e = 1:draw.drv.nel
                % Get element end nodes IDs
                n1 = draw.drv.elems(e).nodes(1).id;
                n2 = draw.drv.elems(e).nodes(2).id;
                
                % Get maximum nodal displacements
                dx1 = draw.drv.D(draw.drv.ID(1,n1));
                dy1 = draw.drv.D(draw.drv.ID(2,n1));
                dz1 = draw.drv.D(draw.drv.ID(3,n1));
                dx2 = draw.drv.D(draw.drv.ID(1,n2));
                dy2 = draw.drv.D(draw.drv.ID(2,n2));
                dz2 = draw.drv.D(draw.drv.ID(3,n2));
                nodeDispl = [dx1,dy1,dz1,dx2,dy2,dz2];
                maxNode = max(abs(nodeDispl));
                
                % Get maximum estimated internal displacement
                maxInt = max(max(abs(draw.drv.elems(e).intDispl)));
                
                % Get maximum element displacement
                m(e) = max(maxNode,maxInt);
            end
            
            % Get maximum estimated model displacement
            max_disp = max(m);
            
            % Set adapted scale value
            dsf = draw.size/(4*sliderm*max_disp);
            if dsf == inf
                dsf = 0;
            end
            setappdata(0,'deform_sf',dsf);
        end
        
        %------------------------------------------------------------------
        % Draws structure deformed configuration on a given scale.
        % Input arguments:
        %  scale: deformed configuration scale factor
        function draw = deformConfig(draw,scale)
            %  Parameters
            clr = [1,0,0];  % deformed configuration line color
            
            % Calculate deformed configuration coordinates of 50 cross-
            % sections along element local axis X and connect them
            for e = 1:draw.drv.nel
                % Get 50 cross-sections coordinates
                coords = draw.drv.elems(e).intCoords;
                
                % Get element axial and transversal internal displacements
                % in local system
                dl = draw.drv.elems(e).intDispl;
                
                % Get rotation transformation matrix
                rot = draw.drv.elems(e).T;
                
                % Rotate displacements vector to global system
                dg = rot' * dl;
                
                % Deformed configuration global coordinates
                dfg = coords + scale * dg;
                
                % Plot deformed configuration
                line(dfg(1,:), dfg(2,:), dfg(3,:), 'Color', clr, 'tag', 'drawDeformConfig');
            end
        end
        
        %------------------------------------------------------------------
        % Computes axial force diagram scale factor value.
        function draw = axialScaleFactor(draw)
            mdata = guidata(findobj('Tag','GUI_Main'));
            sliderm = get(mdata.slider_Scale,'Max');
            
            % Get maximum axial internal force value of each element
            max_elem = zeros(1,draw.drv.nel);
            for e = 1:draw.drv.nel
                % Get maximum value at element ends
                N1 = draw.drv.elems(e).axial_force(1);
                N2 = draw.drv.elems(e).axial_force(2);
                max_end = max(abs(N1),abs(N2));
                
                % Get maximum internal value
                if ~isempty(draw.drv.elems(e).maxAxialForce)
                    max_int = abs(draw.drv.elems(e).maxAxialForce(1));
                else
                    max_int = 0;
                end
                
                max_elem(e) = max(max_end,max_int);
            end
            
            % Get maximum axial internal force value of model
            max_val = max(max_elem);
            
            % Set adapted scale value
            if max_val == 0
                asf = 0;
            else
                asf = draw.size/(2.5*sliderm*max_val);
            end
            setappdata(0,'axial_sf',asf);
        end
        
        %------------------------------------------------------------------
        % Draws resulting axial force diagram on a given scale.
        % Input arguments:
        %  scale: axial force diagram scale factor
        function draw = axialForce(draw,scale)
            include_constants;
            
            % Parameters
            d = draw.size/80;  % distance between force value and diagram
            clr = [1,0,0];     % diagram line color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            % Get target element ID
            elem = get(mdata.popupmenu_ElementResults,'Value') - 1;
            if elem ~= 0
                i = elem;
                j = i;
            else
                i = 1;
                j = draw.drv.nel;
            end
            
            for e = i:j
                % Get element length
                L = draw.drv.elems(e).length;
                
                % Get element orientation angle cosine with axes X and Y
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
                cz = draw.drv.elems(e).cosine_Z;
                
                % Get element internal axial force value at both ends
                N1 = -draw.drv.elems(e).axial_force(1);
                N2 = draw.drv.elems(e).axial_force(2);
                
                % Avoid plotting garbage
                if abs(N1) < 1e-10
                    N1 = 0;
                end
                if abs(N2) < 1e-10
                    N2 = 0;
                end
                
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                z1 = draw.drv.elems(e).nodes(1).coord(3);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Calculate diagram coordinates according to element
                % orientation
                [xd1,yd1,zd1] = draw.coordTransf3D(0, scale*N1, 0, x1, y1, z1, e);
                [xd2,yd2,zd2] = draw.coordTransf3D(L, scale*N2, 0, x1, y1, z1, e);
                
                % Draw diagram extremities
                Xi = [x1, xd1];
                Yi = [y1, yd1];
                Zi = [z1, zd1];
                line(Xi, Yi, Zi, 'Color', clr, 'tag', 'drawAxialForceDiagram');
                
                Xf = [x2, xd2];
                Yf = [y2, yd2];
                Zf = [z2, zd2];
                line(Xf, Yf, Zf, 'Color', clr, 'tag', 'drawAxialForceDiagram');
                
                % Write force values
                if (abs(N1-N2) < 10e-10) && (abs(N1) > 10e-10) && (isempty(draw.drv.elems(e).load.linearLcl) == 1)
                    [xt,yt,zt] = draw.coordTransf3D(L/2, scale*N1 + d*(N1/abs(N1)), 0, x1, y1, z1, e);
                    
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%+.*f kN',dc,N1);
                    else
                        value = sprintf('%+.*f',dc,N1);
                    end
                    text(xt, yt, zt, value, 'Color', clr, 'tag', 'textAxialForceDiagram','UserData',N1);
                    
                else
                    if abs(N1) > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%+.*f kN',dc,N1);
                        else
                            value1 = sprintf('%+.*f',dc,N1);
                        end
                        text(xd1, yd1, zd1, value1, 'Color', clr, 'tag', 'textAxialForceDiagram','UserData',N1);
                    end
                    
                    if abs(N2) > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value2 = sprintf('%+.*f kN',dc,N2);
                        else
                            value2 = sprintf('%+.*f',dc,N2);
                        end
                        text(xd2, yd2, zd2, value2, 'Color', clr, 'tag', 'textAxialForceDiagram','UserData',N2);
                    end
                end
                
                % Connect diagram extremities:
                % Check if element has distributed load.
                % -If so, calculate shear force value along element length
                % -If not, connect both ends with a straight line
                if (isempty(draw.drv.elems(e).load.uniformLcl) == 0) || ...
                   (isempty(draw.drv.elems(e).load.linearLcl) == 0)
               
                    % Get element 50 point division coords and internal
                    % stress values
                    coords = draw.drv.elems(e).intCoords;
                    N = draw.drv.elems(e).intStresses(1,:);
                    Nmax = draw.drv.elems(e).maxAxialForce;
                    
                    % Avoid plotting numeric garbage
                    if ~all(abs(N) < 10^-10)
                    
                        % Get element basis transformation matrix
                        rot = draw.drv.elems(e).T;

                        % Compute diagram coordinates
                        diagramCoords = rot' * [zeros(1,size(N,2)); scale*N; zeros(1,size(N,2))] + coords;

                        % Plot diagram
                        X = diagramCoords(1,:);
                        Y = diagramCoords(2,:);
                        Z = diagramCoords(3,:);
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawAxialForceDiagram');

                        % Check if there is a maximum value within the diagram
                        if ~isempty(Nmax)
                            % Compute maximum stress value position, in global
                            % coordinates
                            NmaxGblCoords = [x1 + Nmax(2) * cx;
                                             y1 + Nmax(2) * cy;
                                             z1 + Nmax(2) * cz];

                            % Compute max point in diagram and text coordinates
                            maxPointCoords = rot' * [0; scale*Nmax(1); 0] + NmaxGblCoords;
                            maxTextCoords = rot' * [0; scale*Nmax(1) + d*(Nmax(1)/abs(Nmax(1))); 0] + NmaxGblCoords;

                            % Plot point indicating maximum value
                            xp = maxPointCoords(1);
                            yp = maxPointCoords(2);
                            zp = maxPointCoords(3);
                            scatter3(xp, yp, zp, 50, clr, '.', 'tag', 'drawAxialForceDiagram')

                            % Plot maximum value in text
                            xe = maxTextCoords(1);
                            ye = maxTextCoords(2);
                            ze = maxTextCoords(3);
                            % Avoid plotting text too close to element end
                            if abs(Nmax(2) - L) >= L/25 && Nmax(2) >= L/25
                                if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                    value = sprintf('%+.*f kN',dc,Nmax(1));
                                else
                                    value = sprintf('%+.*f',dc,Nmax(1));
                                end
                                text(xe, ye, ze, value, 'Color', clr, 'tag', 'textAxialForceDiagram','UserData',Nmax(1));
                            end
                        end
                    else
                        % Connect both ends with a straight line
                        X = [xd1, xd2];
                        Y = [yd1, yd2];
                        Z = [zd1, zd2];
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawAxialForceDiagram');
                    end
                    
                else
                    % Connect both ends with a straight line
                    X = [xd1, xd2];
                    Y = [yd1, yd2];
                    Z = [zd1, zd2]; 
                    line(X, Y, Z, 'Color', clr, 'tag', 'drawAxialForceDiagram');
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws resulting torsion moment diagram.
        function draw = torsionMoment(draw)
            % Parameters
            clr = [1,0,0];         % diagram color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            % Get target element ID
            elem = get(mdata.popupmenu_ElementResults,'Value') - 1;
            if elem ~= 0
                i = elem;
                j = i;
            else
                i = 1;
                j = draw.drv.nel;
            end
            
            for e = i:j
                % Get element internal tortion moment value (always uniform
                % for grillage models) and convert it to string
                T = -draw.drv.elems(e).torsion_moment(1);
                if T ~= 0
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%+.*f kNm',dc,T);
                    else
                        value = sprintf('%+.*f',dc,T);
                    end
                else
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%.*f kNm',dc,abs(T));
                    else
                        value = sprintf('%.*f',dc,abs(T));
                    end
                    T = abs(T);
                end
                
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                z1 = draw.drv.elems(e).nodes(1).coord(3);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Write tortion moment value in the middle of the element
                text((x1+x2)/2, (y1+y2)/2, (z1+z2)/2, value, 'Color', clr, 'tag', 'textTorsionDiagram','UserData',T);
            end
        end
        
        %------------------------------------------------------------------
        % Computes shear force diagram scale factor value in XY plane.
        function draw = shearScaleFactor_XY(draw)
            mdata = guidata(findobj('Tag','GUI_Main'));
            sliderm = get(mdata.slider_Scale,'Max');
            
            % Get maximum internal shear force value of each element
            max_elem = zeros(1,draw.drv.nel);
            for e = 1:draw.drv.nel
                % Get maximum value at element ends
                Q1_Y = draw.drv.elems(e).shear_force_Y(1);
                Q2_Y = draw.drv.elems(e).shear_force_Y(2);
                max_end = max(abs(Q1_Y),abs(Q2_Y));
                
                % Get maximum internal value
                if ~isempty(draw.drv.elems(e).maxShearForce_XY)
                    max_int = abs(draw.drv.elems(e).maxShearForce_XY(1));
                else
                    max_int = 0;
                end
                
                max_elem(e) = max(max_end,max_int);
            end
            
            % Get maximum shear internal force value of model
            max_val = max(max_elem);
            
            % Set adapted scale value
            if max_val == 0
                ssf = 0;
            else
                ssf = draw.size/(2.5*sliderm*max_val);
            end
            setappdata(0,'shearXY_sf',ssf);
        end
        
        %------------------------------------------------------------------
        % Draws resulting shear force diagram in XY plane on a given scale.
        % Input arguments:
        %  scale: shear force diagram scale factor
        function draw = shearForce_XY(draw,scale)
            include_constants;
            
            % Parameters
            d = draw.size/60;  % distance between force value and diagram
            clr = [1,0,0];     % diagram line color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            % Get target element ID
            elem = get(mdata.popupmenu_ElementResults,'Value') - 1;
            if elem ~= 0
                i = elem;
                j = i;
            else
                i = 1;
                j = draw.drv.nel;
            end
            
            for e = i:j
                % Get element length
                L = draw.drv.elems(e).length;
                
                % Get element orientation angle cosine with axes X, Y and Z
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
                cz = draw.drv.elems(e).cosine_Z;
                
                % Get element internal shear force value at both ends
                Q1 = draw.drv.elems(e).shear_force_Y(1);
                Q2 = -draw.drv.elems(e).shear_force_Y(2);
                
                % Avoid plotting garbage
                if abs(Q1) < 1e-10
                    Q1 = 0;
                end
                if abs(Q2) < 1e-10
                    Q2 = 0;
                end
                
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                z1 = draw.drv.elems(e).nodes(1).coord(3);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Calculate diagram extrimities coordinates
                [xd1,yd1,zd1] = draw.coordTransf3D(0, scale*Q1, 0, x1, y1, z1, e);
                [xd2,yd2,zd2] = draw.coordTransf3D(0, scale*Q2, 0, x2, y2, z2, e);
                
                % Draw diagram extremities
                Xi = [x1, xd1];
                Yi = [y1, yd1];
                Zi = [z1, zd1];
                line(Xi, Yi, Zi, 'Color', clr, 'tag', 'drawShearForceXYDiagram');
                
                Xf = [x2, xd2];
                Yf = [y2, yd2];
                Zf = [z2, zd2];
                line(Xf, Yf, Zf, 'Color', clr, 'tag', 'drawShearForceXYDiagram');
                
                % Write force values
                if (abs(Q1-Q2) < 10e-10) && (abs(Q1) > 10e-10) && (isempty(draw.drv.elems(e).load.linearLcl) == 1)
                    [xt,yt,zt] = draw.coordTransf3D(L/2, scale*Q1 + d*(Q1/abs(Q1)), 0, x1, y1, z1, e);
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%+.*f kN',dc,Q1);
                    else
                        value = sprintf('%+.*f',dc,Q1);
                    end
                    text(xt, yt, zt, value, 'Color', clr, 'tag', 'textShearForceXYDiagram','UserData',Q1);
                    
                else
                    if abs(Q1) > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%+.*f kN',dc,Q1);
                        else
                            value1 = sprintf('%+.*f',dc,Q1);
                        end
                        text(xd1, yd1, zd1, value1, 'Color', clr, 'tag', 'textShearForceXYDiagram','UserData',Q1);
                    end
                    
                    if abs(Q2) > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value2 = sprintf('%+.*f kN',dc,Q2);
                        else
                            value2 = sprintf('%+.*f',dc,Q2);
                        end
                        text(xd2, yd2, zd2, value2, 'Color', clr, 'tag', 'textShearForceXYDiagram','UserData',Q2);
                    end
                end
                
                % Connect diagram extremities:
                % Check if element has distributed load.
                % -If so, calculate shear force value along element length
                % -If not, connect both ends with a straight line
                if (isempty(draw.drv.elems(e).load.uniformLcl) == 0) || ...
                   (isempty(draw.drv.elems(e).load.linearLcl) == 0)
               
                    % Get element 50 point division coords and internal
                    % stress values
                    coords = draw.drv.elems(e).intCoords;
                    Q = draw.drv.elems(e).intStresses(2,:);
                    Qmax = draw.drv.elems(e).maxShearForce_XY;
                    
                    % Avoid plotting numeric garbage
                    if ~all(abs(Q) < 10^-10)
                        
                        % Get element basis transformation matrix
                        rot = draw.drv.elems(e).T;

                        % Compute diagram coordinates
                        diagramCoords = rot' * [zeros(1,size(Q,2)); scale*Q; zeros(1,size(Q,2))] + coords;

                        % Plot diagram
                        X = diagramCoords(1,:);
                        Y = diagramCoords(2,:);
                        Z = diagramCoords(3,:);
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawShearForceXYDiagram');

                        % Check if there is a maximum value within the diagram
                        if ~isempty(Qmax)
                            % Compute maximum stress value position, in global
                            % coordinates
                            QmaxGblCoords = [x1 + Qmax(2) * cx;
                                             y1 + Qmax(2) * cy;
                                             z1 + Qmax(2) * cz];

                            % Compute max point in diagram and text coordinates
                            maxPointCoords = rot' * [0; scale*Qmax(1); 0] + QmaxGblCoords;
                            maxTextCoords = rot' * [0; scale*Qmax(1) + d*(Qmax(1)/abs(Qmax(1))); 0] + QmaxGblCoords;

                            % Plot point indicating maximum value
                            xp = maxPointCoords(1);
                            yp = maxPointCoords(2);
                            zp = maxPointCoords(3);
                            scatter3(xp, yp, zp, 50, clr, '.', 'tag', 'drawShearForceXYDiagram')

                            % Plot maximum value in text
                            xe = maxTextCoords(1);
                            ye = maxTextCoords(2);
                            ze = maxTextCoords(3);
                            % Avoid plotting text too close to element end
                            if abs(Qmax(2) - L) >= L/25 && Qmax(2) >= L/25
                                if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                    value = sprintf('%+.*f kN',dc,Qmax(1));
                                else
                                    value = sprintf('%+.*f',dc,Qmax(1));
                                end
                                text(xe, ye, ze, value, 'Color', clr, 'tag', 'textShearForceXYDiagram','UserData',Qmax(1));
                            end
                        end
                    else
                        % Connect both ends with a straight line
                        X = [xd1, xd2];
                        Y = [yd1, yd2];
                        Z = [zd1, zd2];
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawShearForceXYDiagram');
                    end
                    
                else
                    % Connect both ends with a straight line
                    X = [xd1, xd2];
                    Y = [yd1, yd2];
                    Z = [zd1, zd2];
                    line(X, Y, Z, 'Color', clr, 'tag', 'drawShearForceXYDiagram');
                end
            end
        end
        
        %------------------------------------------------------------------
        % Computes shear force diagram scale factor value in XZ plane.
        function draw = shearScaleFactor_XZ(draw)
            mdata = guidata(findobj('Tag','GUI_Main'));
            sliderm = get(mdata.slider_Scale,'Max');
            
            % Get maximum internal shear force value of each element
            max_elem = zeros(1,draw.drv.nel);
            for e = 1:draw.drv.nel
                % Get maximum value at element ends
                Q1_Z = draw.drv.elems(e).shear_force_Z(1);
                Q2_Z = draw.drv.elems(e).shear_force_Z(2);
                max_end = max(abs(Q1_Z),abs(Q2_Z));
                
                % Get maximum internal value
                if ~isempty(draw.drv.elems(e).maxShearForce_XZ)
                    max_int = abs(draw.drv.elems(e).maxShearForce_XZ(1));
                else
                    max_int = 0;
                end
                
                max_elem(e) = max(max_end,max_int);
            end
            
            % Get maximum shear internal force value of model
            max_val = max(max_elem);
            
            % Set adapted scale value
            if max_val == 0
                ssf = 0;
            else
                ssf = draw.size/(2.5*sliderm*max_val);
            end
            setappdata(0,'shearXZ_sf',ssf);
        end
        
        %------------------------------------------------------------------
        % Draws resulting shear force diagram in XZ plane on a given scale.
        % Input arguments:
        %  scale: shear force diagram scale factor
        function draw = shearForce_XZ(draw,scale)
            include_constants;
            
            % Parameters
            d = draw.size/60;  % distance between force value and diagram
            clr = [1,0,0];     % diagram line color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            % Get target element ID
            elem = get(mdata.popupmenu_ElementResults,'Value') - 1;
            if elem ~= 0
                i = elem;
                j = i;
            else
                i = 1;
                j = draw.drv.nel;
            end
            
            for e = i:j
                % Get element length
                L = draw.drv.elems(e).length;
                
                % Get element orientation angle cosine with axes X, Y and Z
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
                cz = draw.drv.elems(e).cosine_Z;
                
                % Get element internal shear force value at both ends
                Q1 = draw.drv.elems(e).shear_force_Z(1);
                Q2 = -draw.drv.elems(e).shear_force_Z(2);
                
                % Avoid plotting garbage
                if abs(Q1) < 1e-10
                    Q1 = 0;
                end
                if abs(Q2) < 1e-10
                    Q2 = 0;
                end
                
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                z1 = draw.drv.elems(e).nodes(1).coord(3);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Calculate diagram extrimities coordinates
                [xd1,yd1,zd1] = draw.coordTransf3D(0, 0, scale*Q1, x1, y1, z1, e);
                [xd2,yd2,zd2] = draw.coordTransf3D(0, 0, scale*Q2, x2, y2, z2, e);
                
                % Draw diagram extremities
                Xi = [x1, xd1];
                Yi = [y1, yd1];
                Zi = [z1, zd1];
                line(Xi, Yi, Zi, 'Color', clr, 'tag', 'drawShearForceXZDiagram');
                
                Xf = [x2, xd2];
                Yf = [y2, yd2];
                Zf = [z2, zd2];
                line(Xf, Yf, Zf, 'Color', clr, 'tag', 'drawShearForceXZDiagram');
                
                % Write force values
                if (abs(Q1-Q2) < 10e-10) && (abs(Q1) > 10e-10) && (isempty(draw.drv.elems(e).load.linearLcl) == 1)
                    [xt,yt,zt] = draw.coordTransf3D(L/2, 0, scale*Q1 + d*(Q1/abs(Q1)), x1, y1, z1, e);
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%+.*f kN',dc,Q1);
                    else
                        value = sprintf('%+.*f',dc,Q1);
                    end
                    text(xt, yt, zt, value, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Q1);
                    
                else
                    if abs(Q1) > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%+.*f kN',dc,Q1);
                        else
                            value1 = sprintf('%+.*f',dc,Q1);
                        end
                        text(xd1, yd1, zd1, value1, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Q1);
                    end
                    
                    if abs(Q2) > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value2 = sprintf('%+.*f kN',dc,Q2);
                        else
                            value2 = sprintf('%+.*f',dc,Q2);
                        end
                        text(xd2, yd2, zd2, value2, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Q2);
                    end
                end
                
                % Connect diagram extremities:
                % Check if element has distributed load.
                % -If so, calculate shear force value along element length
                % -If not, connect both ends with a straight line
                if (isempty(draw.drv.elems(e).load.uniformLcl) == 0) || ...
                   (isempty(draw.drv.elems(e).load.linearLcl) == 0)
               
                    % Get element 50 point division coords and internal
                    % stress values
                    coords = draw.drv.elems(e).intCoords;
                    Q = draw.drv.elems(e).intStresses(3,:);
                    Qmax = draw.drv.elems(e).maxShearForce_XZ;
                    
                    % Avoid plotting numeric garbage
                    if ~all(abs(Q) < 10^-10)
                    
                        % Get element basis transformation matrix
                        rot = draw.drv.elems(e).T;

                        % Compute diagram coordinates
                        diagramCoords = rot' * [zeros(1,size(Q,2)); zeros(1,size(Q,2)); scale*Q] + coords;

                        % Plot diagram
                        X = diagramCoords(1,:);
                        Y = diagramCoords(2,:);
                        Z = diagramCoords(3,:);
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawShearForceXZDiagram');

                        % Check if there is a maximum value within the diagram
                        if ~isempty(Qmax)
                            % Compute maximum stress value position, in global
                            % coordinates
                            QmaxGblCoords = [x1 + Qmax(2) * cx;
                                             y1 + Qmax(2) * cy;
                                             z1 + Qmax(2) * cz];

                            % Compute max point in diagram and text coordinates
                            maxPointCoords = rot' * [0; 0; scale*Qmax(1)] + QmaxGblCoords;
                            maxTextCoords = rot' * [0; 0; scale*Qmax(1) + d*(Qmax(1)/abs(Qmax(1)))] + QmaxGblCoords;

                            % Plot point indicating maximum value
                            xp = maxPointCoords(1);
                            yp = maxPointCoords(2);
                            zp = maxPointCoords(3);
                            scatter3(xp, yp, zp, 50, clr, '.', 'tag', 'drawShearForceXZDiagram')

                            % Plot maximum value in text
                            xe = maxTextCoords(1);
                            ye = maxTextCoords(2);
                            ze = maxTextCoords(3);
                            % Avoid plotting text too close to element end
                            if abs(Qmax(2) - L) >= L/25 && Qmax(2) >= L/25
                                if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                    value = sprintf('%+.*f kN',dc,Qmax(1));
                                else
                                    value = sprintf('%+.*f',dc,Qmax(1));
                                end
                                text(xe, ye, ze, value, 'Color', clr, 'tag', 'textShearForceXZDiagram','UserData',Qmax(1));
                            end
                        end
                    else
                        % Connect both ends with a straight line
                        X = [xd1, xd2];
                        Y = [yd1, yd2];
                        Z = [zd1, zd2];
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawShearForceXZDiagram');
                    end
                    
                else
                    % Connect both ends with a straight line
                    X = [xd1, xd2];
                    Y = [yd1, yd2];
                    Z = [zd1, zd2];
                    line(X, Y, Z, 'Color', clr, 'tag', 'drawShearForceXZDiagram');
                end
            end
        end
        
        %------------------------------------------------------------------
        % Computes bending moment diagram scale factor value in XY plane.
        function draw = bendingMomentScaleFactor_XY(draw)
            mdata = guidata(findobj('Tag','GUI_Main'));
            sliderm = get(mdata.slider_Scale,'Max');
            
            % Get maximum internal bending moment value of each element
            max_elem = zeros(1,draw.drv.nel);
            for e = 1:draw.drv.nel
                % Get maximum value at element ends
                M1_Z = draw.drv.elems(e).bending_moment_Z(1);
                M2_Z = draw.drv.elems(e).bending_moment_Z(2);
                max_end = max(abs(M1_Z), abs(M2_Z));
                
                % Get maximum internal value
                if ~isempty(draw.drv.elems(e).maxBendMoment_XY)
                    max_int = max(abs(draw.drv.elems(e).maxBendMoment_XY(:,1)));
                else
                    max_int = 0;
                end
                
                max_elem(e) = max(max_end,max_int);
            end
            
            % Get maximum axial internal force value of model
            max_val = max(max_elem);
            
            % Set adapted scale value
            if max_val == 0
                bsf = 0;
            else
                bsf = draw.size/(2.5*sliderm*max_val);
            end
            setappdata(0,'bendingXY_sf',bsf);
        end
        
        %------------------------------------------------------------------
        % Draws resulting bending moment diagram in XY plane on a given scale.
        % Input arguments:
        %  scale: bending moment diagram scale factor
        function draw = bendingMoment_XY(draw,scale)
            include_constants
            
            % Parameters
            d = draw.size/60;  % distance between force value and diagram
            clr = [1,0,0];     % diagram line color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            % Get target element ID
            elem = get(mdata.popupmenu_ElementResults,'Value') - 1;
            if elem ~= 0
                i = elem;
                j = i;
            else
                i = 1;
                j = draw.drv.nel;
            end
            
            for e = i:j
                % Get element length
                L = draw.drv.elems(e).length;
                
                % Get element orientation angle with X axis
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
                cz = draw.drv.elems(e).cosine_Z;
                
                % Get element internal forces value at both ends
                M1 = -draw.drv.elems(e).bending_moment_Z(1);
                M2 = draw.drv.elems(e).bending_moment_Z(2);
                
                % Avoid plotting garbage
                if abs(M1) < 1e-10
                    M1 = 0;
                end
                if abs(M2) < 1e-10
                    M2 = 0;
                end
                
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                z1 = draw.drv.elems(e).nodes(1).coord(3);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Calculate diagram extrimities coordinates
                [xd1,yd1,zd1] = draw.coordTransf3D(0, -scale*M1, 0, x1, y1, z1, e);
                [xd2,yd2,zd2] = draw.coordTransf3D(0, -scale*M2, 0, x2, y2, z2, e);
                
                % Draw Mz diagram extremities
                Xi = [x1, xd1];
                Yi = [y1, yd1];
                Zi = [z1, zd1];
                line(Xi, Yi, Zi, 'Color', clr, 'tag', 'drawBendMomentXYDiagram');
                
                Xf = [x2, xd2];
                Yf = [y2, yd2];
                Zf = [z2, zd2];
                line(Xf, Yf, Zf, 'Color', clr, 'tag', 'drawBendMomentXYDiagram');
                
                % Write moment values
                if (abs(M1-M2) < 10e-10) && (abs(M1) > 10e-10) &&...
                   (isempty(draw.drv.elems(e).load.uniformGbl) == 1) &&...
                   (isempty(draw.drv.elems(e).load.linearGbl) == 1)
                    [xt,yt,zt] = draw.coordTransf3D(L/2, -scale*M1 - d*(M1/abs(M1)), 0, x1, y1, z1, e);
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%+.*f kNm',dc,M1);
                    else
                        value = sprintf('%+.*f',dc,M1);
                    end
                    text(xt, yt, zt, value, 'Color', clr, 'tag', 'textBendMomentXYDiagram','UserData',M1);
                    
                else
                    if abs(M1) > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%+.*f kNm',dc,M1);
                        else
                            value1 = sprintf('%+.*f',dc,M1);
                        end
                        text(xd1, yd1, zd1, value1, 'Color', clr, 'tag', 'textBendMomentXYDiagram','UserData',M1);
                    end
                    
                    if abs(M2) > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value2 = sprintf('%+.*f kNm',dc,M2);
                        else
                            value2 = sprintf('%+.*f',dc,M2);
                        end
                        text(xd2, yd2, zd2, value2, 'Color', clr, 'tag', 'textBendMomentXYDiagram','UserData',M2);
                    end
                end
                
                % Connect diagram extremities:
                % Check if element has distributed load.
                % -If so, calculate shear force value along element length
                % -If not, connect both ends with a straight line
                if (isempty(draw.drv.elems(e).load.uniformLcl) == 0) || ...
                   (isempty(draw.drv.elems(e).load.linearLcl) == 0)
               
                    % Get element 50 point division coords and internal
                    % stress values
                    coords = draw.drv.elems(e).intCoords;
                    M = draw.drv.elems(e).intStresses(4,:);
                    Mmax = draw.drv.elems(e).maxBendMoment_XY;
                    
                    % Avoid plotting numeric garbage
                    if ~all(abs(M) < 10^-10)
                    
                        % Get element basis transformation matrix
                        rot = draw.drv.elems(e).T;
                        
                        % Compute diagram coordinates
                        diagramCoords = rot' * [zeros(1,size(M,2)); -scale*M; zeros(1,size(M,2))] + coords;
                        
                        % Plot diagram
                        X = diagramCoords(1,:);
                        Y = diagramCoords(2,:);
                        Z = diagramCoords(3,:);
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawBendMomentXYDiagram');
                        
                        % Check if there is a maximum value within the diagram
                        if ~isempty(Mmax)
                            % Compute maximum stress value position, in global
                            % coordinates
                            MmaxGblCoords = [x1 + (Mmax(:,2))' * cx;
                                             y1 + (Mmax(:,2))' * cy;
                                             z1 + (Mmax(:,2))' * cz];
                            
                            % Get maximum stress values
                            Mmax_val = (Mmax(:,1))';
                            
                            % Get number of maximum values within diagram
                            nm = size(Mmax_val,2);
                            
                            % Compute max point in diagram and text coordinates
                            maxPointCoords = rot' * [zeros(1,nm); -scale*Mmax_val; zeros(1,nm)] + MmaxGblCoords;
                            maxTextCoords = rot' * [zeros(1,nm); -scale*Mmax_val - d*(Mmax_val./abs(Mmax_val)); zeros(1,nm)] + MmaxGblCoords;
                            
                            % Plot point indicating maximum value
                            xp = maxPointCoords(1,:);
                            yp = maxPointCoords(2,:);
                            zp = maxPointCoords(3,:);
                            scatter3(xp, yp, zp, 50, clr, '.', 'tag', 'drawBendMomentXYDiagram')
                            
                            % Plot maximum value in text
                            xe = maxTextCoords(1,:);
                            ye = maxTextCoords(2,:);
                            ze = maxTextCoords(3,:);
                            for np = 1:nm % 2 rounds max
                                % Avoid plotting text too close to element end
                                if abs(Mmax(np,2) - L) >= L/25 && Mmax(np,2) >= L/25
                                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                        value = sprintf('%+.*f kNm',dc,Mmax_val(np));
                                    else
                                        value = sprintf('%+.*f',dc,Mmax_val(np));
                                    end
                                    text(xe(np), ye(np), ze(np), value, 'Color', clr, 'tag', 'textBendMomentXYDiagram','UserData',Mmax_val(np));
                                end
                            end
                        end
                    else
                        % Connect both ends with a straight line
                        X = [xd1, xd2];
                        Y = [yd1, yd2];
                        Z = [zd1, zd2];
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawBendMomentXYDiagram');
                    end
                    
                else
                    % Connect both ends with a straight line
                    X = [xd1, xd2];
                    Y = [yd1, yd2];
                    Z = [zd1, zd2];
                    line(X, Y, Z, 'Color', clr, 'tag', 'drawBendMomentXYDiagram');
                end
            end
        end
        
        %------------------------------------------------------------------
        % Computes bending moment diagram scale factor value in XZ plane.
        function draw = bendingMomentScaleFactor_XZ(draw)
            mdata = guidata(findobj('Tag','GUI_Main'));
            sliderm = get(mdata.slider_Scale,'Max');
            
            % Get maximum internal bending moment value of each element
            max_elem = zeros(1,draw.drv.nel);
            for e = 1:draw.drv.nel
                % Get maximum value at element ends
                M1_Y = draw.drv.elems(e).bending_moment_Y(1);
                M2_Y = draw.drv.elems(e).bending_moment_Y(2);
                max_end = max(abs(M1_Y), abs(M2_Y));
                
                % Get maximum internal value
                if ~isempty(draw.drv.elems(e).maxBendMoment_XZ)
                    max_int = max(abs(draw.drv.elems(e).maxBendMoment_XZ(:,1)));
                else
                    max_int = 0;
                end
                
                max_elem(e) = max(max_end,max_int);
            end
            
            % Get maximum axial internal force value of model
            max_val = max(max_elem);
            
            % Set adapted scale value
            if max_val == 0
                bsf = 0;
            else
                bsf = draw.size/(2.5*sliderm*max_val);
            end
            setappdata(0,'bendingXZ_sf',bsf);
        end
        
        %------------------------------------------------------------------
        % Draws resulting bending moment diagram in XZ plane on a given scale.
        function draw = bendingMoment_XZ(draw,scale)
            include_constants
            
            % Parameters
            d = draw.size/60;  % distance between force value and diagram
            clr = [1,0,0];     % diagram line color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            % Get target element ID
            elem = get(mdata.popupmenu_ElementResults,'Value') - 1;
            if elem ~= 0
                i = elem;
                j = i;
            else
                i = 1;
                j = draw.drv.nel;
            end
            
            for e = i:j
                % Get element length
                L = draw.drv.elems(e).length;
                
                % Get element orientation angle with X axis
                cx = draw.drv.elems(e).cosine_X;
                cy = draw.drv.elems(e).cosine_Y;
                cz = draw.drv.elems(e).cosine_Z;
                
                % Get element internal forces value at both ends
                M1 = draw.drv.elems(e).bending_moment_Y(1);
                M2 = -draw.drv.elems(e).bending_moment_Y(2);
                
                % Avoid plotting garbage
                if abs(M1) < 1e-10
                    M1 = 0;
                end
                if abs(M2) < 1e-10
                    M2 = 0;
                end
                
                % Get nodal coordinates
                x1 = draw.drv.elems(e).nodes(1).coord(1);
                y1 = draw.drv.elems(e).nodes(1).coord(2);
                z1 = draw.drv.elems(e).nodes(1).coord(3);
                x2 = draw.drv.elems(e).nodes(2).coord(1);
                y2 = draw.drv.elems(e).nodes(2).coord(2);
                z2 = draw.drv.elems(e).nodes(2).coord(3);
                
                % Calculate diagram extrimities coordinates
                [xd1,yd1,zd1] = draw.coordTransf3D(0, 0, -scale*M1, x1, y1, z1, e);
                [xd2,yd2,zd2] = draw.coordTransf3D(0, 0, -scale*M2, x2, y2, z2, e);
                
                % Draw Mz diagram extremities
                Xi = [x1, xd1];
                Yi = [y1, yd1];
                Zi = [z1, zd1];
                line(Xi, Yi, Zi, 'Color', clr, 'tag', 'drawBendMomentXZDiagram');
                
                Xf = [x2, xd2];
                Yf = [y2, yd2];
                Zf = [z2, zd2];
                line(Xf, Yf, Zf, 'Color', clr, 'tag', 'drawBendMomentXZDiagram');
                
                % Write force values
                if (abs(M1-M2) < 10e-10) && (abs(M1) > 10e-10) &&...
                   (isempty(draw.drv.elems(e).load.uniformGbl) == 1) &&...
                   (isempty(draw.drv.elems(e).load.linearGbl) == 1)
                    [xt,yt,zt] = draw.coordTransf3D(L/2, 0, -scale*M1 - d*(M1/abs(M1)), x1, y1, z1, e);
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%+.*f kNm',dc,M1);
                    else
                        value = sprintf('%+.*f',dc,M1);
                    end
                    text(xt, yt, zt, value, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',M1);
                    
                else
                    if abs(M1) > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value1 = sprintf('%+.*f kNm',dc,M1);
                        else
                            value1 = sprintf('%+.*f',dc,M1);
                        end
                        text(xd1, yd1, zd1, value1, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',M1);
                    end
                    
                    if abs(M2) > 10e-10
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value2 = sprintf('%+.*f kNm',dc,M2);
                        else
                            value2 = sprintf('%+.*f',dc,M2);
                        end
                        text(xd2, yd2, zd2, value2, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',M2);
                    end
                end
                
                % Connect diagram extremities:
                % Check if element has distributed load.
                % -If so, calculate shear force value along element length
                % -If not, connect both ends with a straight line
                if (isempty(draw.drv.elems(e).load.uniformLcl) == 0) || ...
                   (isempty(draw.drv.elems(e).load.linearLcl) == 0)
               
                    % Get element 50 point division coords and internal
                    % stress values
                    coords = draw.drv.elems(e).intCoords;
                    M = draw.drv.elems(e).intStresses(5,:);
                    Mmax = draw.drv.elems(e).maxBendMoment_XZ;
                    
                    % Avoid plotting numeric garbage
                    if ~all(abs(M) < 10^-10)
                    
                        % Get element basis transformation matrix
                        rot = draw.drv.elems(e).T;

                        % Compute diagram coordinates
                        diagramCoords = rot' * [zeros(1,size(M,2)); zeros(1,size(M,2)); -scale*M] + coords;

                        % Plot diagram
                        X = diagramCoords(1,:);
                        Y = diagramCoords(2,:);
                        Z = diagramCoords(3,:);
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawBendMomentXZDiagram');

                        % Check if there is a maximum value within the diagram
                        if ~isempty(Mmax)
                            % Compute maximum stress value position, in global
                            % coordinates
                            MmaxGblCoords = [x1 + (Mmax(:,2))' * cx;
                                             y1 + (Mmax(:,2))' * cy;
                                             z1 + (Mmax(:,2))' * cz];

                            % Get maximum stress values
                            Mmax_val = (Mmax(:,1))';

                            % Get number of maximum values within diagram
                            nm = size(Mmax_val,2);

                            % Compute max point in diagram and text coordinates
                            maxPointCoords = rot' * [zeros(1,nm); zeros(1,nm); -scale*Mmax_val] + MmaxGblCoords;
                            maxTextCoords = rot' * [zeros(1,nm); zeros(1,nm); -scale*Mmax_val - d*(Mmax_val./abs(Mmax_val))] + MmaxGblCoords;

                            % Plot point indicating maximum value
                            xp = maxPointCoords(1,:);
                            yp = maxPointCoords(2,:);
                            zp = maxPointCoords(3,:);
                            scatter3(xp, yp, zp, 50, clr, '.', 'tag', 'drawBendMomentXZDiagram')

                            % Plot maximum value in text
                            xe = maxTextCoords(1,:);
                            ye = maxTextCoords(2,:);
                            ze = maxTextCoords(3,:);
                            for np = 1:nm % 2 rounds max
                                % Avoid plotting text too close to element end
                                if abs(Mmax(np,2) - L) >= L/25 && Mmax(np,2) >= L/25
                                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                                        value = sprintf('%+.*f kNm',dc,Mmax_val(np));
                                    else
                                        value = sprintf('%+.*f',dc,Mmax_val(np));
                                    end
                                    text(xe(np), ye(np), ze(np), value, 'Color', clr, 'tag', 'textBendMomentXZDiagram','UserData',Mmax_val(np));
                                end
                            end
                        end
                    else
                        % Connect both ends with a straight line
                        X = [xd1, xd2];
                        Y = [yd1, yd2];
                        Z = [zd1, zd2];
                        line(X, Y, Z, 'Color', clr, 'tag', 'drawBendMomentXZDiagram');
                    end
                    
                else
                    % Connect both ends with a straight line
                    X = [xd1, xd2];
                    Y = [yd1, yd2];
                    Z = [zd1, zd2];
                    line(X, Y, Z, 'Color', clr, 'tag', 'drawBendMomentXZDiagram');
                end
            end
        end
        
        %------------------------------------------------------------------
        % Draws reactions indication next to nodal supports.
        function draw = reactions(draw)
            include_constants;
            % Parameters
            shift = draw.size/100;  % distance between reaction symbol and support
            ph = draw.size/30;      % translation constraint symbol (pyramid height)
            sh = draw.size/20;      % spring symbol height
            al = draw.size/10;      % reaction symbol size (arrow length)
            ah = draw.size/30;      % reaction symbol size (pyramid height)
            ab = draw.size/70;      % reaction symbol size (pyramid base)
            t = draw.size/45;       % distance between text and nodal point
            clr = [0,0,1];          % reaction symbol color
            mdata = guidata(findobj('Tag','GUI_Main'));    % handle to main GUI
            dc = getappdata(0,'decPrec'); % decimal precision
            
            for n = 1:draw.drv.nnp
                % Get nodal coordinates
                x = draw.drv.nodes(n).coord(1);
                y = draw.drv.nodes(n).coord(2);
                z = draw.drv.nodes(n).coord(3);
                
                % Get reactions values
                rx = draw.drv.F(draw.drv.ID(1,n));
                ry = draw.drv.F(draw.drv.ID(2,n));
                rz = draw.drv.F(draw.drv.ID(3,n));
                mx = draw.drv.F(draw.drv.ID(4,n));
                my = draw.drv.F(draw.drv.ID(5,n));
                mz = draw.drv.F(draw.drv.ID(6,n));
                
                % Rotation constraint symbol (square side)
                cs = 0;
                
                if (draw.drv.nodes(n).ebc(4) == FIXED_DOF) && (draw.drv.nodes(n).ebc(5) == FIXED_DOF) && (draw.drv.nodes(n).ebc(6) == FIXED_DOF)
                    % Update rotation constraint symbol size value
                    cs = draw.size/100;
                elseif (draw.drv.nodes(n).ebc(4) == SPRING_DOF) && (draw.drv.nodes(n).ebc(5) == SPRING_DOF) && (draw.drv.nodes(n).ebc(6) == SPRING_DOF)
                    cs = draw.size/70;
                end    
                
                % Check if rotation is fixed and draw reaction indication
                if ((draw.drv.nodes(n).ebc(4) == FIXED_DOF) && (draw.drv.nodes(n).ebc(5) == FIXED_DOF) && (draw.drv.nodes(n).ebc(6) == FIXED_DOF))...
                   || ((draw.drv.nodes(n).ebc(4) == SPRING_DOF) && (draw.drv.nodes(n).ebc(5) == SPRING_DOF) && (draw.drv.nodes(n).ebc(6) == SPRING_DOF)) 
          
                    if mx >= 0
                        draw.moment3D(draw, x + cs + shift + al, y, z, al, ah/1.4, ab, 'x+', clr,'drawReactions');
                        
                        % Write reaction value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mx));
                        else
                            value = sprintf('%.*f',dc,abs(mx));
                        end
                        text(x + cs + shift + al, y + t, z + t, value, 'Color', clr,'tag','textMomentReactions','UserData',abs(mx));
                        
                    else
                        draw.moment3D(draw, x + cs + shift, y, z, al, ah/1.4, ab, 'x-', clr,'drawReactions');
                        
                        % Write reaction value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mx));
                        else
                            value = sprintf('%.*f',dc,abs(mx));
                        end
                        text(x + cs + shift + al, y + t, z + t, value, 'Color', clr,'tag','textMomentReactions','UserData',abs(mx));
                    end
                    
                    if my >= 0
                        draw.moment3D(draw, x, y + cs + shift + al, z, al, ah/1.4, ab, 'y+', clr,'drawReactions');
                        
                        % Write reaction value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(my));
                        else
                            value = sprintf('%.*f',dc,abs(my));
                        end
                        text(x + t, y + cs + shift + al, z + t, value, 'Color', clr,'tag','textMomentReactions','UserData',abs(my));
                        
                    else
                        draw.moment3D(draw, x, y + cs + shift, z, al, ah/1.4, ab, 'y-', clr,'drawReactions');
                        
                        % Write reaction value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(my));
                        else
                            value = sprintf('%.*f',dc,abs(my));
                        end
                        text(x + t, y + cs + shift + al, z + t, value, 'Color', clr,'tag','textMomentReactions','UserData',abs(my));
                    end
                    
                    if mz >= 0
                        draw.moment3D(draw, x, y, z + cs + shift + al, al, ah/1.4, ab, 'z+', clr,'drawReactions');
                        
                        % Write reaction value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mz));
                        else
                            value = sprintf('%.*f',dc,abs(mz));
                        end
                        text(x + t, y + t, z + cs + shift + al, value, 'Color', clr,'tag','textMomentReactions','UserData',abs(mz));
                        
                    else
                        draw.moment3D(draw, x, y, z + cs + shift, al, ah/1.4, ab, 'z-', clr,'drawReactions');
                        
                        % Write reaction value
                        if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                            value = sprintf('%.*f kNm',dc,abs(mz));
                        else
                            value = sprintf('%.*f',dc,abs(mz));
                        end
                        text(x + t, y + t, z + cs + shift + al, value, 'Color', clr,'tag','textMomentReactions','UserData',abs(mz));
                    end
                end
                
                % Support (or spring) height in the direction of axis X
                if draw.drv.nodes(n).ebc(1)== FIXED_DOF
                    hx = ph;
                elseif draw.drv.nodes(n).ebc(1)== SPRING_DOF
                    hx = sh;
                end 
                
                % Check if translation is fixed in local axis X and draw
                % reaction indication
                if (draw.drv.nodes(n).ebc(1) == FIXED_DOF) || (draw.drv.nodes(n).ebc(1) == SPRING_DOF)
                    if rx >= 0
                        draw.arrow3D(draw, x - cs - shift - hx, y, z, al, ah, ab, 'x+', clr,'drawReactions');
                    else
                        draw.arrow3D(draw, x - cs - shift - hx - al, y, z, al, ah, ab, 'x-', clr,'drawReactions');
                    end
                    
                    % Write reaction value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%.*f kN',dc,abs(rx));
                    else
                        value = sprintf('%.*f',dc,abs(rx));
                    end
                    text(x - cs - shift - hx - 1.2*al, y, z, value, 'Color', clr,'tag','textForceReactions','UserData',abs(rx));
                end
                
                % Support (or spring) height in the direction of axis Y
                if draw.drv.nodes(n).ebc(2)== FIXED_DOF
                    hy = ph;
                elseif draw.drv.nodes(n).ebc(2)== SPRING_DOF
                    hy = sh;
                end
                
                % Check if translation is fixed in local axis Y and draw
                % reaction indication
                if (draw.drv.nodes(n).ebc(2) == FIXED_DOF) || (draw.drv.nodes(n).ebc(2) == SPRING_DOF)
                    if ry >= 0
                        draw.arrow3D(draw, x, y - cs - shift - hy, z, al, ah, ab, 'y+', clr,'drawReactions');
                    else
                        draw.arrow3D(draw, x, y - cs - shift - hy - al, z, al, ah, ab, 'y-', clr,'drawReactions');
                    end
                    
                    % Write reaction value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%.*f kN',dc,abs(ry));
                    else
                        value = sprintf('%.*f',dc,abs(ry));
                    end
                    text(x, y - cs - shift - hy - 1.2*al, z, value, 'Color', clr,'tag','textForceReactions','UserData',abs(ry));
                end
                
                % Support (or spring) height in the direction of axis Y
                if draw.drv.nodes(n).ebc(3)== FIXED_DOF
                    hz = ph;
                elseif draw.drv.nodes(n).ebc(3)== SPRING_DOF
                    hz = sh;
                end
                
                % Check if vertical translation in z is fixed and draw
                % reaction indication
                if (draw.drv.nodes(n).ebc(3) == FIXED_DOF) || (draw.drv.nodes(n).ebc(3) == SPRING_DOF)
                    if rz >= 0
                        draw.arrow3D(draw, x, y, z - cs - shift - hz, al, ah, ab, 'z+', clr,'drawReactions');
                    else
                        draw.arrow3D(draw, x, y, z - cs - shift - hz - al, al, ah, ab, 'z-', clr,'drawReactions');
                    end
                    
                    % Write reaction value
                    if strcmp(get(mdata.unitsButton,'Checked'),'on') == 1
                        value = sprintf('%.*f kN',dc,abs(rz));
                    else
                        value = sprintf('%.*f',dc,abs(rz));
                    end
                    text(x, y, z - cs - shift - hz - 1.2*al, value, 'Color', clr,'tag','textForceReactions','UserData',abs(rz));
                end
            end
        end
    end
end