%% Anm_Truss2D Class
%
% This is a sub-class, in the Object Oriented Programming (OOP) paradigm,
% of super-class <anm.html *Anm*> in the <main.html LESM> (Linear Elements
% Structure Model) program. This sub-class implements abstract methods,
% declared in super-class *Anm*, that deal with 2D truss analysis model
% of linear structure elements.
%
%% Authors
% Luiz Fernando Martha, Rafael Lopez Rangel and Pedro Cortez Lopes
%
%% Truss 2D Analysis
% A 2D truss model is a common form of analysis model, with the following
% assumptions:
%%%
% * Truss elements are bars connected at their ends only, and they are
%   connected by friction-less pins. Therefore, a truss element does not
%   present any secondary bending moment or torsion moment induced by
%   rotation continuity at joints.
% * A truss model is loaded only at nodes.
%   Any load action along an element, such as self weight, is statically
%   transferred as concentrated forces to the element end nodes.
% * Local bending of elements due to internal loads is neglected, when
%   compared to the effect of global load acting on the truss.
% * Therefore, there is only one type of internal force in a truss
%   element: axial internal force, which may be tension or compression.
% * A 2D truss model is considered to be laid in the XY-plane, with only
%   in-plane behavior, that is, there is no displacement transversal to
%   the truss plane.
% * Each node of a 2D truss model has two d.o.f.'s (degrees of
%   freedom): a horizontal displacement in X direction and a vertical
%   displacement in Y direction.
%
%% Class definition
classdef Anm_Truss2D < Anm
    %% Constructor method
    methods
        %------------------------------------------------------------------
        function anm = Anm_Truss2D()
            include_constants;
            anm = anm@Anm(TRUSS2D_ANALYSIS,2);
        end
    end
    
    %% Public methods
    % Implementation of the abstract methods declared in super-class <anm.html *Anm*>.
    methods
        %------------------------------------------------------------------
        % Assembles element d.o.f. (degree of freedom) rotation transformation
        % matrix from global system to local system.
        % Output:
        %  rot: rotation transformation matrix
        % Input arguments:
        %  elem: handle to an object of the Elem class
        function rot = gblToLocElemRotMtx(~,elem)
            % Get 2x2 basis rotation transformation matrix
            T = elem.T(1:2,1:2);
            
            % Assemble element d.o.f. rotation transformation matrix
            % rot = [ T 0
            %         0 T ]
            rot = blkdiag(T,T);
        end
        
        %------------------------------------------------------------------
        % Initializes global d.o.f (degree of freedom) numbering ID matrix
        % with ones and zeros, and counts total number of equations of free
        % d.o.f.'s, total number of equations of fixed d.o.f.'s and total
		% number of d.o.f's constrained by spring support.
        %  ID matrix initialization:
        %  if ID(k,n) =  0, d.o.f. k of node n is free.
        %  if ID(k,n) =  1, d.o.f. k of node n is constrained by support.
		%  if ID(k,n) =  2, d.o.f. k of node n is constrained by spring.
        %  if ID(k,n) = -1, d.o.f. k of node n is constrained by ficticious support.
        % Input arguments:
        %  drv: handle to an object of the Drv class
        function setupDOFNum(anm,drv)
            include_constants;
            
            % Dimension global d.o.f. numbering ID matrix
            drv.ID = zeros(anm.ndof,drv.nnp);
            
            % Initialize number of fixed d.o.f. and d.o.f. constrained by spring
            drv.neqfixed = 0;
            drv.neqspring = 0;
            
            % Count number of fixed d.o.f. and setup ID matrix
            for n = 1:drv.nnp
                
                % Check for fixed translation in global X direction
                if (drv.nodes(n).ebc(1) == FIXED_DOF)
                    drv.neqfixed = drv.neqfixed + 1;
                    drv.ID(1,n) = FIXED_DOF;
                elseif drv.nodes(n).ebc(1) == SPRING_DOF
                    drv.neqspring = drv.neqspring + 1;
                    drv.ID(1,n) = SPRING_DOF;
                end
                
                % Check for fixed translation in global Y direction
                if (drv.nodes(n).ebc(2) == FIXED_DOF)
                    drv.neqfixed = drv.neqfixed + 1;
                    drv.ID(2,n) = FIXED_DOF;
                elseif drv.nodes(n).ebc(2) == SPRING_DOF
                    drv.neqspring = drv.neqspring + 1;
                    drv.ID(2,n) = SPRING_DOF;
                end
            end
            
            % Compute total number of free d.o.f.
            drv.neqfree = drv.neq - drv.neqfixed - drv.neqspring;
        end
        
        %------------------------------------------------------------------
        % Adds prescribed displacements (known support settlement values)
        % to global displacement vector.
        % Avoids storing a prescribed displacement component in a position
        % of the global displacement vector that corresponds to a free d.o.f.,
		% or a d.o.f. constrained by spring.
        % Input arguments:
        %  drv: handle to an object of the Drv class
        function setupPrescDispl(~,drv)
            for n = 1:drv.nnp
                if ~isempty(drv.nodes(n).prescDispl)
                    
                    % Add prescribed displacement in global X direction
                    id = drv.ID(1,n);
                    if (id > (drv.neqfree + drv.neqspring)) && (drv.nodes(n).prescDispl(1) ~= 0)
                        drv.D(id) = drv.nodes(n).prescDispl(1);
                    end
                    
                    % Add prescribed displacement in global Y direction
                    id = drv.ID(2,n);
                    if (id > (drv.neqfree + drv.neqspring)) && (drv.nodes(n).prescDispl(2) ~= 0)
                        drv.D(id) = drv.nodes(n).prescDispl(2);
                    end
                    
                end
            end
        end
        
        %------------------------------------------------------------------
        % Adds spring stiffness coefficients to global stiffness matrix.
        % Avoids storing a spring stiffness component in a position of the
        % global stiffness matrix that corresponds to a free d.o.f.
        % Input arguments:
        %  drv: handle to an object of the Drv class
        function setupSpringStiff(~,drv)
            drv.gblSprStiff = zeros(1,drv.neqspring);
            for n = 1:drv.nnp
                if ~isempty(drv.nodes(n).springStiff)
                    
                    % Add displacement spring stiffness in global X direction
                    id = drv.ID(1,n);
                    if (id > drv.neqfree) && (id <= drv.neqfree + drv.neqspring) && (drv.nodes(n).springStiff(1) ~= 0)
                        drv.K(id,id) = drv.K(id,id) + drv.nodes(n).springStiff(1);
                        drv.gblSprStiff(id-drv.neqfree) = drv.nodes(n).springStiff(1);
                    end
                    
                   % Add displacement spring stiffness in global Y direction
                    id = drv.ID(2,n);
                    if (id > drv.neqfree) && (id <= drv.neqfree + drv.neqspring) && (drv.nodes(n).springStiff(2) ~= 0)
                        drv.K(id,id) = drv.K(id,id) + drv.nodes(n).springStiff(2);
                        drv.gblSprStiff(id-drv.neqfree) = drv.nodes(n).springStiff(2);
                    end
                end
            end
        end
        
        %------------------------------------------------------------------
        % Assembles element stiffness matrix in local system.
        % Output:
        %  kel: target element stiffness matrix in local system
        % Input arguments:
        %  elem: handle to an object of the Elem class
        function kel = elemLocStiffMtx(~,elem)
            % Compute axial stiffness coefficients
            kea = elem.axialStiffCoeff();
            
            % Assemble element stiffness matrix in local system
            kel = [ kea(1,1)  0     kea(1,2)  0;
                    0         0     0         0;
                    kea(2,1)  0     kea(2,2)  0;
                    0         0     0         0 ];
        end
        
        %------------------------------------------------------------------
        % Adds nodal load components to global forcing vector,
        % including the terms that correspond to constrained d.o.f.
        % Input arguments:
        %  drv: handle to an object of the Drv class
        function nodalLoads(~,drv)
            for n = 1:drv.nnp
                if ~isempty(drv.nodes(n).nodalLoad)
                    
                    % Add applied force in global X direction
                    id = drv.ID(1,n);
                    drv.F(id) = drv.F(id) + drv.nodes(n).nodalLoad(1);
                    
                    % Add applied force in global Y direction
                    id = drv.ID(2,n);
                    drv.F(id) = drv.F(id) + drv.nodes(n).nodalLoad(2);
                    
                end
            end
        end
        
        %------------------------------------------------------------------
        % Assembles element fixed end force (FEF) vector in local system
        % for an applied distributed load.
        % Output:
        %  fel: element fixed end force vector in local system
        % Input arguments:
        %  load: handle to an object of the Lelem class
        function fel = elemLocDistribLoadFEF(~,load)
            % Compute axial fixed end force components
            fea = load.axialDistribLoadFEF();
            
            % Compute flexural (transversal) fixed end force components
            fef = load.flexuralDistribLoadFEF_XY();
            
            % Assemble element fixed end force (FEF) vector in local system
            fel = [ fea(1);
                    fef(1);
                    fea(2);
                    fef(3) ];
        end
        
        %------------------------------------------------------------------
        % Assembles element fixed end force (FEF) vector in local system
        % for an applied thermal load (temperature variation).
        % Output:
        %  fel: element fixed end force vector in local system
        % Input arguments:
        %  load: handle to an object of the Lelem class
        function fel = elemLocThermalLoadFEF(~,load)
            % Compute axial fixed end force components
            fea = load.axialThermalLoadFEF();
            
            % Compute flexural (transversal) fixed end force components
            fef = load.flexuralThermalLoadFEF_XY();
            
            % Assemble element fixed end force (FEF) vector in local system
            fel = [ fea(1);
                    fef(1);
                    fea(2);
                    fef(3) ];
        end
        
        %------------------------------------------------------------------
        % Initializes element internal forces arrays with null values.
        %  axial_force(1,2)
        %   Ni = axial_force(1) - init value
        %   Nf = axial_force(2) - final value
        % Input arguments:
        %  elem: handle to an object of the Elem class
        function initIntForce(~,elem)
            elem.axial_force = zeros(1,2);
        end
        
        %------------------------------------------------------------------
        % Assembles contribution of a given internal force vector to
        % element arrays of internal forces.
        % Input arguments:
        %  elem: handle to an object of the Elem class
        %  fel: element internal force vector in local system
        function assembleIntForce(~,elem,fel)
            elem.axial_force(1) = elem.axial_force(1) + fel(1);
            elem.axial_force(2) = elem.axial_force(2) + fel(3);
        end
        
        %------------------------------------------------------------------
        % Initializes element internal displacements array with null values.
        % Each element is discretized in 50 cross-sections, where internal
        % displacements are computed.
        %  intDispl(1,:) -> du (axial displacement)
        %  intDispl(2,:) -> dv (transversal displacement in local y-axis)
        % Input arguments:
        %  elem: handle to an object of the Elem class
        function initIntDispl(~,elem)
            elem.intDispl = zeros(2,50);
        end
        
        %------------------------------------------------------------------
        % Assembles displacement shape function matrix evaluated at a 
        % given cross-section position.
        % Output:
        %  N: displacement shape function matrix
        % Input arguments:
        %  elem: handle to an object of the Elem class
        %  x: cross-section position on element local x-axis
        function N = displShapeFcnMtx(~,elem,x)
            % Get number of points
            np = size(x,2);
            
            % Compute axial displacement shape functions vector
            Nu = elem.axialDisplShapeFcnVector(x);
            
            % Compute transversal displacement shape functions vector
            Nv = elem.flexuralDisplShapeFcnVector_XY(x);
            
            % Initialize displacement shape function matrix
            N = zeros(2*np,4);
            
            % Assemble displacement shape function matrix
            N(2*(1:np)-1, [1 3]) = Nu;
            N(2*(1:np), [2 4]) = Nv(:,[1 3]);
        end
        
        %------------------------------------------------------------------
        % Computes internal displacements matrix from global analysis
        % Output:
        %  del: element internal displacements matrix
        % Input arguments:
        %  N: displacement shape function matrix
        %  dl: nodal displacements, on local coordinates
        function del = gblAnlIntDisplMtx(~,N,dl)
            % Get number of points
            np = size(N,1)/2;
            
            % Compute internal displacements matrix
            del_aux = N * dl;
            
            % Rearrange internal displacements matrix
            del = [ (del_aux(2*(1:np)-1))';
                    (del_aux(2*(1:np)))'  ];
        end
        
        %------------------------------------------------------------------
        % Computes element internal displacements vector in local system,
        % in a given cross-section position, for the local analysis from
        % element loads (distributed loads and thermal loads).
        % Output:
        %  del: a 2x1 vector of with element internal displacements:
        %      du -> axial displacement
        %      dv -> transversal displacement in local y-axis direction
        %      x  -> cross-section position on element local x-axis
        %  del = [ du(x);
        %          dv(x) ]
        function del = lclAnlIntDispl(~,~,x)
            % Get number of points
            np = size(x,2);
            
            % Internal displacements from local analysis are neglected in
            % truss models.
            del = zeros(2,np);
        end
        
        %------------------------------------------------------------------
        function stressValues = intStress(~,elem,x)
            % Get axial force values
            [N,~] = elem.intAxialForce(x);
            
            % Assemble stress values matrix
            stressValues = N;
        end
    end
end