%% Element Dialog Callback Functions
% This file contains the callback functions associated with the "Elements"
% dialog of the graphical version of the LESM program.
% Common input arguments for all callback functions:
%  hObject: handle to interface object related to the function
%  eventdata: reserved - to be defined in a future version of MATLAB
%  handles: structure with handles and user data
%
%% Authors
% Luiz Fernando Martha, Rafael Lopez Rangel and Pedro Cortez Lopes
%
%% ------------------------------------------------------------------------
% GUI initialization function.
function varargout = GUI_Elements(varargin)
gui_Singleton = 1;
gui_State = struct('gui_Name',       mfilename, ...
                   'gui_Singleton',  gui_Singleton, ...
                   'gui_OpeningFcn', @GUI_Elements_OpeningFcn, ...
                   'gui_OutputFcn',  @GUI_Elements_OutputFcn, ...
                   'gui_LayoutFcn',  [] , ...
                   'gui_Callback',   []);
if nargin && ischar(varargin{1})
    gui_State.gui_Callback = str2func(varargin{1});
end

if nargout
    [varargout{1:nargout}] = gui_mainfcn(gui_State, varargin{:});
else
    gui_mainfcn(gui_State, varargin{:});
end

%--------------------------------------------------------------------------
% Executes just before Elements GUI is made visible.
% Sets GUI initial properties.
function GUI_Elements_OpeningFcn(hObject, eventdata, handles, varargin) %#ok<*INUSL>
% Choose default command line output for GUI_Elements
handles.output = hObject;

% Move GUI to the center of the screen
if getappdata(0,'move') == 1
    movegui(hObject,'center')
end

% Make GUI a modal window
set(hObject,'WindowStyle','modal');

% Set table data
nel = getappdata(0,'nel');
mdata = guidata(findobj('Tag','GUI_Main'));
anm = get(mdata.popupmenu_Anm,'Value');
if nel > 0
    elems = getappdata(0,'elems');
    elem = cell(nel,11);
    for e = 1:nel
        id = e;
        etype = elems(e).type;
        if anm == 1 || anm == 4
            type = 'Truss';
        else
            if etype == 1
                type = 'Timoshenko';
            elseif etype == 0
                type = 'Navier';
            end
        end
        mat = elems(e).material.id;
        sec = elems(e).section.id;
        ni = elems(e).nodes(1).id;
        nf = elems(e).nodes(2).id;
        
        hi = elems(e).hingei;
        if hi == 1
            hi= 'No';
        else
            hi= 'Yes';
        end
        
        hf = elems(e).hingef;
        if hf == 1
            hf = 'No';
        else
            hf = 'Yes';
        end

        v = [elems(e).vz(1), elems(e).vz(2), elems(e).vz(3)];
        v = v / norm(v);
        
        elem(e,:) = {id,type,mat,sec,ni,nf,hi,hf,v(1),v(2),v(3)};
    end
    
    nmat = getappdata(0,'nmat');
    mats = cell(1,nmat);
    for nm = 1:nmat
        mats(nm) = {num2str(nm)};
    end
    
    nsec = getappdata(0,'nsec');
    secs = cell(1,nsec);
    for ns = 1:nsec
        secs(ns) = {num2str(ns)};
    end
    
    if anm == 1 % TRUSS 2D
        cEdit = [false(1,2) true(1,2) false(1,2) true(1,2) false(1,3)];
        cFormat = {[] [] mats secs [] [] [] [] [] [] []};
    elseif anm == 2 || anm == 3  % FRAME 2D or GRILLAGE
        cEdit = [false(1,1) true(1,3) false(1,2) true(1,2) false(1,3)];
        cFormat = {[] {'Navier' 'Timoshenko'} mats secs [] [] {'Yes' 'No'} {'Yes' 'No'} [] [] []};
    elseif anm == 4 % TRUSS 3D
        cEdit = [false(1,2) true(1,2) false(1,2) true(1,5)];
        cFormat = {[] [] mats secs [] [] [] [] [] [] []};
    else  % FRAME 3D
        cEdit = [false(1,1) true(1,3) false(1,2) true(1,5)];
        cFormat = {[] {'Navier' 'Timoshenko'} mats secs [] [] {'Yes' 'No'} {'Yes' 'No'} [] [] []};
    end
    set(handles.uitable_Elements,'Data',elem,...
        'CellEditCallback',@uitable_Elements_CellEditCallback,...
        'CellSelectionCallback',@uitable_Elements_CellSelectionCallback,...
        'ColumnEditable',cEdit,'ColumnFormat',cFormat);
else
    set(handles.uitable_Elements,'Data',{'','','','','','','','','','',''},...
        'CellEditCallback',@uitable_Elements_CellEditCallback,...
        'CellSelectionCallback',@uitable_Elements_CellSelectionCallback);
end

% Create list of materials
nmat = getappdata(0,'nmat');
m = zeros(1,nmat);
for i = 1:nmat
    m(i) = i;
end
m = num2str(m,'%d\n');
set(handles.popupmenu_Material,'string',m)

% Create list of cross-sections
nsec = getappdata(0,'nsec');
s = zeros(1,nsec);
for i = 1:nsec
    s(i) = i;
end
s = num2str(s,'%d\n');
set(handles.popupmenu_Section,'string',s)

% Create list of nodes
nnp = getappdata(0,'nnp');
e = zeros(1,nnp);
for i = 1:nnp
    e(i) = i;
end
e = num2str(e,'%d\n');
set(handles.popupmenu_Node1,'string',e)
set(handles.popupmenu_Node2,'string',e)

% Create list of elements to be deleted
set(handles.popupmenu_Delete,'Value',nel)
if nel ~= 0
    elm = 1:nel;
    elm = num2str(elm,'%d\n');
    set(handles.popupmenu_Delete,'Value',nel,'string',elm)
else
    set(handles.popupmenu_Delete,'Value',1,'string','No itens')
end

% If the analysis model is a 2D truss or a 3D truss, set default option to 
% hinged ends.
mdata = guidata(findobj('Tag','GUI_Main'));
anm = get(mdata.popupmenu_Anm,'Value');
if (anm == 1) || (anm == 4)
    set(handles.popupmenu_Hinge1,'Value',2,'Enable','off')
    set(handles.popupmenu_Hinge2,'Value',2,'Enable','off')
else    
    set(handles.popupmenu_Hinge1,'Value',1,'Enable','on')
    set(handles.popupmenu_Hinge2,'Value',1,'Enable','on')
end

% If the analysis model is 2D truss or 3D truss, disable shear and 
% flexural deformation checkbox. If the analysis model is grillage,
% disable axial deformation checkbox.
mdata = guidata(findobj('Tag','GUI_Main'));
anm = get(mdata.popupmenu_Anm,'Value');
if (anm == 1) || (anm == 4)
    set(handles.checkbox_AxialDeformation,'Value',1,'Enable','off')
    set(handles.checkbox_FlexuralDeformation,'Value',0,'Enable','off')
    set(handles.checkbox_ShearDeformation,'Value',0,'Enable','off')
    set(handles.text_Timoshenko,'Enable','off')
elseif anm == 3
    set(handles.checkbox_AxialDeformation,'Value',0,'Enable','off')
    set(handles.checkbox_ShearDeformation,'Value',0,'Enable','on')
    set(handles.text_Timoshenko,'Enable','on')
%    set(handles.checkbox_FlexuralDeformation,'Value',1,'Enable','on')
else  
   set(handles.checkbox_AxialDeformation,'Value',1,'Enable','off')
    set(handles.checkbox_ShearDeformation,'Value',0,'Enable','on')
    set(handles.text_Timoshenko,'Enable','on')
%    set(handles.checkbox_FlexuralDeformation,'Value',1,'Enable','on')
end


% If the analysis model is a 3D truss or a 3D frame, enable Vz setting option
mdata = guidata(findobj('Tag','GUI_Main'));
anm = get(mdata.popupmenu_Anm,'Value');
if (anm == 4) || (anm == 5)
    set(handles.radiobutton_SetZ,'Enable','on')
end

% Update handles structure
guidata(hObject, handles);

%--------------------------------------------------------------------------
% Outputs from this function are returned to the command line.
function varargout = GUI_Elements_OutputFcn(hObject, eventdata, handles)
% Get default command line output from handles structure
varargout{1} = handles.output;

%--------------------------------------------------------------------------
% Executes on button press in "Add" pushbutton.
% Adds an Elemt object with input properties to the list of elements
function pushbutton_Add_Callback(hObject, eventdata, handles) %#ok<*DEFNU>
% Check if end nodes are not the same of a previously created element
include_constants;
equal = 0;
nel = getappdata(0,'nel');
n1 = get(handles.popupmenu_Node1,'Value');
n2 = get(handles.popupmenu_Node2,'Value');
if nel ~= 0
    elems = getappdata(0,'elems');
    for i = 1:nel
        n1j = elems(i).nodes(1).id;
        n2j = elems(i).nodes(2).id;
        if ((n1 == n1j) && (n2 == n2j)) || ((n1 == n2j) && (n2 == n1j))
            equal = 1;
        end
        break
    end
end

if equal == 1
    msgbox('There is already an element with this nodal incidence', 'Error','error');
elseif n1 == n2
    msgbox('Initial and final nodes are the same', 'Error','error');
else
    mdata = guidata(findobj('Tag','GUI_Main'));
    drv = getappdata(0,'drv');
    nodes = getappdata(0,'nodes');
    
    % Get material and cross-section IDs
    matid = get(handles.popupmenu_Material,'Value');
    mat = drv.materials(matid);
    secid = get(handles.popupmenu_Section,'Value');
    sec = drv.sections(secid);
    
    % Get hinge information
    hingei = get(handles.popupmenu_Hinge1,'Value');
    if hingei == 1
        hi = 1;
    elseif hingei == 2
        hi = 0;
    end
    hingef = get(handles.popupmenu_Hinge2,'Value');
    if hingef == 1
        hf = 1;
    elseif hingef == 2
        hf = 0;
    end
    
    % Set initial and final nodes
    ni = nodes(n1);
    nf = nodes(n2);
    
    % Get Vz coordinates
    vzx = str2double(get(handles.edit_Vzx,'String'));
    vzy = str2double(get(handles.edit_Vzy,'String'));
    vzz = str2double(get(handles.edit_Vzz,'String'));
    vz = [vzx vzy vzz];
    
    % Verify if Vz is in the same direction of local axis X:
    % Get nodal coordinates
    xi = ni.coord(1);
    yi = ni.coord(2);
    zi = ni.coord(3);
    xf = nf.coord(1);
    yf = nf.coord(2);
    zf = nf.coord(3);
    % Calculate element local axis X orientation versor
    x = [xf-xi, yf-yi, zf-zi];
	z = [vzx, vzy, vzz];
    % Compare vectors 'x' and 'z'
    w = cross(x,z);
    if (abs(w(1)) < 1e-10) && (abs(w(2)) < 1e-10) && (abs(w(3)) < 1e-10)
        msgbox('Local axes X and Z are parallels. Please, change vector Z coordinates', 'Error','error');
        return
    end
    
    % Check if model is 2D
    anm = get(mdata.popupmenu_Anm,'Value');
    elemConnect = [];
    newNodes = [];
    if anm <= 3
        coords = [xi yi;
                  xf yf];
        
        crossNodesOutput = auxModelFctn('getCrossNodePoints',coords);
        crossNodePoints = crossNodesOutput{1};
        collinearElems = crossNodesOutput{2};
        elemConnect = crossNodesOutput{3};
        
        mouse = getappdata(0,'mouse');
        cnvs = mouse.getMouseProperty('Canvas');
        % Get canvas borders
        dfltUnits = get(cnvs,'units');
        set(cnvs,'units','normalized');
        limits = get(cnvs,'Position');
        set(cnvs,'units',dfltUnits);
        axisWidth = limits(3);
        tol = axisWidth/50;
        newNodes = auxModelFctn('getNewNodes',{crossNodePoints,collinearElems,elemConnect,tol});
    else
        tol = 10^(-10);
    end
    
    % Get elem type
    if get(handles.checkbox_ShearDeformation,'Value') == 1 % TIMOSHENKO
        type = 1;
    else % NAVIER
        type = 0;
    end
    
    % Create Elem objects and alocate them in elems vector
    if ~isempty(elemConnect)
        for e = 1:size(elemConnect,1)
            elem = Elem(type,drv.anm,mat,sec,[nodes(elemConnect(e,1)) nodes(elemConnect(e,2))],hi,hf,vz,[],[],[],[],[],[],[],[]);
            nel = nel + 1;
            elems(nel) = elem;
        end
        newElements = size(elemConnect,1);
    else
        elem = Elem(type,drv.anm,mat,sec,[ni nf],hi,hf,vz,[],[],[],[],[],[],[],[]);
        nel = nel + 1;
        elems(nel) = elem;
        newElements = 1;
    end
    
    % Update list of elements to be deleted
    if nel ~= 0
        elm = 1:nel;
        elm = num2str(elm,'%d\n');
        set(handles.popupmenu_Delete,'Value',nel,'string',elm,'Max',nel)
    else
        set(handles.popupmenu_Delete,'Value',1,'string','No itens','Max',1)
    end
    
    % Update elements uitable
    newElems = cell(newElements,11);
    for i = 1:newElements
        e = nel - (i-1);
        id = e;
        etype = elems(e).type;
        if anm == 1 || anm == 4
            type = 'Truss';
        else
            if etype == 1
                type = 'Timoshenko';
            elseif etype == 0
                type = 'Navier';
            end
        end
        mat = elems(e).material.id;
        sec = elems(e).section.id;
        ni = elems(e).nodes(1).id;
        nf = elems(e).nodes(2).id;
        
        hi = elems(e).hingei;
        if hi == 1
            hi= 'No';
        else
            hi= 'Yes';
        end
        
        hf = elems(e).hingef;
        if hf == 1
            hf = 'No';
        else
            hf = 'Yes';
        end
        
        v = [elems(e).vz(1), elems(e).vz(2), elems(e).vz(3)];
        v = v / norm(v);
        
        newElems(end-(i-1),:) = {id,type,mat,sec,ni,nf,hi,hf,v(1),v(2),v(3)};
    end

    if anm == 1 % TRUSS 2D
        cEdit = [false(1,2) true(1,2) false(1,2) true(1,2) false(1,3)];
    elseif anm == 2 || anm == 3  % FRAME 2D or GRILLAGE
        cEdit = [false(1,1) true(1,3) false(1,2) true(1,2) false(1,3)];
    elseif anm == 4 % TRUSS 3D
        cEdit = [false(1,2) true(1,2) false(1,2) true(1,5)];
    else  % FRAME 3D
        cEdit = [false(1,1) true(1,3) false(1,2) true(1,5)];
    end
    tableData = get(handles.uitable_Elements,'Data');
    if isempty(tableData)
        % Reset table format
        nmat = getappdata(0,'nmat');
        mats = cell(1,nmat);
        for nm = 1:nmat
            mats(nm) = {num2str(nm)};
        end
        
        nsec = getappdata(0,'nsec');
        secs = cell(1,nsec);
        for ns = 1:nsec
            secs(ns) = {num2str(ns)};
        end
        if anm == 1 || anm == 4
            cFormat = {[] [] mats secs [] [] [] [] [] [] []};
        else
            cFormat = {[] {'Navier' 'Timoshenko'} mats secs [] [] {'Yes' 'No'} {'Yes' 'No'} [] [] []};
        end
        set(handles.uitable_Elements,'Data',newElems,'ColumnEditable',cEdit,'ColumnFormat',cFormat)
    elseif strcmp(tableData{1},'')
        % Reset table format
        nmat = getappdata(0,'nmat');
        mats = cell(1,nmat);
        for nm = 1:nmat
            mats(nm) = {num2str(nm)};
        end
        
        nsec = getappdata(0,'nsec');
        secs = cell(1,nsec);
        for ns = 1:nsec
            secs(ns) = {num2str(ns)};
        end
        if anm == 1 || anm == 4
            cFormat = {[] [] mats secs [] [] [] [] [] [] []};
        else
            cFormat = {[] {'Navier' 'Timoshenko'} mats secs [] [] {'Yes' 'No'} {'Yes' 'No'} [] [] []};
        end
        set(handles.uitable_Elements,'Data',newElems,'ColumnEditable',cEdit,'ColumnFormat',cFormat)
    else
        set(handles.uitable_Elements,'Data',vertcat(tableData,newElems),'ColumnEditable',cEdit)
    end
    
    % Set Drv object properties
    drv.elems = elems;
    drv.nel = nel;
    
    % Enable "Process Data" button in main GUI
    set(mdata.pushbutton_ProcessData,'Enable','on');
    
    % Disable result buttons
    if get(mdata.popupmenu_Results,'value') ~= 1
        loadsNeedToBeRedrawn = true;
    else
        loadsNeedToBeRedrawn = false;
    end
    set(mdata.popupmenu_Results,'Enable','off','value',1);
    set(mdata.pushbutton_Textual,'Enable','off');
    set(mdata.checkbox_Reactions,'Enable','off', 'Value', 0);
    set(mdata.popupmenu_ElementResults,'Enable','off','value',1);
    
    % Update information panel in GUI_Main
    infoPanelData = get(mdata.uitable_infoPanel,'Data');
    infoPanelData(4,:) = {'Elements',nel};
    set(mdata.uitable_infoPanel,'Data',infoPanelData)
    
    % Update mouse property
    mouse = getappdata(0,'mouse');
    if ~isempty(mouse.originalData)
        mouse.originalData = infoPanelData;
    end
    setappdata(0,'mouse',mouse)
    
    % Return variables to root
    setappdata(0,'resultType',0);
    setappdata(0,'elems',elems);
    setappdata(0,'nel',nel);
    setappdata(0,'drv',drv);
    setappdata(0,'nodes',nodes);
    
    % Check if there are crossing points
    if ~isempty(newNodes)
        for nn = 1:size(newNodes,1)
            % Get crossing point coordinates
            x = newNodes(nn,1);
            y = newNodes(nn,2);
            z = 0;

            % Get id of intersected elements
            whichElems = auxModelFctn('isPointInElem',{[x y z],tol});

            % Update vector of handles to intersections (does not
            % create new nodes)
            if ~isempty(whichElems)
                intersections = getappdata(0,'intersections');
                existingIntSect = 0;
                for nis = 1:size(intersections,2)
                   if norm(intersections(nis).coord - [x y z]) <= tol
                       existingIntSect = nis;
                       break
                   end
                end
                if existingIntSect == 0
                    newIntSect.coord = [x y z];
                    newIntSect.elems = whichElems;
                    if ~isempty(intersections)
                        intSects = horzcat(intersections,newIntSect);
                    else
                        intSects = newIntSect;
                    end
                else
                    intersections(existingIntSect).elems = whichElems;
                end
                setappdata(0,'intersections',intSects);
            end
        end
    end
    
    % Enable/disable solve intersections pushbutton (toolbar)
    if size(getappdata(0,'intersections'),2) >= 1
        set(mdata.pushbutton_SolveIntSects,'enable','on')
    else
        set(mdata.pushbutton_SolveIntSects,'enable','off')
    end
    
    % Make GUI a normal window
    gui = findobj('Tag','GUI_Elements');
    set(gui,'WindowStyle','normal');
    
    % Draw updated model
    redraw(mdata,'Elements')
    if loadsNeedToBeRedrawn == true
        redraw(mdata,'Loads')
    end
    
    % Make GUI a modal window
    set(gui,'WindowStyle','modal');
end

%--------------------------------------------------------------------------
% Executes on button press in "Delete" pushbutton.
% Deletes an Elem object from the list of elements.
function pushbutton_Delete_Callback(hObject, eventdata, handles)
mdata = guidata(findobj('Tag','GUI_Main'));
drv = getappdata(0,'drv');
draw = getappdata(0,'draw');
elems = getappdata(0,'elems');
nel = getappdata(0,'nel');

if ~strcmp(get(handles.popupmenu_Delete,'String'),'No itens')
    % Get ID of deleted element
    del_elem = get(handles.popupmenu_Delete,'Value');
    
    % Check if deleted element had distributed or thermal loads
    if ~isempty(elems(del_elem).load.uniformGbl) || ~isempty(elems(del_elem).load.uniformLcl) ||...
       ~isempty(elems(del_elem).load.linearGbl) || ~isempty(elems(del_elem).load.linearLcl) ||...
       elems(del_elem).load.tempVar_X ~= 0 || elems(del_elem).load.tempVar_Y ~= 0 ||...
       elems(del_elem).load.tempVar_Z ~= 0
        loadsNeedToBeRedrawn = true;
    else
        loadsNeedToBeRedrawn = false;
    end
    
    % Remove deleted element from vector of elements
    elems(del_elem) = [];
    
    % Update number of elements
    nel = nel - 1;
    
    % Update list of elements to be deleted
    if nel ~= 0
        elm = 1:nel;
        elm = num2str(elm,'%d\n');
        set(handles.popupmenu_Delete,'value',nel,'string',elm,'Max',nel)
    else
        set(handles.popupmenu_Delete,'Value',1,'string','No itens','Max',1)
    end
    
    % Set updated elements uitable data
    tableData = get(handles.uitable_Elements,'Data');
    if nel ~= 0
        tableData(del_elem,:) = [];
        for ne = 1:nel
            tableData(ne,1) = {ne};
        end
        set(handles.uitable_Elements,'Data',tableData)
    else
        set(handles.uitable_Elements,'Data',{'','','','','','','','','','',''})
    end
    
    % Set Drv object properties
    drv.elems = elems;
    drv.nel = nel;
    
    % Check if element had any unresolved intersections
    intersections = getappdata(0,'intersections');
    if ~isempty(intersections)
        delIntSec = zeros(1,size(intersections,2));
        for nis = 1:size(intersections,2)
            if ~all(intersections(nis).elems ~= del_elem)
                intersections(nis).elems = nonzeros(intersections(nis).elems - del_elem)' + del_elem;
                if size(intersections(nis).elems,2) <= 1
                    delIntSec(nis) = nis;
                end
            end
            for nis_e = 1:size(intersections(nis).elems,2)
                if intersections(nis).elems(nis_e) > del_elem
                    intersections(nis).elems(nis_e) = intersections(nis).elems(nis_e) - 1;
                end
            end
        end
        if ~all(delIntSec == 0)
            intersections(nonzeros(delIntSec)') = [];
        end
        setappdata(0,'intersections',intersections)
    end
    
    % Enable/disable solve intersections pushbutton (toolbar)
    if size(getappdata(0,'intersections'),2) >= 1
        set(mdata.pushbutton_SolveIntSects,'enable','on')
    else
        set(mdata.pushbutton_SolveIntSects,'enable','off')
    end
    
    % Enable "Process Data" button in main GUI
    set(mdata.pushbutton_ProcessData,'Enable','on');
    
    % Disable result buttons
    if get(mdata.popupmenu_Results,'value') ~= 1
        allLoadsNeedToBeRedrawn = true;
    else
        allLoadsNeedToBeRedrawn = false;
    end
    set(mdata.popupmenu_Results,'Enable','off','value',1);
    set(mdata.pushbutton_Textual,'Enable','off');
    set(mdata.checkbox_Reactions,'Enable','off', 'Value', 0);
    set(mdata.popupmenu_ElementResults,'Enable','off','value',1);
    set(mdata.edit_Scale,'enable','off','visible','off');
    
    % Update information panel in GUI_Main
    infoPanelData = get(mdata.uitable_infoPanel,'Data');
    infoPanelData(4,:) = {'Elements',nel};
    set(mdata.uitable_infoPanel,'Data',infoPanelData)
    
    % Update mouse property
    mouse = getappdata(0,'mouse');
    if ~isempty(mouse.originalData)
        mouse.originalData = infoPanelData;
    end
    setappdata(0,'mouse',mouse)
    
    % Return variables to root
    setappdata(0,'resultType',0);
    setappdata(0,'elems',elems);
    setappdata(0,'nel',nel);
    setappdata(0,'drv',drv);
    draw.drv = drv;
    setappdata(0,'draw',draw);
    
    % Make GUI a normal window
    gui = findobj('Tag','GUI_Elements');
    set(gui,'WindowStyle','normal');
    
    % Draw updated model
    redraw(mdata,'Elements')
    if loadsNeedToBeRedrawn == true && allLoadsNeedToBeRedrawn == false
        redraw(mdata,'Element Loads')
    elseif allLoadsNeedToBeRedrawn == true
        redraw(mdata,'Loads')
    end
    
    % Make GUI a modal window
    set(gui,'WindowStyle','modal');
end

%--------------------------------------------------------------------------
% Executes on button press in "Cancel" pushbutton.
% Return to main GUI without creating any Element object.
function pushbutton_Cancel_Callback(hObject, eventdata, handles) %#ok<*INUSD>
delete(gcf)

%--------------------------------------------------------------------------
% Executes during popupmenu_Node1 creation, after setting all properties.
function popupmenu_Node1_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end

% Executes on selection change in popupmenu_Node1.
function popupmenu_Node1_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes during popupmenu_Node2 creation, after setting all properties.
function popupmenu_Node2_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end

% Executes on selection change in popupmenu_Node2.
function popupmenu_Node2_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes during popupmenu_Hinge1 creation, after setting all properties.
function popupmenu_Hinge1_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end

% Executes on selection change in popupmenu_Hinge1.
function popupmenu_Hinge1_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes during popupmenu_Hinge2 creation, after setting all properties.
function popupmenu_Hinge2_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end

% Executes on selection change in popupmenu_Hinge2.
function popupmenu_Hinge2_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes during popupmenu_Material creation, after setting all properties.
function popupmenu_Material_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end

% Executes on selection change in popupmenu_Material.
function popupmenu_Material_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes during popupmenu_Section creation, after setting all properties.
function popupmenu_Section_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end

% Executes on selection change in popupmenu_Section.
function popupmenu_Section_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes during popupmenu_Delete creation, after setting all properties.
function popupmenu_Delete_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end

% Executes on selection change in popupmenu_Delete.
function popupmenu_Delete_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes during edit_Vzx creation, after setting all properties.
function edit_Vzx_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end

function edit_Vzx_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes during edit_Vzy creation, after setting all properties.
function edit_Vzy_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end

function edit_Vzy_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes during edit_Vzz creation, after setting all properties.
function edit_Vzz_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end

function edit_Vzz_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes on button press in radiobutton_DefaultZ.
function radiobutton_DefaultZ_Callback(hObject, eventdata, handles)
set(handles.edit_Vzx,'Enable','off','String','0')
set(handles.edit_Vzy,'Enable','off','String','0')
set(handles.edit_Vzz,'Enable','off','String','1')

%--------------------------------------------------------------------------
% Executes on button press in radiobutton_SetZ.
function radiobutton_SetZ_Callback(hObject, eventdata, handles)
set(handles.edit_Vzx,'Enable','on')
set(handles.edit_Vzy,'Enable','on')
set(handles.edit_Vzz,'Enable','on')

%--------------------------------------------------------------------------
% Executes on button press in checkbox_ShearDeformation.
function checkbox_ShearDeformation_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes on button press in checkbox_FlexuralDeformation.
function checkbox_FlexuralDeformation_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes on button press in checkbox_AxialDeformation.
function checkbox_AxialDeformation_Callback(hObject, eventdata, handles)

%--------------------------------------------------------------------------
% Executes when cell is edited in uitable_Elements
function uitable_Elements_CellEditCallback(hObject, eventdata, handles)
mdata = guidata(findobj('tag','GUI_Elements'));
if getappdata(0,'nel') > 0
    set(mdata.pushbutton_Apply,'Enable','on')
else
    set(mdata.pushbutton_Apply,'Enable','off')
end

%--------------------------------------------------------------------------
% Executes when cell is selected in uitable_Elements
function uitable_Elements_CellSelectionCallback(hObject, eventdata, handles)
if isempty(eventdata.Indices)
    return
elseif size(eventdata.Indices,1) > 1
    id = [eventdata.Indices(end,1),1];
else
    id = eventdata.Indices;
end
mdata = guidata(findobj('tag','GUI_Elements'));
drv = getappdata(0,'drv');
anm = drv.anm.analysis_type;
include_constants
if getappdata(0,'nel') > 0 && ...
   ((id(2) <= 4 && anm ~= TRUSS2D_ANALYSIS && anm ~= TRUSS3D_ANALYSIS) ||...
   ((id(2) == 1 || id(2) == 3 || id(2) == 4) &&...
   (anm == TRUSS2D_ANALYSIS || anm == TRUSS3D_ANALYSIS)))
    set(mdata.pushbutton_ApplyMultiElem,'Enable','on')
else
    set(mdata.pushbutton_ApplyMultiElem,'Enable','off')
end
set(mdata.uitable_Elements,'UserData',id)

%--------------------------------------------------------------------------
% Executes on button press in pushbutton_Apply.
function pushbutton_Apply_Callback(hObject, eventdata, handles)
% Get handles and intialize new hinge flag and elemOrientFlag
nel = getappdata(0,'nel');
drv = getappdata(0,'drv');
elems = getappdata(0,'elems');
tableData = get(handles.uitable_Elements,'Data');
newHingeFlag = 0;
elemOrientFlag = 0;

% Get new info and set it to element objects
for e = 1:nel
   % Type
   newType = char(tableData(e,2));
   switch newType
       case 'Timoshenko'
           elems(e).type = 1;
       case 'Navier'
           elems(e).type = 0;
   end
   
   % Material
   newMat = cell2mat(tableData(e,3));
   elems(e).material = drv.materials(newMat);
   tableData(e,3) = num2cell(newMat);
   
   % Section
   newSec = cell2mat(tableData(e,4));
   elems(e).section = drv.sections(newSec);
   tableData(e,4) = num2cell(newSec);
   
   % Hinge_i
   newHingei = char(tableData(e,7));
   hingei = elems(e).hingei;
   switch newHingei
       case 'Yes'
           elems(e).hingei = 0;
       case 'No'
           elems(e).hingei = 1;
   end
   
   % Hinge_f
   newHingef = char(tableData(e,8));
   hingef = elems(e).hingef;
   switch newHingef
       case 'Yes'
           elems(e).hingef = 0;
       case 'No'
           elems(e).hingef = 1;
   end
   
   % Check if hinges were added or removed
   if elems(e).hingei ~= hingei || elems(e).hingef ~= hingef
       newHingeFlag = 1;
   end
   
   % vz_x
   newVz_x = cell2mat(tableData(e,9));
   
   % vz_y
   newVz_y = cell2mat(tableData(e,10));
   
   % vz_z
   newVz_z = cell2mat(tableData(e,11));
   
   newVz = [newVz_x, newVz_y, newVz_z];
    
   % Verify if Vz is in the same direction of local axis X:
   % Get nodal coordinates
   ni = elems(e).nodes(1);
   nf = elems(e).nodes(2);
   xi = ni.coord(1);
   yi = ni.coord(2);
   zi = ni.coord(3);
   xf = nf.coord(1);
   yf = nf.coord(2);
   zf = nf.coord(3);
   % Calculate element local axis X orientation versor
   x = [xf-xi, yf-yi, zf-zi];
   % Compare vectors 'x' and 'newVz'
   w = cross(x,newVz);
   if (abs(w(1)) > 1e-10) || (abs(w(2)) > 1e-10) || (abs(w(3)) > 1e-10)
       mdata = guidata(findobj('tag','GUI_Main'));
       if newVz(1) ~= elems(e).vz(1) || newVz(2) ~= elems(e).vz(2) ||...
          newVz(3) ~= elems(e).vz(3)
          if strcmp(get(mdata.orientationButton,'Checked'),'on')
            elemOrientFlag = 1;
          end
       end
       elems(e).vz = newVz;
   end
   tableData(e,9) = num2cell(elems(e).vz(1));
   tableData(e,10) = num2cell(elems(e).vz(2));
   tableData(e,11) = num2cell(elems(e).vz(3));
end

% Set new info in drv and save them in root
drv.elems = elems;
setappdata(0,'elems',elems);
setappdata(0,'drv',drv);

% Reset table data
set(handles.uitable_Elements,'Data',tableData)

% Disable button
set(hObject,'Enable','off')

% Disable results and enable process data pushbutton
mdata = guidata(findobj('tag','GUI_Main'));
resType = 0;
if strcmp(get(mdata.popupmenu_Results,'Enable'),'on')
    resType = get(mdata.popupmenu_Results,'value');
    % Disable result buttons
    set(mdata.popupmenu_Results,'Enable','off','value',1);
    set(mdata.pushbutton_Textual,'Enable','off');
    set(mdata.checkbox_Reactions,'Enable','off', 'Value', 0);
    set(mdata.popupmenu_ElementResults,'Enable','off','value',1);
    set(mdata.edit_Scale,'enable','off','visible','off');
    % Enable process data
    set(mdata.pushbutton_ProcessData,'Enable','on');
    % Redraw
    if resType ~= 1
        % Make GUI a normal window
        gui = findobj('Tag','GUI_Elements');
        set(gui,'WindowStyle','normal');
        
        redraw(mdata,'Elements');
        redraw(mdata,'Loads');
        
        % Make GUI a modal window
        set(gui,'WindowStyle','modal');
    end
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

% If any hinge was added or removed, or element orientaion changed,
% redraw model.
if (newHingeFlag ~= 0 || elemOrientFlag ~= 0) && (resType == 0 || resType == 1)
    % Make GUI a normal window
    gui = findobj('Tag','GUI_Elements');
    set(gui,'WindowStyle','normal');
    
    redraw(mdata,'Elements');
    
    % Make GUI a modal window
    set(gui,'WindowStyle','modal');
end

%--------------------------------------------------------------------------
% Executes on button press in pushbutton_ApplyMultiElem.
function pushbutton_ApplyMultiElem_Callback(hObject, eventdata, handles)
% Get which cell was selected by user
tableIndex = get(handles.uitable_Elements,'UserData');

% Determine what should be asked of user, based on selected cell
switch tableIndex(2)
    case 1  % ELEMENT ID (TYPE, MATERIAL AND SECTION)
        inputStr = sprintf('Apply properties of element %i (type, material and cross-section) to multiple elements:\nExample: 1;3-5;7',...
                          tableIndex(1));
        sz = [1 90];
    case 2  % ELEMENT TYPE
        inputStr = sprintf('Apply type of element %i to multiple elements:\nExample: 1;3-5;7',...
                          tableIndex(1));
        sz = [1 60];
    case 3  % ELEMENT MATERIAL
        inputStr = sprintf('Apply material of element %i to multiple elements:\nExample: 1;3-5;7',...
                          tableIndex(1));
        sz = [1 60];
    case 4  % ELEMENT SECTION
        inputStr = sprintf('Apply cross-section of element %i to multiple elements:\nExample: 1;3-5;7',...
                          tableIndex(1));
        sz = [1 60];
end

% Get string input to determine the elements to have properties modified
e_str = char(inputdlg(inputStr,'Apply properties to multiple elements',sz));

% Read string entered by user
if isempty(e_str)
    return
elseif strcmp(e_str,'all')
    e_ID = 1:getappdata(0,'nel');
elseif strcmp(e_str,'ALL')
    e_ID = 1:getappdata(0,'nel');
elseif strcmp(e_str,'All')
    e_ID = 1:getappdata(0,'nel');
else
    [flag,e_ID] = readStr(e_str,getappdata(0,'nel'));
    if ~flag
        msgbox('Invalid input data', 'Error','error');
        return
    end
end

% Get handles to the drv object and elem objects
drv = getappdata(0,'drv');
elems = getappdata(0,'elems');

% Get uitable_Elements data
tableData = get(handles.uitable_Elements,'Data');

% Assign information to elements, based on selected cell
switch tableIndex(2)
    case 1  % ELEMENT ID (TYPE, MATERIAL AND SECTION)
        switch tableData{tableIndex(1),2}
            case 'Timoshenko'
                type = 1;
            case 'Navier'
                type = 0;
            case 'Truss'
                type = 0;
        end
        material = drv.materials(tableData{tableIndex(1),3});
        section = drv.sections(tableData{tableIndex(1),4});
        for e = e_ID
            elems(e).type = type;
            elems(e).material = material;
            elems(e).section = section;
            tableData(e,2) = tableData(tableIndex(1),2);
            tableData(e,3) = tableData(tableIndex(1),3);
            tableData(e,4) = tableData(tableIndex(1),4);
        end
    case 2  % ELEMENT TYPE
        switch tableData{tableIndex(1),2}
            case 'Timoshenko'
                type = 1;
            case 'Navier'
                type = 0;
            case 'Truss'
                type = 0;
        end
        for e = e_ID
            elems(e).type = type;
            tableData(e,2) = tableData(tableIndex(1),2);
        end
    case 3  % ELEMENT MATERIAL
        material = drv.materials(tableData{tableIndex(1),3});
        for e = e_ID
            elems(e).material = material;
            tableData(e,3) = tableData(tableIndex(1),3);
        end
    case 4  % ELEMENT SECTION
        section = drv.sections(tableData{tableIndex(1),4});
        for e = e_ID
            elems(e).section = section;
            tableData(e,4) = tableData(tableIndex(1),4);
        end
end

% Update drv object
drv.elems = elems;

% Return drv and elems to root
setappdata(0,'elems',elems);
setappdata(0,'drv',drv);

% Reset graphic components
set(handles.uitable_Elements,'Data',tableData)
set(hObject,'Enable','off')
set(handles.pushbutton_Apply,'Enable','off')

% Disable results and enable process data pushbutton
mdata = guidata(findobj('tag','GUI_Main'));
if strcmp(get(mdata.popupmenu_Results,'Enable'),'on')
    resType = get(mdata.popupmenu_Results,'value');
    % Disable result buttons
    set(mdata.popupmenu_Results,'Enable','off','value',1);
    set(mdata.pushbutton_Textual,'Enable','off');
    set(mdata.checkbox_Reactions,'Enable','off', 'Value', 0);
    set(mdata.popupmenu_ElementResults,'Enable','off','value',1);
    set(mdata.edit_Scale,'enable','off','visible','off');
    % Enable process data
    set(mdata.pushbutton_ProcessData,'Enable','on');
    % Redraw
    if resType ~= 1
        % Make GUI a normal window
        gui = findobj('Tag','GUI_Elements');
        set(gui,'WindowStyle','normal');
    
        redraw(mdata,'Elements');
        redraw(mdata,'Loads');
        
        % Make GUI a modal window
        set(gui,'WindowStyle','modal');
    end
    delete(findobj('tag','drawReactions'));
    delete(findobj('tag','textForceReactions'));
    delete(findobj('tag','textMomentReactions'));
end

%--------------------------------------------------------------------------
% Auxiliary function
% Reads string to get numbers or vectors
% Input:
% * str -> string to be read
% * max -> maximum value to be read (pre-dimensions output)
% Output:
% * flag -> flag for erros (0 = error; 1 = success)
% * output -> vector of integer numbers read from string
function [flag,output] = readStr(str,max)
output = zeros(1,max);
count = 0;           % counter for output index
numFlag = false;     % flag for number being read
vctrFlag = false;    % flag for vector being read
errorFlag = false;   % flag for errors on string input

for aux = 1:size(str,2)
    if strcmp(str(aux),' ')
        numFlag = false;
    elseif ~numFlag
        if ~isnan(str2double(str(aux)))
            numFlag = true;
            count = count + 1;
            output(count) = str2double(str(aux));
            if vctrFlag && aux == size(str,2)
                vctr = linspace(output(count-1),output(count),abs(output(count)-output(count-1))+1);
                if ~all(vctr <= max)
                    errorFlag = true;
                    break
                end
                output(count-1:count-2+size(vctr,2)) = vctr;
            end
        else
            errorFlag = true;
            break
        end
    elseif numFlag
        if ~isnan(str2double(str(aux)))
            numFlag = true;
            output(count) = output(count)*10 + str2double(str(aux));
            if vctrFlag && aux == size(str,2)
                vctr = linspace(output(count-1),output(count),abs(output(count)-output(count-1))+1);
                if ~all(vctr <= max)
                    errorFlag = true;
                    break
                end
                output(count-1:count-2+size(vctr,2)) = vctr;
            end
        elseif strcmp(str(aux),';')
            numFlag = false;
            if vctrFlag
                vctr = linspace(output(count-1),output(count),abs(output(count)-output(count-1))+1);
                if ~all(vctr <= max)
                    errorFlag = true;
                    break
                end
                output(count-1:count-2+size(vctr,2)) = vctr;
                count = count-1+size(vctr,2);
                vctrFlag = false;
            end
        elseif strcmp(str(aux),'-')
            if vctrFlag || aux == size(str,2)
                errorFlag = true;
                break
            end
            numFlag = false;
            vctrFlag = true;
        else
            errorFlag = true;
            break
        end
    end
end
output = nonzeros(output)';

if errorFlag || ~all(output <= max) || isempty(output)
    flag = false;
    output = [];
else
    flag = true;
end
