package opendreams.proxy;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.logic.RemoteFileChannel;
import tecgraf.ftc.common.logic.RemoteFileChannelImpl;
import tecgraf.ftc.utils.Utils;
import tecgraf.openbus.data_service.DataDescription;
import tecgraf.openbus.data_service.DataKey;
import tecgraf.openbus.data_service.IHierarchicalDataService;
import tecgraf.openbus.data_service.Metadata;
import tecgraf.openbus.data_service.UnstructuredData;
import tecgraf.openbus.data_service.UnstructuredDataHelper;
import tecgraf.openbus.data_service.project.ProjectItemDescriptionImpl;
import tecgraf.openbus.project.ProjectItemDescription;
import tecgraf.openbus.project.ProjectItemDescriptionHelper;

/**
 * Um projeto usado para escrever e ler arquivos usados na submisso de comandos
 * pelo OpenDreams.
 * 
 * @author Tecgraf PUC-Rio
 */
public class Project {
  private IHierarchicalDataService dataService;
  private DataDescription projectDesc;
  private String owner;
  private byte[] currentDir;

  /**
   * Constri uma representao do projeto.
   * 
   * @param projectDesc descritor do projeto
   * @param owner nome do usurio
   * @param dataService servio de projeto
   */
  Project(DataDescription projectDesc, String owner,
    IHierarchicalDataService dataService) {
    this.projectDesc = projectDesc;
    this.dataService = dataService;
    this.owner = owner;
    this.currentDir = projectDesc.fKey;
  }

  /**
   * Altera o diretrio corrente para um outro que seja filho do diretrio
   * corrente.
   * 
   * @param dirName nome do diretrio filho do diretrio corrente
   * @return {@code true}, se o diretrio corrente foi alterado ou {@code false}
   *         , caso contrrio.
   * @throws OpenDreamsException se ocorrer algum erro durante a mudana do
   *         diretrio corrente.
   * 
   */
  public boolean changeDirectory(String dirName) throws OpenDreamsException {
    return changeDirectory(dirName, false);
  }

  /**
   * Altera o diretrio corrente para um outro que seja filho do diretrio
   * corrente. Possibilita que o diretrio filho seja criado, se no existir.
   * 
   * @param dirName nome do diretrio filho
   * @param create se {@code true}, cria o diretrio se no existir
   * @return {@code true}, se o diretrio corrente foi alterado ou {@code false}
   *         , caso contrrio.
   * @throws OpenDreamsException se ocorrer algum erro durante a mudana do
   *         diretrio corrente.
   */
  public boolean changeDirectory(String dirName, boolean create)
    throws OpenDreamsException {
    byte[] fkey = this.find(dirName);
    if (fkey != null) {
      this.currentDir = fkey;
      return true;
    }
    if (create) {
      return createDirectory(dirName, true);
    }
    return false;
  }

  /**
   * Altera o diretrio corrente para o diretrio pai.
   * 
   * @return {@code true}, se o diretrio corrente foi alterado ou {@code false}
   *         , caso contrrio.
   * @throws OpenDreamsException se o diretrio corrente j estiver no prprio
   *         diretrio do projeto ou se ocorrer algum erro na navegao para o
   *         diretrio pai.
   */
  public boolean changeDirectoryUp() throws OpenDreamsException {
    try {
      if (this.currentDir.equals(projectDesc.fKey)) {
        throw new OpenDreamsException(
          "O diretrio corrente  o prprio diretrio do projeto");
      }
      DataDescription dataDescription = dataService.getParent(this.currentDir);
      if (dataDescription == null) {
        throw new OpenDreamsException("O diretrio pai retornou null");
      }
      this.currentDir = dataDescription.fKey;
    }
    catch (Exception e) {
      throw new OpenDreamsException("Erro na navegao para o diretrio pai", e);
    }
    return true;
  }

  /**
   * Cria um diretrio filho do diretrio corrente. Possibilita que o novo
   * sub-diretrio passe a ser o diretrio corrente.
   * 
   * @param dirName nome do diretrio
   * @param change se {@code true}, faz com o que o novo diretrio seja o
   *        corrente
   * @return {@code true}, se o diretrio foi criado e {@code false}, caso
   *         contrrio.
   * @throws OpenDreamsException se ocorrer algum erro durante a criao do novo
   *         diretrio.
   */
  public boolean createDirectory(String dirName, boolean change)
    throws OpenDreamsException {
    Set<String> views = new HashSet<String>();
    views.add(ProjectItemDescriptionHelper.id());
    long currentDate = Calendar.getInstance().getTimeInMillis();
    ProjectItemDescription prototype =
      new ProjectItemDescriptionImpl(dirName, views, new ArrayList<Metadata>(),
        owner, null, null, null, 0, true, true, true, currentDate, currentDate);
    try {
      byte[] key = dataService.createData(currentDir, prototype);
      if (change) {
        this.currentDir = key;
      }
      return true;
    }
    catch (Exception e) {
      throw new OpenDreamsException("Erro na criao do diretrio " + dirName
        + " no projeto " + projectDesc.fName, e);
    }
  }

  /**
   * Cria um diretrio filho do diretrio corrente.
   * 
   * @param dirName nome do diretrio
   * @return {@code true}, se o diretrio foi criado e {@code false}, caso
   *         contrrio.
   * @throws OpenDreamsException se ocorrer algum erro durante a criao do novo
   *         diretrio.
   */
  public boolean createDirectory(String dirName) throws OpenDreamsException {
    return createDirectory(dirName, false);
  }

  /**
   * Obtm os dados de um arquivo que est no diretrio corrente da rea do
   * projeto.
   * 
   * @param fileName nome do arquivo
   * @return o array com os bytes lidos
   * @throws OpenDreamsException se ocorrer um erro na recuperao dos dados do
   *         arquivo.
   */
  public byte[] getDataFrom(String fileName) throws OpenDreamsException {
    byte[] fileKey = this.find(fileName);
    if (fileKey == null) {
      throw new OpenDreamsException("Arquivo " + fileName + " no encontrado");
    }
    RemoteFileChannel rfc = null;
    try {
      UnstructuredData view =
        (UnstructuredData) dataService.getDataView(fileKey,
          UnstructuredDataHelper.id());
      DataKey dataKey = new DataKey(view.fKey);
      rfc =
        new RemoteFileChannelImpl(dataKey.getDataId().getBytes(
          Utils.CHARSET_ENCODING), view.fWritable, view.fHost, view.fPort,
          view.fAccessKey);
      rfc.open(true);
      int fileSize = (int) rfc.getSize();
      byte[] buffer = new byte[fileSize];
      if (fileSize != 0) {
        rfc.read(buffer);
      }
      return buffer;
    }
    catch (Exception e) {
      throw new OpenDreamsException("Erro na leitura de um dado no projeto", e);
    }
    finally {
      if (rfc != null) {
        try {
          rfc.close();
        }
        catch (FailureException e) {
          throw new OpenDreamsException(
            "Erro ao fechar o remote file channel na leitura de um dado no projeto",
            e);
        }
      }
    }
  }

  /**
   * Cria um novo arquivo com o contedo passado como parmetro.
   * 
   * @param fileName nome do arquivo
   * @param data o array com os bytes a serem escritos no arquivo
   * @throws OpenDreamsException se o arquivo j existir ou se ocorrer algum
   *         erro durante a criao do arquivo.
   */
  public void createFile(String fileName, byte[] data)
    throws OpenDreamsException {
    byte[] fileKey = this.find(fileName);
    if (fileKey != null) {
      throw new OpenDreamsException("Arquivo " + fileName + " j existe");
    }
    RemoteFileChannel rfc = null;
    try {
      Set<String> views = new HashSet<String>();
      views.add(ProjectItemDescriptionHelper.id());
      long currentDate = Calendar.getInstance().getTimeInMillis();
      ProjectItemDescription prototype =
        new ProjectItemDescriptionImpl(fileName, views,
          new ArrayList<Metadata>(), owner, null, null, null, 0, false, true,
          true, currentDate, currentDate);
      fileKey = dataService.createData(currentDir, prototype);
      UnstructuredData view =
        (UnstructuredData) dataService.getDataView(fileKey,
          UnstructuredDataHelper.id());
      DataKey dataKey = new DataKey(view.fKey);
      rfc =
        new RemoteFileChannelImpl(dataKey.getDataId().getBytes(
          Utils.CHARSET_ENCODING), view.fWritable, view.fHost, view.fPort,
          view.fAccessKey);
      rfc.open(false);
      rfc.write(data);
    }
    catch (Exception e) {
      throw new OpenDreamsException("Erro na criao de um arquivo no projeto",
        e);
    }
    finally {
      if (rfc != null) {
        try {
          rfc.close();
        }
        catch (FailureException e) {
          throw new OpenDreamsException(
            "Erro ao fechar o remote file channel na criao de um arquivo no projeto",
            e);
        }
      }
    }
  }

  /**
   * Remove um arquivo do projeto. O arquivo precisa existir.
   * 
   * @param fileName nome do arquivo
   * @throws OpenDreamsException se o arquivo no existir ou se ocorrer algum
   *         erro durante a remoo do arquivo.
   */
  public void removeFile(String fileName) throws OpenDreamsException {
    byte[] fileKey = this.find(fileName);
    if (fileKey == null) {
      throw new OpenDreamsException("Arquivo " + fileName + " no existe");
    }
    try {
      dataService.deleteData(fileKey);
    }
    catch (Exception e) {
      throw new OpenDreamsException("Erro na remoo do arquivo " + fileName, e);
    }
  }

  /**
   * Verifica se um arquivo ou diretrio existe no projeto.
   * 
   * @param fileName nome do arquivo ou diretrio
   * @return {@code true}, se existe e {@code false}, caso contrrio
   * @throws OpenDreamsException se ocorrer algum erro no acesso ao arquivo
   */
  public boolean hasFile(String fileName) throws OpenDreamsException {
    try {
      return this.find(fileName) != null;
    }
    catch (Exception e) {
      throw new OpenDreamsException("Erro na consulta se o arquivo " + fileName
        + " existe", e);
    }
  }

  /**
   * Nome do projeto.
   * 
   * @return o nome do projeto
   */
  public String getName() {
    return projectDesc.fName;
  }

  /**
   * Retorna a chave para um arquivo ou um diretrio filho do diretrio
   * corrente.
   * 
   * @param name nome do arquivo ou diretrio procurado
   * @return a chave do arquivo procurado ou null caso ele no exista
   * @throws OpenDreamsException se ocorrer algum erro no acesso ao arquivo
   */
  private byte[] find(String name) throws OpenDreamsException {
    byte[] fileKey = null;
    DataDescription[] descriptions;
    try {
      descriptions = dataService.getChildren(currentDir);
      for (DataDescription descr : descriptions) {
        if (descr.fName.equals(name)) {
          fileKey = descr.fKey;
          break;
        }
      }
    }
    catch (Exception e) {
      throw new OpenDreamsException(
        "Erro na procura pelo arquivo ou diretrio " + name, e);
    }
    return fileKey;
  }

  /**
   * Retorna lista de nomes de arquivos e diretrios do diretrio corrente.
   * 
   * @return lista com OpenDreamsException de arquivos e diretrios do diretrio
   *         corrente.
   * @throws OpenDreamsException se ocorrer algum erro no acesso ao diretrio
   *         corrente
   */
  public List<String> list() throws OpenDreamsException {
    ArrayList<String> fileNames = new ArrayList<String>();
    DataDescription[] descriptions;
    try {
      descriptions = dataService.getChildren(currentDir);
      for (DataDescription descr : descriptions) {
        fileNames.add(descr.fName);
      }
    }
    catch (Exception e) {
      throw new OpenDreamsException("Erro na listagem do diretrio: "
        + currentDir, e);
    }
    return fileNames;
  }

  /**
   * Verifica se uma dada entrada  um diretrio ou no.
   * 
   * @param entryName o nome do diretrio ou do arquivo
   * @return {@code true} se  diretrio, {@code false} se for arquivo
   * @throws OpenDreamsException se ocorrer erro durante o acesso ao diretrio
   *         corrente.
   */
  public boolean isDirectory(String entryName) throws OpenDreamsException {
    try {
      ProjectItemDescription descr =
        (ProjectItemDescription) dataService.getDataDescription(this
          .find(entryName));
      return descr.fIsContainer;
    }
    catch (Exception e) {
      throw new OpenDreamsException("Erro ao checar se " + entryName
        + "  diretrio.", e);
    }
  }
}
