package opendreams.proxy;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import openbus.util.OpenBusProxy;
import openbus.util.OpenBusProxyException;
import tecgraf.openbus.DRMAA.JobInfoHelper;
import tecgraf.openbus.data_service.DataAccessDenied;
import tecgraf.openbus.data_service.DataDescription;
import tecgraf.openbus.data_service.IHierarchicalDataService;
import tecgraf.openbus.data_service.IHierarchicalDataServiceHelper;
import tecgraf.openbus.data_service.ServiceFailure;
import tecgraf.openbus.data_service.UnstructuredDataFactory;
import tecgraf.openbus.data_service.UnstructuredDataHelper;
import tecgraf.openbus.data_service.project.ProjectItemDescriptionFactory;
import tecgraf.openbus.opendreams.IOpenDreams;
import tecgraf.openbus.opendreams.IOpenDreamsHelper;
import tecgraf.openbus.opendreams.JobInfoFactory;
import tecgraf.openbus.opendreams.OpenDreamsJobTemplateFactory;
import tecgraf.openbus.opendreams.OpenDreamsJobTemplateHelper;
import tecgraf.openbus.project.ProjectItemDescription;
import tecgraf.openbus.project.ProjectItemDescriptionHelper;

/**
 * O <code>OpenDreamsProxy</code> tem como objetivo encapsular os mecanismos de
 * acesso ao OpenDreams. Faz o login no barramento e recupara as interfaces dos
 * servios usados pelo cliente desse barramento.
 * 
 * @author Tecgraf PUC-Rio
 */
public class OpenDreamsProxy {
  /**
   * Nome do arquivo default com as propriedades para acesso ao OpenDreams.
   */
  public static String DEFAULT_PROPERTIES_FILE = "opendreams.properties";
  /**
   * Nome da propriedade que possui o nome do projeto configurado.
   */
  public static String PROJECT_PROPERTY = "opendreams.project.name";
  /**
   * O arquivo com a chave privada, quando a conexo  por certificado.
   */
  private InputStream privateKey;
  /**
   * O arquivo com o certificado digital, quando a conexo  por certificado.
   */
  private InputStream acsCertificate;
  /**
   * O proxy para o OpenBus
   */
  private OpenBusProxy proxy;

  /**
   * Constri um proxy para o servio OpenDreams do OpenBus, usando as
   * propriedades especificadas.
   * 
   * @param properties as propriedades previamente configuradas
   * @throws OpenDreamsException se houve falha na carga das propriedades
   */
  public OpenDreamsProxy(Properties properties) throws OpenDreamsException {
    try {
      this.proxy = new OpenBusProxy(properties);
    }
    catch (OpenBusProxyException e) {
      throw new OpenDreamsException("Erro na construo do OpenDreamsProxy", e);
    }
  }

  /**
   * Constri um proxy para o servio OpenDreams do OpenBus, usando as
   * propriedades do arquivo default de propriedades
   * 
   * @see #DEFAULT_PROPERTIES_FILE
   * @throws OpenDreamsException se houve falha na carga das propriedades
   */
  public OpenDreamsProxy() throws OpenDreamsException {
    try {
      this.proxy = new OpenBusProxy(DEFAULT_PROPERTIES_FILE);
    }
    catch (OpenBusProxyException e) {
      throw new OpenDreamsException("Erro na construo do OpenDreamsProxy", e);
    }
  }

  /**
   * Atribui o arquivo com a chave privada para ser usada na conexo por
   * certificado. Se esse arquivo no estiver definido, a conexo usa o nome do
   * arquivo definido na propriedade openbus.acs.private.key. Esse mtodo deve
   * ter sido chamado antes de abrir o proxy, ou seja, antes de chamar o mtodo
   * {@link #init()}
   * 
   * @param privateKey o arquivo com chave privada usada para a conexo com o
   *        barramento
   */
  public void setPrivatekey(InputStream privateKey) {
    this.privateKey = privateKey;
  }

  /**
   * Atribui o arquivo com certificado pblico do servio de acesso (ACS) , para
   * ser usado na conexo por certificado. Se esse arquivo no estiver definido,
   * a conexo usa o nome do arquivo definido na propriedade
   * openbus.acs.certificate. Esse mtodo deve ter sido chamado antes de abrir o
   * proxy, ou seja, antes de chamar o mtodo {@link #init()}
   * 
   * @param acsCertificate o arquivo com o certificado pblico do ACS, usado
   *        para a conexo com o barramento
   */
  public void setACSCertificate(InputStream acsCertificate) {
    this.acsCertificate = acsCertificate;
  }

  /**
   * Inicializa o contexto de acesso ao barramento, atravs de certificados.
   * 
   * @throws OpenDreamsException falha no acesso ao openbus
   */
  public void init() throws OpenDreamsException {
    if (!isEnabled()) {
      throw new OpenDreamsException("O proxy para o servio opendreams no"
        + " est habilitado. Verifique as propriedades openbus.enabled e "
        + "opendreams.component.export");
    }
    try {
      if (privateKey != null) {
        proxy.setPrivatekey(privateKey);
      }
      if (acsCertificate != null) {
        proxy.setACSCertificate(acsCertificate);
      }
      proxy.open();
    }
    catch (OpenBusProxyException e) {
      throw new OpenDreamsException("Erro ao abrir o proxy do OpenBus", e);
    }
    registerFactories();
  }

  /**
   * Iniciliza o contexto de acesso ao barramento, atravs de login/senha.
   * 
   * @param user usurio LDAP
   * @param password senha
   * 
   * @throws OpenDreamsException falha no acesso ao openbus
   */
  public void init(String user, String password) throws OpenDreamsException {
    if (!isEnabled()) {
      throw new OpenDreamsException("O proxy para o servio opendreams no"
        + " est habilitado. Verifique as propriedades openbus.enabled e "
        + "opendreams.component.export");
    }
    try {
      proxy.open(user, password);
    }
    catch (OpenBusProxyException e) {
      throw new OpenDreamsException("Erro ao abrir o proxy do OpenBus", e);
    }
    registerFactories();
  }

  /**
   * Faz o registro no ORB das fbricas necessrias para construo
   * (marshalling) dos value types.
   */
  private void registerFactories() {
    proxy.registerFactory(OpenDreamsJobTemplateHelper.id(),
      new OpenDreamsJobTemplateFactory());
    proxy.registerFactory(JobInfoHelper.id(), new JobInfoFactory());
    proxy.registerFactory(ProjectItemDescriptionHelper.id(),
      new ProjectItemDescriptionFactory());
    proxy.registerFactory(UnstructuredDataHelper.id(),
      new UnstructuredDataFactory());
  }

  /**
   * Obtm o objeto registrado no openbus que implementa a interface
   * <code>IOpenDreams</code>
   * 
   * @return o servio <code>IOpenDreams</code>
   * @throws OpenDreamsException se o servio no foi encontrado
   */
  public IOpenDreams getIOpenDreams() throws OpenDreamsException {
    try {
      return IOpenDreamsHelper.narrow(proxy.getComponent("opendreams")
        .getFacet(IOpenDreamsHelper.id()));
    }
    catch (OpenBusProxyException e) {
      throw new OpenDreamsException("Erro ao recuperar o servio opendreams", e);
    }
  }

  /**
   * Obtm o objeto registrado no openbus que implementa a interface
   * <code>IOpenDreams</code>
   * 
   * @return o servio <code>IOpenDreams</code>
   * @throws OpenDreamsException se o servio no foi encontrado
   */
  public IHierarchicalDataService getIDataService() throws OpenDreamsException {
    try {
      return IHierarchicalDataServiceHelper.narrow(proxy.getComponent(
        "ProjectDataService").getFacet(IHierarchicalDataServiceHelper.id()));
    }
    catch (OpenBusProxyException e) {
      throw new OpenDreamsException(
        "Erro ao recuperar o servio ProjectDataService", e);
    }
  }

  /**
   * Obtm um proxy para um projeto do usurio. O nome do projeto deve estar
   * definido na propriedade <code>opendreams.project.name</code>. do arquivo de
   * configurao.
   * 
   * @return um projeto.
   * @throws OpenDreamsException se ocorrer um erro no acesso ao servio de
   *         projetos
   */
  public Project getProject() throws OpenDreamsException {
    return getProject(getProjectName());
  }

  /**
   * Obtm o nome do projeto usado para acesso ao OpenDreams. Essa propriedade 
   * opcional, mas
   * 
   * @return o nome do projeto
   * @throws OpenDreamsException se o nome do projeto no estiver configurado
   */
  public String getProjectName() throws OpenDreamsException {
    String projectName = proxy.getProperties().getProperty(PROJECT_PROPERTY);
    if (projectName == null || projectName.trim().equals("")) {
      throw new OpenDreamsException("Propriedade " + PROJECT_PROPERTY
        + " no definida");
    }
    return projectName;
  }

  /**
   * Obtm um proxy para um projeto do usurio.
   * 
   * @param projectName nome do projeto
   * 
   * @return um projeto.
   * @throws OpenDreamsException se ocorrer um erro no acesso ao servio de
   *         projetos
   */
  public Project getProject(String projectName) throws OpenDreamsException {
    IHierarchicalDataService dataService = this.getIDataService();
    DataDescription[] rootDescList;
    try {
      rootDescList = dataService.getRoots();
      if (rootDescList.length < 1) {
        throw new OpenDreamsException("O usurio no possui projetos");
      }
      for (int i = 0; i < rootDescList.length; i++) {
        DataDescription rootDesc = rootDescList[i];
        if (!(rootDesc instanceof ProjectItemDescription)) {
          throw new OpenDreamsException("Descritor invlido:"
            + rootDesc.toString());
        }
        if (rootDesc.fName.equals(projectName)) {
          return new Project(rootDesc, proxy.getProperties().getUser(),
            dataService);
        }
      }
    }
    catch (ServiceFailure e) {
      throw new OpenDreamsException("Falha no acesso ao servio de projetos", e);
    }
    catch (DataAccessDenied e) {
      throw new OpenDreamsException("Falha no acesso ao servio de projetos", e);
    }
    return null;
  }

  /**
   * Obtm uma lista de proxies para os projetos do usurio.
   * 
   * @return lista de projetos, que pode ser vazia, caso o usurio no tenha
   *         nenhum projeto.
   * @throws OpenDreamsException se ocorrer um erro no acesso ao servio de
   *         projetos
   */
  public List<Project> getAllProjects() throws OpenDreamsException {
    ArrayList<Project> ret = new ArrayList<Project>();

    IHierarchicalDataService dataService = this.getIDataService();
    DataDescription[] rootDescList;
    try {
      rootDescList = dataService.getRoots();

      for (int i = 0; i < rootDescList.length; i++) {
        DataDescription rootDesc = rootDescList[i];
        if (!(rootDesc instanceof ProjectItemDescription)) {
          throw new OpenDreamsException("Descritor invlido:"
            + rootDesc.toString());
        }
        ret.add(new Project(rootDesc, proxy.getProperties().getUser(),
          dataService));
      }
    }
    catch (ServiceFailure e) {
      throw new OpenDreamsException(
        "Falha no acesso ao servio de projetos (getAllProjects())", e);
    }
    catch (DataAccessDenied e) {
      throw new OpenDreamsException(
        "Falha no acesso ao servio de projetos (getAllProjects())", e);
    }
    return ret;
  }

  /**
   * Fecha a conexo com o openbus.
   * 
   * @param className nome cannico da classe que implementa o servio
   * @return {@code true} caso o componente seja registrado ou {@code false},
   *         caso contrrio.
   * @throws OpenDreamsException Caso o proxy no esteja aberto para fazer o
   *         registro das ofertas de servio.
   */
  public boolean addComponent(String className) throws OpenDreamsException {
    try {
      return proxy
        .addComponent("opendreams", IOpenDreamsHelper.id(), className);
    }
    catch (OpenBusProxyException e) {
      throw new OpenDreamsException(
        "Falha na publicao do componente opendreams", e);
    }
  }

  /**
   * Verifica se o proxy est habilitado. Para o proxy estar habilitado 
   * necessrio que o barramento esteja habilitado (openbus.enabled) e o
   * componente esteja habilitado (opendreams.enabled).
   * 
   * @return {@code true} se o proxy est habilitado ou {@code false} caso
   *         contrrio.
   */
  public boolean isEnabled() {
    return proxy.isEnabled()
      && proxy.getProperties().mayExportComponent("opendreams");
  }

  /**
   * Fecha o proxy
   */
  public void close() {
    proxy.close();
  }

  /**
   * Altera o usurio para o qual a credencial est sendo delegada. Essa
   * delegao  feita na thread.
   * 
   * @param user o login do usurio para o qual a credencial est sendo delegada
   */
  public void setThreadDelegate(String user) {
    proxy.setThreadDelegate(user);
  }

  /**
   * Obtm as propriedades usadas para a configurao do proxy.
   * 
   * @return as propriedades usadas para configurao do proxy
   */
  public Properties getProperties() {
    return proxy.getProperties();
  }
}
