import java.io.IOException;
import java.io.InputStream;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.util.Properties;
import java.util.logging.Level;

import org.omg.CORBA.UserException;
import org.omg.CORBA_2_3.ORB;

import scs.core.IComponent;
import tecgraf.ftc.common.logic.RemoteFileChannel;
import tecgraf.ftc.common.logic.RemoteFileChannelImpl;
import tecgraf.ftc.utils.Utils;
import tecgraf.openbus.Openbus;
import tecgraf.openbus.DRMAA.AlreadyActiveSessionException;
import tecgraf.openbus.DRMAA.AuthorizationException;
import tecgraf.openbus.DRMAA.DefaultContactStringException;
import tecgraf.openbus.DRMAA.DeniedByDrmException;
import tecgraf.openbus.DRMAA.DrmCommunicationException;
import tecgraf.openbus.DRMAA.DrmsExitException;
import tecgraf.openbus.DRMAA.DrmsInitException;
import tecgraf.openbus.DRMAA.ExitTimeoutException;
import tecgraf.openbus.DRMAA.InternalException;
import tecgraf.openbus.DRMAA.InvalidArgumentException;
import tecgraf.openbus.DRMAA.InvalidContactStringException;
import tecgraf.openbus.DRMAA.InvalidJobException;
import tecgraf.openbus.DRMAA.InvalidJobTemplateException;
import tecgraf.openbus.DRMAA.JobInfoHelper;
import tecgraf.openbus.DRMAA.NoActiveSessionException;
import tecgraf.openbus.DRMAA.NoDefaultContactStringSelectedException;
import tecgraf.openbus.DRMAA.OutOfMemoryException;
import tecgraf.openbus.DRMAA.Session;
import tecgraf.openbus.DRMAA.TryLaterException;
import tecgraf.openbus.core.v1_05.registry_service.IRegistryService;
import tecgraf.openbus.core.v1_05.registry_service.Property;
import tecgraf.openbus.core.v1_05.registry_service.ServiceOffer;
import tecgraf.openbus.data_service.DataAccessDenied;
import tecgraf.openbus.data_service.DataDescription;
import tecgraf.openbus.data_service.DataKey;
import tecgraf.openbus.data_service.IHierarchicalDataService;
import tecgraf.openbus.data_service.IHierarchicalDataServiceHelper;
import tecgraf.openbus.data_service.ServiceFailure;
import tecgraf.openbus.data_service.UnstructuredData;
import tecgraf.openbus.data_service.UnstructuredDataFactory;
import tecgraf.openbus.data_service.UnstructuredDataHelper;
import tecgraf.openbus.data_service.project.ProjectItemDescriptionFactory;
import tecgraf.openbus.exception.OpenbusAlreadyInitializedException;
import tecgraf.openbus.exception.RSUnavailableException;
import tecgraf.openbus.opendreams.IOpenDreams;
import tecgraf.openbus.opendreams.IOpenDreamsHelper;
import tecgraf.openbus.opendreams.JobInfoFactory;
import tecgraf.openbus.opendreams.OpenDreamsJobInfo;
import tecgraf.openbus.opendreams.OpenDreamsJobTemplate;
import tecgraf.openbus.opendreams.OpenDreamsJobTemplateFactory;
import tecgraf.openbus.opendreams.OpenDreamsJobTemplateHelper;
import tecgraf.openbus.project.ProjectItemDescription;
import tecgraf.openbus.project.ProjectItemDescriptionHelper;
import tecgraf.openbus.util.CryptoUtils;
import tecgraf.openbus.util.Log;

/**
 * Exemplo de um cliente que se conecta ao barramento e usa o servio OpenDreams
 * publicado pelo servio de projetos de um sistema CSBase.
 * 
 * O exemplo executa um algoritmo em uma das mquinas remotas.
 * 
 * @author Tecgraf PUC-Rio
 */
public class OpenDreamsClient {
  /**
   * Nome do arquivo que possui as propriedades para execuo do cliente.
   */
  public static final String CLIENT_PROP_FILE = "/client.properties";

  /**
   * Referncia para o componente que implementa a faceta IOpenDreams
   */
  private IOpenDreams opendreams;

  /**
   * Referncia para o componente que implementa a faceta
   * IHierarchicalDataService
   */
  private IHierarchicalDataService dataService;

  /**
   * Executa as etapas de construo do um cliente que utiliza o openbus para
   * procurar por uma oferta de um servio que oferece a faceta
   * IHierchicalDataService.
   * 
   * @param args os parmetros passados pela linha de comando
   * @throws OpenDreamsClientException se ocorrer algum erro durante a execuo
   *         do exemplo
   */
  private void run(String[] args) throws OpenDreamsClientException {
    Properties props = loadProperties();
    Openbus bus = initOpenbus(args, props);
    /*
     * Registra no ORB as fbricas dos objetos que so passados como parmetros
     * nas chamadas ao servio para que o mecanismo de marshall e unmarshall
     * saiba como criar esses objetos.
     */
    ORB orb = (ORB) bus.getORB();
    orb.register_value_factory(OpenDreamsJobTemplateHelper.id(),
      new OpenDreamsJobTemplateFactory());
    orb.register_value_factory(JobInfoHelper.id(), new JobInfoFactory());
    orb.register_value_factory(ProjectItemDescriptionHelper.id(),
      new ProjectItemDescriptionFactory());
    orb.register_value_factory(UnstructuredDataHelper.id(),
      new UnstructuredDataFactory());
    IRegistryService registryService = connectWithOpenBus(props);

    opendreams = getIOpenDreams(props, registryService);
    String projectName = props.getProperty("dataservice.project.name");
    Session session = null;
    String jobName = null;

    try {
      session = opendreams.getSession(projectName);
      session.init("");
      OpenDreamsJobTemplate jt =
        (OpenDreamsJobTemplate) session.createJobTemplate();
      jt.remoteCommand = "execAlgo";
      jt.args =
        new String[] { "-name", "Teste da SimpleDemoOpenDreams", "-version",
            "1.0.0" };
      jt.jobDescription = "Teste da Demo Simples do OpenDreams pela Julia";
      jt.jobParameters = new String[][] { { "qtde", "3" }, { "duracao", "2" } };
      jt.outputPath = "saida.out";
      //jt.email = new String[] { "mjulia@tecgraf.puc-rio.br" };
      jobName = session.runJob(jt);
      session.deleteJobTemplate(jt);
    }
    catch (AuthorizationException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (InternalException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (InvalidContactStringException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (NoDefaultContactStringSelectedException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (DefaultContactStringException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (DrmCommunicationException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (DrmsInitException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (AlreadyActiveSessionException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (OutOfMemoryException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (InvalidArgumentException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (NoActiveSessionException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (TryLaterException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (DeniedByDrmException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }
    catch (InvalidJobTemplateException e) {
      throw new OpenDreamsClientException("Erro na submisso do algoritmo: "
        + e.message, e);
    }

    OpenDreamsJobInfo jobInfo;
    try {
      jobInfo =
        (OpenDreamsJobInfo) session
          ._wait(jobName, Session.TIMEOUT_WAIT_FOREVER);
    }
    catch (InvalidJobException e) {
      throw new OpenDreamsClientException(
        "Erro na sincronizao de trmino do algoritmo: " + e.message, e);
    }
    catch (NoActiveSessionException e) {
      throw new OpenDreamsClientException(
        "Erro na sincronizao de trmino do algoritmo: " + e.message, e);
    }
    catch (AuthorizationException e) {
      throw new OpenDreamsClientException(
        "Erro na sincronizao de trmino do algoritmo: " + e.message, e);
    }
    catch (InvalidArgumentException e) {
      throw new OpenDreamsClientException(
        "Erro na sincronizao de trmino do algoritmo: " + e.message, e);
    }
    catch (OutOfMemoryException e) {
      throw new OpenDreamsClientException(
        "Erro na sincronizao de trmino do algoritmo: " + e.message, e);
    }
    catch (DrmCommunicationException e) {
      throw new OpenDreamsClientException(
        "Erro na sincronizao de trmino do algoritmo: " + e.message, e);
    }
    catch (InternalException e) {
      throw new OpenDreamsClientException(
        "Erro na sincronizao de trmino do algoritmo: " + e.message, e);
    }
    catch (ExitTimeoutException e) {
      throw new OpenDreamsClientException(
        "Erro na sincronizao de trmino do algoritmo: " + e.message, e);
    }
    System.out.println("** Resultado: hasExited = " + jobInfo.hasExited
      + ": exitStatus = " + jobInfo.exitStatus);
    System.out.println("** Tipo de finalizacao: " + jobInfo.finalizationType);

    dataService = getIDataService(props, registryService);
    DataDescription projectDesc = openProject(projectName);
    if (projectDesc == null) {
      throw new OpenDreamsClientException(
        "O usurio no possui um projeto com o nome " + projectName);
    }
    DataDescription fileDesc = this.find(projectDesc, "saida.out");
    byte[] buffer = readRemoteFile(fileDesc);
    System.out.println("** O log de saida do comando e':");
    System.out.println(new String(buffer));

    try {
      session.exit();
    }
    catch (AuthorizationException e) {
      throw new OpenDreamsClientException("Erro no fechamento da sesso: "
        + e.message, e);
    }
    catch (NoActiveSessionException e) {
      throw new OpenDreamsClientException("Erro no fechamento da sesso: "
        + e.message, e);
    }
    catch (DrmsExitException e) {
      throw new OpenDreamsClientException("Erro no fechamento da sesso: "
        + e.message, e);
    }
    catch (OutOfMemoryException e) {
      throw new OpenDreamsClientException("Erro no fechamento da sesso: "
        + e.message, e);
    }
    catch (DrmCommunicationException e) {
      throw new OpenDreamsClientException("Erro no fechamento da sesso: "
        + e.message, e);
    }
    catch (InternalException e) {
      throw new OpenDreamsClientException("Erro no fechamento da sesso: "
        + e.message, e);
    }

    bus.disconnect();
  }

  /**
   * Obtm o descritor utilizado pelo DataService para a pasta do projeto cujo
   * nome  informado. Retorna null se no for encontrado um projeto com o nome
   * informado.
   * 
   * @param projectName nome do projeto
   * @return o descritor da pasta do projeto ou null se o projeto no for
   *         encontrado
   * @throws OpenDreamsClientException se houver um erro ao abrir a pasta do
   *         projeto
   */
  public DataDescription openProject(String projectName)
    throws OpenDreamsClientException {
    DataDescription[] rootDescList;
    try {
      rootDescList = dataService.getRoots();
      if (rootDescList.length < 1) {
        throw new OpenDreamsClientException("O usurio no possui projetos");
      }
      for (int i = 0; i < rootDescList.length; i++) {
        DataDescription rootDesc = rootDescList[i];
        if (!(rootDesc instanceof ProjectItemDescription)) {
          throw new OpenDreamsClientException("Descritor invlido:"
            + rootDesc.toString());
        }
        if (rootDesc.fName.equals(projectName)) {
          return rootDesc;
        }
      }
    }
    catch (ServiceFailure e) {
      throw new OpenDreamsClientException(
        "Falha no acesso ao servio. Verifique se o usurio  um usurio vlido",
        e);
    }
    catch (DataAccessDenied e) {
      throw new OpenDreamsClientException("Falha no acesso ao servio.", e);
    }
    return null;
  }

  /**
   * Retorna a chave para um arquivo ou um diretrio filho do diretrio
   * corrente.
   * 
   * @param projectDesc descritor do projeto
   * @param name nome do arquivo ou diretrio procurado
   * @return a chave do arquivo procurado ou null caso ele no exista
   * @throws OpenDreamsClientException se ocorrer algum erro no acesso ao
   *         arquivo
   */
  private DataDescription find(DataDescription projectDesc, String name)
    throws OpenDreamsClientException {
    DataDescription found = null;
    DataDescription[] descriptions;
    try {
      descriptions = dataService.getChildren(projectDesc.fKey);
      for (DataDescription descr : descriptions) {
        if (descr.fName.equals(name)) {
          found = descr;
          break;
        }
      }
    }
    catch (Exception e) {
      throw new OpenDreamsClientException(
        "Erro na procura pelo arquivo ou diretrio " + name, e);
    }
    return found;
  }

  /**
   * L um arquivo remoto
   * 
   * @param desc o descritor do arquivo a ser lisdo
   * @return os bytes lidos
   * @throws OpenDreamsClientException se o arquivo j existir ou se ocorrer
   *         algum erro durante a criao do arquivo.
   */
  public byte[] readRemoteFile(DataDescription desc)
    throws OpenDreamsClientException {
    try {
      UnstructuredData view =
        (UnstructuredData) dataService.getDataView(desc.fKey,
          UnstructuredDataHelper.id());
      DataKey dataKey = new DataKey(view.fKey);
      RemoteFileChannel rfc =
        new RemoteFileChannelImpl(dataKey.getDataId().getBytes(
          Utils.CHARSET_ENCODING), view.fWritable, view.fHost, view.fPort,
          view.fAccessKey);
      rfc.open(true);
      int fileSize = (int) rfc.getSize();
      byte[] buffer = new byte[fileSize];
      if (fileSize != 0) {
        rfc.read(buffer);
      }
      rfc.close();
      return buffer;
    }
    catch (Exception e) {
      throw new OpenDreamsClientException(
        "Erro na leitura de um dado da pasta", e);
    }
  }

  /**
   * Utiliza o servio de registro do barramento para procurar pelo componente
   * DataService de acordo com as propriedades <b>dataservice.component.name</b>
   * e <b>dataservice.component.version</b>.
   * 
   * @param props as propriedades que possuem as informaes com o nome e a
   *        verso do componente
   * @param registryService o servio de registro
   * @return o componente DataService que implementa a faceta
   *         <b>IHierarchicalDataService</b>.
   * @throws OpenDreamsClientException se houver algum erro na busca pelo
   *         servio DataService
   */
  private IHierarchicalDataService getIDataService(Properties props,
    IRegistryService registryService) throws OpenDreamsClientException {
    String componentNameProperty =
      props.getProperty("dataservice.component.name").trim();
    String componentVersionProperty =
      props.getProperty("dataservice.component.version").trim();
    String componentId = componentNameProperty + ":" + componentVersionProperty;
    Property property =
      new Property("component_id", new String[] { componentId });
    ServiceOffer[] servicesOffers =
      registryService.findByCriteria(
        new String[] { IHierarchicalDataServiceHelper.id() },
        new Property[] { property });
    if (servicesOffers.length == 0) {
      throw new OpenDreamsClientException(
        "No foi encontrado um servio IHierarchicalDataService com identificador: "
          + componentId);
    }
    if (servicesOffers.length > 1) {
      throw new OpenDreamsClientException(
        "Foi encontrado mais de um servio IHierarchicalDataService com identificador: "
          + componentId);
    }
    ServiceOffer serviceOffer = servicesOffers[0];
    IComponent component = serviceOffer.member;
    IHierarchicalDataService dataService =
      IHierarchicalDataServiceHelper.narrow(component
        .getFacet(IHierarchicalDataServiceHelper.id()));

    return dataService;
  }

  /**
   * Utiliza o servio de registro do barramento para procurar pelo componente
   * OpenDreams de acordo com as propriedades <b>opendreams.component.name</b> e
   * <b>opendreams.component.version</b>.
   * 
   * @param props as propriedades que possuem as informaes com o nome e a
   *        verso do componente
   * @param registryService o servio de registro
   * @return o componente OpenDreams que implementa a faceta <b>IOpenDreams</b>.
   * @throws OpenDreamsClientException se houver algum erro na busca pelo
   *         servio OpenDreams
   */
  private IOpenDreams getIOpenDreams(Properties props,
    IRegistryService registryService) throws OpenDreamsClientException {
    String componentNameProperty =
      props.getProperty("opendreams.component.name").trim();
    String componentVersionProperty =
      props.getProperty("opendreams.component.version").trim();
    String componentId = componentNameProperty + ":" + componentVersionProperty;
    Property property =
      new Property("component_id", new String[] { componentId });
    ServiceOffer[] servicesOffers =
      registryService.findByCriteria(new String[] { IOpenDreamsHelper.id() },
        new Property[] { property });
    if (servicesOffers.length == 0) {
      throw new OpenDreamsClientException(
        "No foi encontrado um servio OpenDreams com identificador: "
          + componentId);
    }
    if (servicesOffers.length > 1) {
      throw new OpenDreamsClientException(
        "Foi encontrado mais de um servio OpenDreams com identificador: "
          + componentId);
    }
    ServiceOffer serviceOffer = servicesOffers[0];
    IComponent component = serviceOffer.member;
    IOpenDreams openDreamsService =
      IOpenDreamsHelper.narrow(component.getFacet(IOpenDreamsHelper.id()));

    return openDreamsService;
  }

  /**
   * Carrega as propriedades do exemplo.
   * 
   * @return as propriedades carregadas a partir do arquivo
   *         {@link #CLIENT_PROP_FILE}
   * @throws OpenDreamsClientException se houver um erro durante a carga das
   *         propriedades
   */
  private Properties loadProperties() throws OpenDreamsClientException {
    Properties props = new Properties();
    InputStream in =
      OpenDreamsClient.class.getResourceAsStream(CLIENT_PROP_FILE);
    if (in != null) {
      try {
        props.load(in);
      }
      catch (IOException e) {
        throw new OpenDreamsClientException(
          "Erro ao carregar o arquivo de propriedades " + CLIENT_PROP_FILE, e);
      }
      finally {
        try {
          in.close();
        }
        catch (IOException e) {
          throw new OpenDreamsClientException(
            "Erro ao fechar o arquivo de propriedades " + CLIENT_PROP_FILE, e);
        }
      }
      return props;
    }
    else {
      throw new OpenDreamsClientException(
        "Erro ao abrir o arquivo de propriedades " + CLIENT_PROP_FILE);
    }
  }

  /**
   * Faz a inicializao do Openbus. Configura as propriedades do ORB com os
   * argumentos passados na linha de comando e com as propriedades para o uso do
   * Jacorb.
   * 
   * @param args argumentos passados pela linha de comando
   * @param props propriedades do arquivo
   * @return o openbus iniciado
   * @throws OpenDreamsClientException se ocorrer um erro durante a
   *         inicializao do Openbus.
   */
  private Openbus initOpenbus(String[] args, Properties props)
    throws OpenDreamsClientException {
    String host = props.getProperty("host.name");
    String portString = props.getProperty("host.port");
    int port = Integer.valueOf(portString);
    String ip = props.getProperty("OAIAddr");

    Log.setLogsLevel(Level.WARNING);
    Properties orbProps = new Properties();
    orbProps.setProperty("org.omg.CORBA.ORBClass", "org.jacorb.orb.ORB");
    orbProps.setProperty("org.omg.CORBA.ORBSingletonClass",
      "org.jacorb.orb.ORBSingleton");
    if (ip != null) {
      orbProps.setProperty("OAIAddr", ip);
    }
    Openbus bus = Openbus.getInstance();
    try {
      bus.init(args, orbProps, host, port);
      return bus;

    }
    catch (UserException e) {
      throw new OpenDreamsClientException(
        "Erro durante a inicializao do Openbus.", e);
    }
    catch (OpenbusAlreadyInitializedException e) {
      throw new OpenDreamsClientException(
        "Erro durante a inicializao do Openbus: Barramento j foi inicializado.",
        e);
    }
  }

  /**
   * Obtm o servio de registro resultado da conexo com o Openbus.
   * 
   * @param props as propriedades que possuem as informaes com o nome da
   *        entidade do certificado, o nome do arquivo com a chave privada
   *        correspondente e o nome do arquivo com o certificado pblico do
   *        servio de acesso.
   * @return a referncia para o servio de registro
   * 
   * @throws OpenDreamsClientException se houver algum erro durante a conexo
   *         com o servio de acesso do openbus.
   */
  private IRegistryService connectWithOpenBus(Properties props)
    throws OpenDreamsClientException {
    String entityName = props.getProperty("entity.name");
    String privateKeyFile = props.getProperty("private.key");
    String acsCertificateFile = props.getProperty("acs.certificate");

    /* Arquivo com a chave privada. */
    RSAPrivateKey privateKey;
    try {
      privateKey = CryptoUtils.readPrivateKey(privateKeyFile);
    }
    catch (Exception e) {
      throw new OpenDreamsClientException(
        "Erro ao ler o arquivo com a chave privada", e);
    }
    /* Arquivo com o certificado pblico do Servio de Acesso */
    X509Certificate acsCertificate;
    try {
      acsCertificate = CryptoUtils.readCertificate(acsCertificateFile);
    }
    catch (Exception e) {
      throw new OpenDreamsClientException(
        "Erro ao ler o arquivo com o certificado pblico do ACS", e);
    }
    Openbus bus = Openbus.getInstance();
    try {
      IRegistryService registryService =
        bus.connect(entityName, privateKey, acsCertificate);
      if (registryService == null) {
        throw new RSUnavailableException();
      }
      return registryService;
    }
    catch (Exception e) {
      throw new OpenDreamsClientException(
        "Erro ao fazer a conexo com o Openbus", e);
    }
  }

  /**
   * Programa principal que executa o cliente.
   * 
   * @param args os argumentos passados pela linha de comando
   */
  public static void main(String[] args) {
    OpenDreamsClient client = new OpenDreamsClient();
    try {
      client.run(args);
    }
    catch (Throwable e) {
      e.printStackTrace();
    }
  }

  /**
   * Exceo usada no exemplo para reportar os erros durante a execuo.
   */
  class OpenDreamsClientException extends Exception {
    /**
     * Construtor.
     * 
     * @param msg a mensagem do erro
     */
    public OpenDreamsClientException(String msg) {
      super(msg);
    }

    /**
     * Construtor.
     * 
     * @param msg a mensagem do erro
     * @param t a causa do erro
     */
    public OpenDreamsClientException(String msg, Throwable t) {
      super(msg, t);
    }
  }
}
